/*                                                                -*-c++-*-
    Copyright (C) 1993 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  Blob.hh
//
//  Declaration of the Blob object, a simple example of a Thing.
//
//  6 - 7 - 93
//  Sidd Puri
//
//-----------------------------------------------------------------------------

#ifndef Blob_h
#define Blob_h

#include <stream.h>
#include <stdio.h>
#include "Tracker.hh"
#include "FTypes.hh"
#include "Image.hh"
#include "BasicFeature.hh"

//-----------------------------------------------------------------------------
//  Default Blob parameters
//-----------------------------------------------------------------------------

int   const Blob_height = 40;
int   const Blob_width  = 40;
int   const Blob_statevars = 2;      // x,y

//-----------------------------------------------------------------------------
//  Blob class declaration
//-----------------------------------------------------------------------------

#define CYC_TRACK    1

typedef struct {
  int    id;
  int    row,     col;
  int    height,  width;
  int    r_height, r_width;
  int    d_row,   d_col;
  int    pri;
  int    active;
  int    lpix_thresh;
  int    upix_thresh;
  int    num_thresh;
/*
  float  m2_max_off, m2_min_rat, m2_max_rat;
*/
} window_spec_t;

typedef struct {
  window_spec_t   win;
  unsigned int    id;
  unsigned int    status;
  unsigned int    time;			/* age of the data */
  unsigned int    clock;		/* time of the image (absolute) */
  unsigned int    f_count;		/* frame count */
  float    row,     col;
  float    m0;				/* zeroth moment */
  float    xx,yy,xy;			/* 2nd moments */
  float    l0, l1, d;			/* eigen structure of 2nd mom */
} image_report_t;


class Blob: public BasicFeature/*, public PointFeature*/ 
{

  // Height is the height of the Blob; width is the total size
  // of the area perpendicular to the Blob to search
  // samp_height and samp_width determine subsampling of these
  // two dimensions.

  window_spec_t win_spec;
  int samp_height, samp_width;
  int mmapped;
  float win_percent;

  // Local pattern and video pointers.

  static const char *_typeid;
  Image im_buffer;

public:

  // Two ways of creating a Blob. 


  Blob (Video *v,
	int height_in = Blob_height, int width_in = Blob_width,
	int lthresh_in = 100,int uthresh_in = 255,
	int lsamp = 1, int usamp = 1,
	float min_percent = 0.1);

  Blob (Blob &l);

  ~Blob() {};

  int in_view(int x, int y)
	{return v->in_view(x,y,int(width()),int(height()));}

  // Blob initialization 

  int state_init(float *s) {return state_init(s[0],s[1]);};
  int state_init (float x_in, float y_in);
  int interactive_init (CWindow &w);

  BasicFeature *dup() { return new Blob (*this); }

  // Display routines

  int show(Color color_in = default_color);
  int clear();

  int in_view() {return in_view(round(x()),round(y()));}

  // Update routines

  int update (Video &v);
  int update(){return update(*v);};

  // Changing sampling.

  int &height_sampling() {return samp_height;};
  int &width_sampling() {return samp_width;};

  // information accessors

  double width()  const {return win_spec.width*samp_width;};
  double height() const {return win_spec.height*samp_height;};

  // state variable accessors

  float x()	      const   { return state[0];}	     
  float y()	      const   { return state[1];}

  Video *source()	{ return v; } //video source

  // state variable changers

  float set_x(float newx)	      {return state[0] = newx;}	     
  float set_y(float newy)	      {return state[1] = newy;}

  // Output

  void show (CWindow &w)
    { w.show(im_buffer.magnify(width_sampling(),height_sampling())); }

  // Typing

  static const char *info() {return _typeid;};

};

//-----------------------------------------------------------------------------
#endif
