/*
    Copyright (C) 1991 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  Blob.cc
//
//  Definition of Blob member functions.
//
//  6 - 7 - 93
//  Sidd Puri
//
//-----------------------------------------------------------------------------

#include <stdlib.h> // for exit()
#include "CWindow.hh"
#include "Video.hh"
#include "Blob.hh"

const char *Blob::_typeid = "Blob";
static int ids = 0;

// Creation

Blob::Blob (Video *v,
	    int height_in, int width_in, 
	    int lthresh_in, int uthresh_in,
	    int hsamp, int wsamp,
	    float min_percent):
     BasicFeature (Blob_statevars,v,_typeid),/* PointFeature(this),*/
     im_buffer(10,10)
{ 
  int h = height_in/hsamp; int w = width_in/wsamp; 

  mmapped = (v->bytesperpixel());

  if (mmapped != 1) {
    im_buffer.resize(w,h);
    win_spec.d_row = win_spec.d_col = 1;
    samp_width = wsamp;
    samp_height = hsamp;
  }
  else {
    samp_height = win_spec.d_row = hsamp;
    samp_width =  win_spec.d_col = wsamp;
  }
 
  win_percent = min_percent;

  win_spec.id = ids++;
  win_spec.height = h;
  win_spec.width = w;
  win_spec.lpix_thresh = lthresh_in;
  win_spec.upix_thresh = uthresh_in;
  win_spec.num_thresh = round(h*w * win_percent);

}


Blob::Blob (Blob &l): BasicFeature (l),/* PointFeature(l)//(void *)this),*/
    im_buffer(l.im_buffer)
{ 
  win_spec = l.win_spec;
  samp_height = l.samp_height; 
  samp_width = l.samp_width;
  mmapped = l.mmapped;
  win_percent = l.win_percent;

}

//-----------------------------------------------------------------------------
//  Thing member functions
//-----------------------------------------------------------------------------

int _show = 0;

int Blob::state_init (float x_in, float y_in) {
  set_x(x_in);
  set_y(y_in);
  win_spec.row = round(y_in);
  win_spec.col = round(x_in);
  return 1;
}

int Blob::interactive_init (CWindow &w) {
  position* pos = w.get_region (*source(), (int)width(),
			      (int)height(),"Position Blob");
  state_init( (pos[0].x+pos[1].x)/2, (pos[0].y+pos[1].y)/2 );
  delete pos;
  return 1;
}

int Blob::show (int color_in)
{
  int colr = color_in == no_color? color: color_in;
 
  v->line(x(),(y()-(float)height()/2.),int(width()),0.,colr);
  v->line(x(),(y()+(float)height()/2.),int(width()),0.,colr);
  v->line((x()-(float)width()/2.),y(),int(height()),M_PI/2,colr);
  v->line((x()+(float)width()/2.),y(),int(height()),M_PI/2,colr);

  return 1;
}

int Blob::clear () {
  show(erase_color);
  return 1;
}

static void calc_centroid(char *CYC_MEM_PTR,
			  int i0, int i1, int di,
			  int j0, int j1, int dj,
			  int lthresh, int uthresh,
			  int *r_m0, int *r_x,  int *r_y,
			  int *r_xx, int *r_yy, int *r_xy)
{
  const int d_row    = 640 * di;
  int i,j;
  volatile char *pixel,*row;

  int m0;
  int x,  y;
  int xx, yy, xy;

  m0=x=y=xx=yy=xy= 0;

  row = CYC_MEM_PTR + 640*i0 + j0;
  
  for(i=i0; i<i1; i+=di) {
    pixel = row;
    for(j=j0; j<j1; j+=dj) {
      if ((*pixel > lthresh) && (*pixel < uthresh)) {
	m0++;
	x += j;
	y += i;
	xx += j*j;
	yy += i*i;
	xy += i*j;
      }
      pixel += dj;			/* go to the next pixel */
    }
    row += d_row;			/* go to the next row */
  }

  *r_m0 += m0;
  *r_x  += x;
  *r_y  += y;
  *r_xx += xx;
  *r_yy += yy;
  *r_xy += xy;
}

static void calc_centroid(const Image &im,
			  int lthresh, int uthresh,
			  int *r_m0, int *r_x,  int *r_y,
			  int *r_xx, int *r_yy, int *r_xy)
{

  int *pixel = im.data();
  int w = im.width(), h = im.height();
  int *endim = pixel + w*h;
  int i,j,k;

  int m0;
  int x,  y;
  int xx, yy, xy;

  m0=x=y=xx=yy=xy= 0;

  for (k=0;pixel < endim;pixel++, k++)
      if ((*pixel > lthresh) && (*pixel < uthresh)) {
	m0++;
	i = k%w;
	j = k/w;
	x += i;
	y += j;
	xx += j*j;
	yy += i*i;
	xy += i*j;
      }

  *r_m0 += m0;
  *r_x  += x;
  *r_y  += y;
  *r_xx += xx;
  *r_yy += yy;
  *r_xy += xy;
}

static void calc_res(window_spec_t *win,
		     image_report_t *res,
		     int n,
		     int x,   int y,
		     int xx,  int yy,  int xy)
{
  float m02;
  float fx, fy;
  float fxx, fyy, fxy;
  float sum, dif, rad;

  memset(res, 0, sizeof(*res));
  res->id = win->id;
  res->status = 0;
  res->m0 = n;
  res->win = *win;
  if (n >= win->num_thresh) {
    fx = x;
    fy = y;
    res->row = fy / n;
    res->col = fx / n;

    m02 = n * n;
    res->xx = fxx = (n * ((float)xx) - (fx * fx)) / m02;
    res->yy = fyy = (n * ((float)yy) - (fy * fy)) / m02;
    res->xy = fxy = (n * ((float)xy) - (fy * fx)) / m02;

    sum = 0.5f*(fxx+fyy);
    dif = fxx-fyy;
    rad = 0.5f*(sqrt(dif*dif + 4.0f * fxy*fxy));
    res->l0 = sum+rad;
    res->l1 = sum-rad;
    res->d  = atan2(-dif + 2.0f * rad, 2.0f * fxy);
    if (res->d > M_PI_2)
      res->d -= M_PI;
    else if (res->d < -M_PI_2)
      res->d += M_PI;
    res->status |= CYC_TRACK;
    return;
/*
    if (fabsf(res->xy) <= win->m2_max_off &&
	res->yy/res->xx >= win->m2_min_rat     &&
	res->yy/res->xx <= win->m2_max_rat) {
      res->status |= CYC_TRACK;
      return;
    } else {
      res->status |= CYC_M2_ERR;
    }
*/
  }

  res->row = ((float) win->row) + ((float) win->height)/2.0f;
  res->col = ((float) win->col) + ((float) win->width)/2.0f;
  res->xx = res->yy = res->xy = 0.0f;
}


static void find_centroid(char *im_ptr,
			  window_spec_t   *win,
			  image_report_t  *res)
{
  int m0;
  int x,y;
  int xx,yy,xy;

  m0=x=y=xx=yy=xy= 0;

  calc_centroid(im_ptr,win->row, win->row+win->height, win->d_row,
		win->col, win->col+win->width,  win->d_col,
		win->lpix_thresh,win->upix_thresh,
		&m0, &x, &y, &xx, &yy, &xy);

  calc_res(win, res,
	   m0, x, y, xx, yy, xy);
}
static void find_centroid(const Image &im,
			  window_spec_t   *win,
			  image_report_t  *res)
{
  int m0;
  int x,y;
  int xx,yy,xy;

  m0=x=y=xx=yy=xy= 0;

  calc_centroid(im,
		win->lpix_thresh,win->upix_thresh,
		&m0, &x, &y, &xx, &yy, &xy);

  calc_res(win, res,
	   m0, x, y, xx, yy, xy);

}  


int Blob::update (Video &v) {
  
  image_report_t res;
  char *tmp;

  if (mmapped) {
    if ((tmp = (char *)(v.direct_memptr())) != NULL) {
      find_centroid(tmp,&win_spec,&res);

      if (res.status & CYC_TRACK) {

	// Update Blob parameters
	
	set_x(res.col);
	set_y(res.row);
	return 1;
      }
      else
	{ puts("Blob: not in view."); 
	  return 0;
	}
    }
    else
      return 0;
  }
  else {
    float fx = round(x());
    float fy = round(y());
    source()->grab_aligned(im_buffer,
			   fx,fy,
			   width_sampling(),height_sampling());
		    
    find_centroid(im_buffer,&win_spec,&res);

    if (res.status & CYC_TRACK) {

      // Update Blob parameters

      set_x((res.col-im_buffer.width()/2)*width_sampling() + x());
      set_y((res.row-im_buffer.height()/2)*height_sampling() + y());
      return 1;
    }
    else 
      return 0;
  }
}
















