/*                                                                -*-c++-*-
    Copyright (C) 1993 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  BasicBlob.hh
//
//  Declaration of the Blob object which uses a segmentation algorithm
//  followed by a moments computation.  Credits to Al Rizzi for 
//  loaning me a C version of his code to start with.
//
//  7 - 18 -96

//  Greg Hager
//
//-----------------------------------------------------------------------------

#ifndef BasicBlob_h
#define BasicBlob_h

#include <stream.h>
#include "stdio.h"
#include "Tracker.hh"
#include "FTypes.hh"
#include "Image.hh"
#include "BasicFeature.hh"

//-----------------------------------------------------------------------------
//  BasicBlob class declaration
//-----------------------------------------------------------------------------

const int default_Blob_height = 40;
const int default_Blob_width = 40;
const float default_Blob_percent = 0.4;
const int BasicBlob_statevars = 6;

class BasicBlob: public BasicFeature, public PointFeature
{

protected:

  // Height is the height of the Blob; width is the total size
  // of the area perpendicular to the Blob to search
  // samp_height and samp_width determine subsampling of these
  // two dimensions.  win_percent is the minimal number of pixels
  // in the window which have to satisfy the membership criteria.

  int a_height, a_width;                    // Actual area covered
  int samp_height, samp_width;              // Subsampling
  int npixels;                              // Actual number of pixels
  float min_percent;                        // Minimum needed to accept

  int dynamically_resize_blob;
  float min_membership_percent;
  float max_membership_percent;
  float growth_factor;
  float shrink_factor;

  // Local pattern and video pointers.

  static const char *_typeid;
  Image im_buffer;

  // This is a pixel test function which is really the only variation
  // in a generic blob tracker.  For best performance, make a local
  // copy of calc_centroid and inline your test.  Gives a few % better
  // performance depending on what you're doing.

  virtual int member_test(int) = 0;
  virtual int calc_centroid(const Image&);

  void grow_blob(float off_by_x_percent);
  void shrink_blob(float off_by_x_percent);

public:

  // Two ways of creating a Blob. 

  BasicBlob (Video *v,
	int height_in = default_Blob_height, int width_in = default_Blob_width,
	int hsamp = 1, int wsamp = 1,
	float min_percent = default_Blob_percent);

  BasicBlob (const BasicBlob &l);

  ~BasicBlob() {};

  int in_view(int x, int y)
	{return v->in_view(x,y,width(),height());}

  // Blob changes

  void resize(int rows, int cols);

  // Blob initialization 

  int state_init(float *s) {return state_init(s[0],s[1]);};
  int state_init (float x_in, float y_in);
  int interactive_init (CWindow &w);
  void autoresize(float max_percent, 
		  float min_percent,
		  float shrinkby, 
		  float growby);
		  

  // Display routines

  int show(Color color_in = default_color);
  int clear();

  int in_view() {return in_view(round(x()),round(y()));}

  // Update routines

  int update (Video &v);
  int update(){return update(*v);};

  // Changing sampling.

  int &height_sampling() {return samp_height;};
  int &width_sampling() {return samp_width;};

  // information accessors

  int width()  const {return a_width;};
  int height() const {return a_height;};

  // state variable accessors

  float x()	      const   { return state[0];}	     
  float y()	      const   { return state[1];}

  float major_axis()  const   { return state[2];}	     
  float minor_axis()  const   { return state[3];}

  float orientation() const   { return state[4];}	     
  float strength()    const   { return state[5];}

  Video *source()	{ return v; } //video source

  // state variable changers

  float set_x(float newx)	      {return state[0] = newx;}	     
  float set_y(float newy)	      {return state[1] = newy;}

  float set_major_axis(float newx)    {return state[2] = newx;}	     
  float set_minor_axis(float newy)    {return state[3] = newy;}

  float set_orientation(float newx)   {return state[4] = newx;}	     
  float set_strength(float newy)      {return state[5] = newy;}

  // Output

  void show (CWindow &w)
  {w.show(im_buffer.magnify(width_sampling(),height_sampling())); };

  // Typing

  static const char *info() {return _typeid;};

};

//-----------------------------------------------------------------------------
#endif
