
/*
    Copyright (C) 1991 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/
//-----------------------------------------------------------------------------
//
//  BasicBlob.cc
//
//  Definition of BasicBlob member functions.
//
//  7 - 18 -96
//  Greg Hager
//
//-----------------------------------------------------------------------------

#include <stdlib.h> // for exit()
#include "CWindow.hh"
#include "Video.hh"
#include "BasicBlob.hh"
#include <math.h>
#define __max(x,y) (x>y)?x:y

const char *BasicBlob::_typeid = "BasicBlob";
static int ids = 0;

// Creation

void
BasicBlob::resize(int height_in,int width_in)
{
  a_height = height_in;
  a_width = width_in;

  npixels = (a_width/ samp_width ) * (a_height/samp_height);
  im_buffer.resize(a_height/samp_height,a_width/samp_width);
}  




BasicBlob::BasicBlob (Video *v,
	    int height_in, int width_in, 
	    int hsamp, int wsamp,
	    float min_percent_in):
     BasicFeature (BasicBlob_statevars,v,_typeid), PointFeature(this),
     im_buffer(0,0)
{ 
  samp_width = wsamp;
  samp_height = hsamp;
  resize(height_in,width_in);
  min_percent = min_percent_in;
  dynamically_resize_blob = 0;
  
#ifdef MMAPPING_USED
  mmapped = (v->bytesperpixel());

  if (mmapped != 1) {
    im_buffer.resize(w,h);
    win_spec.d_row = win_spec.d_col = 1;
    samp_width = wsamp;
    samp_height = hsamp;
  }
  else {
    samp_height = win_spec.d_row = hsamp;
    samp_width =  win_spec.d_col = wsamp;
  }
#endif
}

BasicBlob::BasicBlob (const BasicBlob &l): BasicFeature (l), PointFeature(this),
    im_buffer(l.im_buffer)
{ 
  resize(l.height(),l.width());
  //  mmapped = l.mmapped;
  min_percent = l.min_percent;
  // guards against the constant allocation of memory.
  // by making the image buffer large enough to accomodate any blob size up
  // to 800 x 800 before it has to futz with memory
  dynamically_resize_blob = l.dynamically_resize_blob;
  if (dynamically_resize_blob)
    {
      im_buffer.resize(800,800);
      im_buffer.resize(a_height/samp_height,a_width/samp_width);
    }  
  min_membership_percent = l.min_membership_percent;
  max_membership_percent = l.max_membership_percent;
  growth_factor = l.growth_factor;
  shrink_factor = l.shrink_factor;
}

//-----------------------------------------------------------------------------
//  BasicFeature member functions
//-----------------------------------------------------------------------------

int BasicBlob::state_init (float x_in, float y_in) {
  set_x(x_in);
  set_y(y_in);
  return 1;
}

int BasicBlob::interactive_init (CWindow &w) {
  position* pos = w.get_region (*source(), (int)width(),
			      (int)height(),"Position BasicBlob");
  state_init((pos[0].x+pos[1].x)/2, (pos[0].y+pos[1].y)/2 );
  delete pos;
  return 1;
}

void BasicBlob::autoresize(float max_percent, 
			   float min_percent,
			   float shrinkby,
			   float growby)
{
  dynamically_resize_blob = 1;
  min_membership_percent = min_percent;
  max_membership_percent = max_percent;
  growth_factor = growby;
  shrink_factor = shrinkby;
  im_buffer.resize(800,800);
  im_buffer.resize(a_height/samp_height,a_width/samp_width);
};

void BasicBlob::grow_blob(float off_by_x_percent)
{

  /*  
  float maj = major_axis();
  float min = minor_axis(); 
  float total = maj+min;
  int widthcomponent = __max (abs(maj * (cos(orientation()))), 
			    abs(min * (sin(orientation()))));
  int heightcomponent= __max (abs(maj * (sin(orientation()))), 
			    abs(min * (cos(orientation()))));  
  int newwidth = widthcomponent;  
  int newheight = heightcomponent;  
  cout << "growing to (" << newheight << "," << newwidth << ") "  
  << " o= " << orientation()
  << " mj= " << maj 
  << " mn= " << min 
  << " wc= " << widthcomponent 
  << " hc= " << heightcomponent 
  << endl;         */ 


  //  cout << "resizing offby " << off_by_x_percent << endl;

  float newheight = height() * (1.0f + off_by_x_percent);
  float newwidth = width() * (1.0f + off_by_x_percent);

  // cout << "newheight,newwidth want to be " 
  //      << newheight << " " << newwidth << endl;
  // cout << "shrinking" << endl;
  
  float integrated_height = (1.0f - growth_factor) * height();
  integrated_height += growth_factor*newheight;
  float integrated_width = (1.0f - growth_factor)*width();
  integrated_width += growth_factor*newwidth;
  
  resize(int(ceil(integrated_height)), int(ceil(integrated_width)));

  
	 
  // cout << "actual resize to " 
  //     << integrated_height
  //     << " " 
  //     << integrated_width
  //     << endl; 


};

void BasicBlob::shrink_blob(float off_by_x_percent)
{  
  /*
  float maj = major_axis();
  float min = minor_axis(); 
  float total = maj+min;  
  int widthcomponent = __max (abs(maj * (cos(orientation()))), 
			    abs(min * (sin(orientation()))));
  int heightcomponent= __max (abs(maj * (sin(orientation()))), 
			    abs(min * (cos(orientation()))));

  
 
  int newwidth = widthcomponent;
  int newheight = heightcomponent;

  cout << "shrinking to (" << newheight << "," << newwidth << ") " 
       << " o= " << orientation()
       << " mj= " << maj 
       << " mn= " << min 
       << " wc= " << widthcomponent
       << " hc= " << heightcomponent
       << endl; 
       
       
       */

  //  cout << "resizing offby " << off_by_x_percent << endl;

  float newheight = height() * (1.0f + off_by_x_percent);
  float newwidth = width() * (1.0f + off_by_x_percent);

  // cout << "newheight,newwidth want to be " 
  //      << newheight << " " << newwidth << endl;
  // cout << "shrinking" << endl;
 
  float integrated_height = (1.0f - growth_factor) * height();
  integrated_height += growth_factor*newheight;
  float integrated_width = (1.0f - growth_factor)*width();
  integrated_width += growth_factor*newwidth;
  
  resize(int(ceil(integrated_height)), int(ceil(integrated_width)));
	 
  //cout << "actual resize to " 
  //     << integrated_height
  //     << " " 
  //     << integrated_width
  //     << endl; 

};


int BasicBlob::show (int color_in)
{
  int colr = color_in == no_color? color: color_in;
 
  v->line(x(),(y()-(float)height()/2.),width(),0.,colr);
  v->line(x(),(y()+(float)height()/2.),width(),0.,colr);
  v->line((x()-(float)width()/2.),y(),height(),M_PI/2,colr);
  v->line((x()+(float)width()/2.),y(),height(),M_PI/2,colr);
  return 1;
}

int BasicBlob::clear () {
  show(erase_color);
  return 1;
}

// This calculates the moment representation of the pixels
int BasicBlob::calc_centroid(const Image &im)
{

  int *pixel = im.data();
  int *endim = pixel + npixels;
  int i,j,k;

  int m0 = 0;
  int mx=0,  my=0;
  int xx=0, yy=0, xy=0;

  // Here is the segmentation

  for (k=0; pixel < endim; k++)
    if (member_test(*pixel++)) {
	m0++;
	i = k%im.width();
	j = k/im.width();
	mx += i;
	my += j;
	xx += i*i; //* Bug fix by Angie 1/30/98
	yy += j*j;
	xy += i*j;
#if 0
	xx += j*j; //* Bug fix by Angie 1/30/98
	yy += i*i;
	xy += i*j;
#endif 
      }

  // Did we even find anything?
  // printf("membership: %d pixels out of %d total\n", m0, npixels);
  if (m0 < (npixels*min_percent)) return 0;

  // Calculate the state variables

  float 
        fx = mx,
        fy = my;

  // Position offsets
  //  cout << fx/m0 << " " << fy/m0 << endl;

  set_x((fx/m0 - (im.width()-1)/2.0)*samp_width + x());
  set_y((fy/m0 - (im.height()-1)/2.0)*samp_height + y());

  // Set strength
  set_strength((float)m0/(float)npixels);

  // Compute second moments
  float m02 = m0 * m0;
  float fxx = (m0 * ((float)xx) - (fx * fx)) / m02;
  float fyy = (m0 * ((float)yy) - (fy * fy)) / m02;
  float fxy = (m0 * ((float)xy) - (fy * fx)) / m02;
    

  // Compute ellipse parameters
  float sum = 0.5f*(fxx+fyy);
  float dif = fxx-fyy;
  float rad = 0.5f*(sqrt(dif*dif + 4.0f * fxy*fxy));
  set_major_axis(sqrt(sum+rad));
  set_minor_axis(sqrt(sum-rad));
   
  float d  = atan2(-dif + 2.0f * rad, 2.0f * fxy);
  if (d > M_PI_2)
    d -= M_PI;
  else if (d < -M_PI_2)
    d += M_PI;
  
  set_orientation(d);

  if (dynamically_resize_blob)
    {

      // cout << strength() << " is the curent strength " << endl;
      int pixels;
      if (strength() < min_membership_percent)
	{
	  shrink_blob(strength() - min_membership_percent);
	}
      else 
	if (strength() > max_membership_percent)
	  {
	    grow_blob(strength() - max_membership_percent);
	  };
    };
     
  return 1;

}

int BasicBlob::update (Video &v) 
{  
#ifdef MMAPPED
  if (mmapped) 
    {
      if ((tmp = (char *)(v.memptr())) != NULL) 
	{
	  find_centroid(tmp,&win_spec,&res);
	  
	  if (res.status & CYC_TRACK) 
	    {	    
	      // Update BasicBlob parameters	    
	      return 1;
	    }
	  else
	    { 
	      puts("BasicBlob: not in view."); 
	      return 0;
	    }
	}
      else
	{
	  return 0;
	}
    }
  else 
    {

#endif
      
      float fx = round(x());
      float fy = round(y());
      //cout << "grabbing samp "  << width_sampling() << " " << height_sampling() << endl;
      source()->grab_aligned(im_buffer,
			     fx,fy,
			     width_sampling(),height_sampling());               
      return calc_centroid(im_buffer);       
    }
#ifdef MMAPPED
}
#endif

// void BasicBlob::show (CWindow &w) 
//{w.show(im_buffer.magnify(width_sampling(),height_sampling())); }


