module Window
	( openConsole     -- Bool -> Bool -> IO Console
        , closeConsole    -- Console -> IO ()
        , createWindow    -- Console -> IO Window
        , destroyWindow   -- Window -> IO ()
        , openWindow      -- Window -> Size -> String -> IO () 
        , closeWindow     -- Window -> IO ()
        , newSizedWindow  -- Console -> Size -> String -> IO Window
        , showImage       -- Window -> ImageRGB -> IO ()
        , clearWindow     -- Window -> Color -> IO ()
        , flushWindow     -- Window -> IO ()
        , displayWins     -- Console -> [[ImageRGB]] -> IO ()
	) where

import XVision
import Prelude hiding (sum)
import IOExts
import XVTypes
import XVUtilities
import Geometry(toISize)
import Image

-- Console stuff thrown in here for now.

----------------------------------------------------------------
-- Windows
----------------------------------------------------------------

clearWindow = clearW
flushWindow = flushW

-- Use Size instead of ISize here:
openWindow :: Window -> Size -> String -> IO () 
openWindow w sz name = openWindowI w (toISize sz) name

newSizedWindow :: Console -> Size -> String -> IO Window
newSizedWindow c sz nm =
  do 
   w <- createWindow c
   openWindow w sz nm
   return w



-- Display a set of image streams, one per window
-- Window attached to an image

type Win = (Window,IORef [ImageRGB])

displayWins :: Console -> [[ImageRGB]] -> IO ()
displayWins c iss = do
   ws <- openWins c iss
   while (return True) $ do
     prodWins ws
     putChar '*'  -- hack to let me guestimate frame rate
    
   closeWins ws


openWins :: Console -> [[ImageRGB]] -> IO [Win]
openWins c iss = mapem iss (\ is -> openWin c "A Window" is)

prodWins :: [Win] -> IO ()
prodWins = mapM_ prodWin

closeWins :: [Win] -> IO ()
closeWins = mapM_ prodWin

----------------------------------------------------------------
-- Win
----------------------------------------------------------------

openWin :: Console -> String -> [ImageRGB] -> IO Win
openWin c nm (i:is) = do
  { w <- createWindow c
  ; openWindow w (sizeOf i) nm
  ; showImage w i
  ; v <- newIORef is
  ; return (w,v)
  }

closeWin :: Win -> IO ()
closeWin (w,is) = do
  { closeWindow w
  ; destroyWindow w
  }

prodWin :: Win -> IO ()
prodWin (w,v) = do
  { is <- readIORef v
  ; case is of 
    { (j:js) -> do
                { showImage w j
                ; writeIORef v js
                }
    ; [] -> do
            { putChar '?'
            ; return () 
            }
    }
  }

