
module Transform where

-- A 2D transformation, similar to Conal's Transform2.

import StaticTypesCore hiding (
           Transform2
         , Transformable2
         , (*%)
         , translateUscaleRotate2
         , identity2             
         , translate2            
         , rotate2               
         , compose2              
         , uscale2               
         , inverse2              
         , factorTransform2)      

infixr 7 *%,`compose2`           -- transform apply and compose


-- Form is translate of scale of rotate, with the rotation angle in
-- radians.

data Transform2 =
    TScale Vector2 Radians RealVal -- Translation, rotation, and  scale
  | TAffine Scalar Scalar Scalar Scalar Scalar Scalar -- Full matrix
 deriving Show

-- TAffine a b c d e f =
--  a d 0
--  b e 0
--  c f 1

translateUscaleRotate2 ::  Vector2 -> RealVal -> Radians -> Transform2
translateUscaleRotate2 = TScale

factorTransform2 :: Transform2 -> (Vector2, RealVal, Radians)
factorTransform2 (TScale v s a ) = (v,s,a)
factorTransform2 _ = error "Too lazy to factor general transforms!"

identity2 :: Transform2
identity2 = (TScale 0 1 0)

rotate2 :: RealVal -> Transform2
rotate2 theta = (TScale 0 1 theta)

-- For now just uniform scale, sorry.  The reason is that it's harder to
-- do the normalization with non-uniform scale.
uscale2 :: RealVal -> Transform2
uscale2 sc = TScale 0 sc 0

iscale2 :: Int -> Transform2
iscale2 sc = TScale 0 (fromInt sc) 0

translate2 :: Vector2 -> Transform2
translate2 v =  TScale v 1 0

compose2 :: Transform2 -> Transform2 -> Transform2
xfO@(TScale motO scO rotO) `compose2` TScale motI scI rotI =
  TScale mot sc rot
 where
   -- Move motI left through rotO and scO and add motO.  (Recall that
   -- translation has no effect on vectors.)
   mot = motO + xfO *% motI
   -- Then move scI left past rotO.  Uniform scale commutes with rotation.
   sc  = scO  * scI
   rot = rotO + rotI
t1 `compose2` t2 =
  case (asAffine2 t1, asAffine2 t2) of
    (Just (a1,b1,c1,d1,e1,f1), Just (a2,b2,c2,d2,e2,f2)) ->
      TAffine (a1*a2+d1*b2)
              (b1*a2+e1*b2)
              (c1*a2+f1*b2+c2)
              (a1*d2+d1*e2)
              (b1*d2+e1*e2)
              (c1*d2+f1*e2+f2)
    _ -> error "Can't compose in Transform2.compose2"
 
inverse2 :: Transform2 -> Transform2
inverse2 (TScale mot sc rot) =
            rotate2    (-rot)
 `compose2` uscale2    (1/sc)
 `compose2` translate2 (- mot)
inverse2 _ = error "Can't invert in Transform2.inverse2"

class Transformable2 a where
  (*%)  ::  Transform2 -> a -> a

instance (Transformable2 a, Transformable2 b) => Transformable2 (a,b) where
  xf *% (a,b) = (xf *% a, xf *% b)

instance (Transformable2 a, Transformable2 b, Transformable2 c)
  => Transformable2 (a,b,c) where
  xf *% (a,b,c) = (xf *% a, xf *% b, xf *% c)

instance Transformable2 Point2 where
 TScale (Vector2XY dx dy) scale angle *% Point2XY x y =
   Point2XY (dx + scale * (x * c - y * s))
            (dy + scale * (x * s + y * c))
   where
     c = cos angle
     s = sin angle
 TAffine a b c d e f *% Point2XY x y =
   Point2XY (a*x+b*y+c) (d*x+e*y+f)

-- Vector transformation is defined as customary, by applying just the
-- linear portion (no translation).
instance Transformable2 Vector2 where
 TScale _ scale angle *% Vector2XY x y =
   Vector2XY (scale * (x * c - y * s))
             (scale * (x * s + y * c))
    where
      c = cos angle
      s = sin angle
 TAffine a b c d e f *% Vector2XY x y =
   Vector2XY (a*x+b*y) (d*x+e*y)  -- ??? jcp 

-- New functions added by jcp

isIdentity2 :: Transform2 -> Bool
isIdentity2 (TScale 0 1 0) = True
isIdentity2 (TAffine 1 0 0 0 1 0) = True
isIdentity2 _ = False

asUScale2 :: Transform2 -> Maybe (Vector2, RealVal, Radians)
asUScale2 (TScale v s a) = Just (v,s,a)
asUScale2 _ = Nothing -- too lazy to factor matrix

asAffine2 :: Transform2 ->
               Maybe (RealVal,RealVal,RealVal,RealVal,RealVal,RealVal)
asAffine2 (TScale (Vector2XY x y) u a) = Just (u*c,u*s,x,u*(-s),c,y) where
         c = cos a
         s = sin a
asAffine2 (TAffine x1 x2 x3 x4 x5 x6) = Just (x1,x2,x3,x4,x5,x6)
asAffine2 _ = Nothing

scale2XY :: RealVal -> RealVal -> Transform2
scale2XY x y = TAffine x 0 0 0 y 0

-- could add a shear too.
