----------------------------------------------------------------
-- Stream Processors
--
-- Choice of symbols is inspired by John Hughes' Arrow combinators
----------------------------------------------------------------

module SP
        ( module SP
        ) where

import Pipe
import IOExts( unsafePerformIO )
import XVTypes 
import Error

-- Ought to be in XVTypes:
type SP a b = Pipe a -> Pipe b

-- Ought to be in XVTypes:
-- Not clear if this should be here - but it is used an awful lot
type Tracker a = SP a (WithError a)

-- use this guy on the root of the tree
runSP :: Eval a => (a -> IO ()) -> a -> Tracker a -> IO ()
runSP display init trk =
   runPipe (lift1 (display . valOf) (loopSP init valOf trk))

loopSP :: a -> (b -> a) -> SP a b -> Pipe b
loopSP a0 f sp = let b = sp (delay a0 a)
                     a = lift1 f b
                 in b

sp0 :: b -> SP a b
sp0 b as = lift0 b

sp1 :: (a -> b) -> SP a b
sp1 f as = lift1 f as

sp2 :: (a -> b -> c) -> SP (a,b) c
sp2 f as = lift2 f (lift1 fst as) (lift1 snd as)

sp3 :: (a -> b -> c -> d) -> SP (a,b,c) d
sp3 f as = lift3 f (lift1 pi1 as) (lift1 pi2 as) (lift1 pi3 as)
 where
  pi1 (a,b,c) = a
  pi2 (a,b,c) = b
  pi3 (a,b,c) = c

sp4 :: (a -> b -> c -> d -> e) -> SP (a,b,c,d) e
sp4 f as = lift4 f (lift1 pi1 as) (lift1 pi2 as) (lift1 pi3 as) (lift1 pi4 as)
 where
  pi1 (a,b,c,d) = a
  pi2 (a,b,c,d) = b
  pi3 (a,b,c,d) = c
  pi4 (a,b,c,d) = d

spIO1 :: (a -> IO b) -> SP a b
spIO1 f as = liftIO1 f as

spTrace :: (a -> IO ()) -> SP a a
spTrace f as = liftIO1 (\a -> do { f a; return a }) as

-- A stateful stream processor 
statefulSP :: IO s -> (s -> a -> IO b) -> SP a b
statefulSP init step = unsafePerformIO $ do
  { s <- init
  ; return (spIO1 (step s))
  }

(>>>) :: SP a b -> SP b c -> SP a c
(f >>> g) as = g (f as)

(&&&) :: SP a b -> SP a c -> SP a (b,c)
(f &&& g) as = lift2 (,) (f as) (g as)

(***) :: SP a1 b1 -> SP a2 b2 -> SP (a1,a2) (b1,b2)
(f *** g) as = lift2 (,) (f (lift1 fst as)) (g (lift1 snd as))

par2 :: SP a1 b1 -> SP a2 b2 -> SP (a1,a2) (b1,b2)
par2 f1 f2 as = lift2 (,) (f1 (lift1 fst as)) (f2 (lift1 snd as))

par3 :: SP a1 b1 
     -> SP a2 b2 
     -> SP a3 b3
     -> SP (a1,a2,a3) (b1,b2,b3)
par3 f1 f2 f3 as = 
  lift3 (,,) 
    ((f1.lift1 pi1) as)
    ((f2.lift1 pi2) as)
    ((f3.lift1 pi3) as)
 where
  pi1 (a1,a2,a3) = a1
  pi2 (a1,a2,a3) = a2
  pi3 (a1,a2,a3) = a3

par4 :: SP a1 b1 
     -> SP a2 b2 
     -> SP a3 b3
     -> SP a4 b4 
     -> SP (a1,a2,a3,a4) (b1,b2,b3,b4)
par4 f1 f2 f3 f4 as = 
  lift4 (,,,) 
    ((f1.lift1 pi1) as)
    ((f2.lift1 pi2) as)
    ((f3.lift1 pi3) as)
    ((f4.lift1 pi4) as)
 where
  pi1 (a1,a2,a3,a4) = a1
  pi2 (a1,a2,a3,a4) = a2
  pi3 (a1,a2,a3,a4) = a3
  pi4 (a1,a2,a3,a4) = a4


----------------------------------------------------------------
-- Building composite trackers out of simple trackers
----------------------------------------------------------------

composite2 :: (Eval a, Eval a1, Eval a2) =>
              (a -> (a1,a2)) 
           -> (a1 -> a2 -> a)
           -> Tracker a1
           -> Tracker a2
           -> Tracker a
composite2 split join t1 t2 = sp1 split >>> par2 t1 t2 >>> sp2 (liftE2 join)

composite3 :: (Eval a, Eval a1, Eval a2, Eval a3) =>
              (a -> (a1,a2,a3)) 
           -> (a1 -> a2 -> a3 -> a)
           -> Tracker a1
           -> Tracker a2
           -> Tracker a3
           -> Tracker a
composite3 split join t1 t2 t3 = sp1 split >>> par3 t1 t2 t3 >>> sp3 (liftE3 join)

composite4 :: (Eval a, Eval a1, Eval a2, Eval a3, Eval a4) =>
              (a -> (a1,a2,a3,a4)) 
           -> (a1 -> a2 -> a3 -> a4 -> a)
           -> Tracker a1
           -> Tracker a2
           -> Tracker a3
           -> Tracker a4
           -> Tracker a
composite4 split join t1 t2 t3 t4 = sp1 split >>> par4 t1 t2 t3 t4 >>> sp4 (liftE4 join)

----------------------------------------------------------------
-- Tracing trackers
----------------------------------------------------------------

traceSP :: (Eval a, Show a) => String -> Tracker a -> Tracker a
traceSP s t = spTrace before >>> t >>> spTrace after
 where
  before x = putStrLn $ s ++ " before: " ++ show x
  after  x = putStrLn $ s ++ " after:  " ++ show x

----------------------------------------------------------------
-- End
----------------------------------------------------------------
