/*
    Copyright (C) 1996,1997 Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.
*/

//&&Section pipe_test
/** pipe_test performs "live" video display from an input device.  
    Various parameters such as input device type, position, orientation, and 
    sampling rate can be modified as described below.  The XConsole and 
    XWindow classes are demonstrated, as well as the Video class.    
 **/

//&&Usage: pipe_test [OPTIONS]
//*     -dev            use a specific device type for video input
//*                       [K2T_MONO K2T_COLOR IT_FG101 DT3155
//*                        METEOR_MONO METEOR_COLOR24 METEOR_COLOR16
//*                        INDYCAM_MONO INDYCAM_COLOR]
//*     -mpeg           use a mpeg in place of video input. 
//*                       [filename]          
//*     -image_sequence use an image sequence in place of video input
//*                     your images should be monochrome,labeled with a basename
//*                     followed by a number, like image1, image2, image3, etc.
//*                     the first image in the sequence should be image0.
//*                     the filename to pass is then "image%d", without the quotes.
//*                       [filename]         
//*     -color_convert  project color video to a monochrome image of type
//*                       [R]ed,[G]reen,[B]lue,
//*                       [L]uminance,[S]aturation,[H]ue,[I]ntensity
//*     -height         height (number of rows) in window
//*                       [integer]          
//*     -width          width (number of columns) in window         
//*                       [integer]          
//*     -centerx        x coordinate of center of window
//*                       [integer]          
//*     -centery        y coordinate of center of window
//*                       [integer]          
//*     -widthsamp      width sampling in window (in pixels)  
//*                       [integer]          
//*     -heightsamp     height sampling in window (in pixels)
//*                       [integer]          
//*     -angle          orientation of window (in degrees)
//*                       [float]          
//*     -spin           incremental change in orientation of window 
//*                       [float]          
//*     -rate           calculates frame rate every n cycles, where n > 0,
//*                     and prints the framerate. The default is 
//*                     0 (no computation or display)
//*                       [integer]  
//&&endUsage


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <rate.hh>
#include <site_config.h>
#include <Tracker.hh>
#include <XConsole.hh>
#include <Devices.hh>
#include <Image.hh>
#include "deviceparse.hh"
#include <Pipeline.hh>


//&&stopdoc
char * usage_string = "\n" 
"-----------------------------USAGE------------------------------------\n"
"pipe_test:\n"   
"     -dev            [K2T_MONO K2T_COLOR IT_FG101 DT3155\n"
"                      METEOR_MONO METEOR_COLOR24 METEOR_COLOR16\n"
"                      INDYCAM_MONO INDYCAM_COLOR]\n" 
"     -mpeg           [filename]         \n" 
"     -image_sequence [filename] -a bit tricky, please consult documentation\n" 
"     -color_convert  [R G B L S H I]       \n" 
"     -height         [height of image window in pixels]\n" 
"     -width          [width of image window in pixels]\n" 
"     -centerx        [center of image window along x axis]\n"
"     -centery        [center of image window along y axis]\n"
"     -widthsamp      [sampling rate (in pixels) along x axis]\n"
"     -heightsamp     [sampling rate (in pixels) along y axis]\n"
"     -angle          [orientation (in degrees) at which to grab image]\n"
"     -spin           [change in orientation (in degrees) per timestep]\n"
"     -rate           [number of cycles used to calculate frame rate]\n"  
 ; 
//&&startdoc


//&&Subsection Annotated Main Program
int
main(int argc, char *argv[]) {
  //* buffer to hold name used to initialize MPEG or image sequence 
  char filenamebuf[256];
  sprintf(filenamebuf, "mpeg_file.mpg");

  //* parameters for grabbing from video input, and displaying image
  int height = -1;
  int width = -1;
  float xloc = -1;
  float yloc = -1;

  //* information used to compute rate (program performance)
  //* Usualy, rate information is computed by tracker elements, but
  //* this demonstration program does not perform tracking.
  int COMPUTE_RATE_EVERY_N_FRAMES = 0; 

  //* these variables hold information about the color capabilities 
  //* of the input device
  int is_color_device = 0;
  Device_output_type in_color = PIX_PACKED;

  //* sampling rates for height, and width
  int ws = 1;
  int hs = 1;

  //* angle to grab at, and change in angle (used in -spin mode).    
  float angle = 0;
  float delta_angle = 0;

  //* used in for loops
  int i;

  //* A pointer to a video device, the software abstraction for video input.
  //* First, we check to see if the user fores a specific type of video device.
  //* If not, then we decide what kind of video device we have here and
  //* set the appropriate default color mapping.  These values
  //* are inherited from files included in the site_config.h file.
  Video *v = NULL;    

  //* all arguments for this program come in pairs of two
  for (i = 1; i < argc; i += 2) {
    if ((i+1) == argc) {
      cerr << "Warning, please check that you"
	   << " are including an argument to every flag, if you "
	   << "don't the program will probably segfault" << endl;
      //* just a warning in case user inputs an odd number of arguments 
    };


    if (strcmp(argv[i], "-dev") == 0) {
      //* a special purpose routine located in "deviceparse.cc", 
      //* 'make_device' takes advantage of the C preprocessor to compile 
      //* appropriate Video class constructors based on what the user 
      //* specifies in "site_config.h"
      v = make_device(&is_color_device, argv[i+1]);      
      //* If the device type is unknown, the flag and 
      //* its argument are ignored
      if (v == NULL) {     
	printf("Unknown device type %s encountered, and ignored \n", 
	       argv[i+1]);      
      }
    } 

    //* Rudimentary argument parsing
    else if (strcmp(argv[i], "-color_convert") == 0) {

      //* If the user specifies that they want a color conversion,
      //* we assume that they have provided a device capable of color 
      //* and set this flag

      is_color_device = 1;
      switch (argv[i+1][0]) {
      case 'L':
	cerr << "Video device will convert color images to a PIX_LUMINANCE representation" << endl; 
	in_color = PIX_LUMINANCE;
	break;
	
      case 'R':
	cerr << "Video device will convert color images to a PIX_RED representation" << endl; 
	in_color = PIX_RED;
	break;
	
      case 'G':
	cerr << "Video device will convert color images to a PIX_GREEN representation" << endl; 
	in_color = PIX_GREEN;
	break;

      case 'B':
	cerr << "Video device will convert color images to a PIX_BLUE representation" << endl; 
	in_color = PIX_BLUE;
	break;

      case 'I':
	cerr << "Video device will convert color images to a PIX_INTENSITY representation" << endl; 
	in_color = PIX_INTENSITY;
	break;

      case 'H':
	cerr << "Video device will convert color images to a PIX_HUE representation" << endl; 
	in_color = PIX_HUE;
	break;

      case 'S':
	cerr << "Video device will convert color images to a PIX_SATURATION representation" << endl; 
	in_color = PIX_SATURATION;
	break;

      default:
	printf("Unknown color mode %c \n", argv[i+1][0]);
	break;
      }
    }
    else  if (strcmp(argv[i], "-height") == 0) {
      height = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-width") == 0) {
      width = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-centerx") == 0) {
      xloc = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-centery") == 0) {
      yloc = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-widthsamp") == 0) {
      ws = atoi(argv[i+1]);
    }
    else if ((strcmp(argv[i], "-heightsamp") == 0)) {
      hs = atoi(argv[i+1]);
    }
    else if ((strcmp(argv[i], "-spin") == 0)) {
      delta_angle = atof(argv[i+1])*M_PI/180.0;
    }
    else if ((strcmp(argv[i], "-angle") == 0)) {
      angle = atof(argv[i+1])*M_PI/180.0;
    }
    else if ((strcmp(argv[i], "-mpeg") == 0)) {
      cerr << "Opening MPEG video stream for file " 
	   << argv[i+1] << endl;
      v = new MPEG(argv[i+1]);
    }
    else if ((strcmp(argv[i], "-image_sequence") == 0)) {
      cerr << "Opening *Monochrome* ImageSequence starting with file "
	   << argv[i+1] << "0 " <<  endl;
      ImageSequence_Mono* bla = new ImageSequence_Mono(argv[i+1]);
      bla->auto_step(1);
      v = (Video*)bla;
    }
    else if (strcmp(argv[i], "-rate") == 0) {		  
      if (argc == i+1)
	{
	  printf("\n%s\n", usage_string);
	  exit(1);
	};
      COMPUTE_RATE_EVERY_N_FRAMES = atoi(argv[i + 1]);
      if (COMPUTE_RATE_EVERY_N_FRAMES <= 0)
	{
	  printf("Cycle count for computing frame rate must be > 0\n");
	  exit(1);
	};
    }
    else {
      cerr << "Unknown flag " << argv[i] << endl;
      cerr << usage_string << endl;
      exit(-1);
    }

  }   


  //* If a Video device has not been specified, we create it.  
  if (v == NULL)
    {
      //* Since the pointer is NULL, we want to know whether the 
      //* user specified a display conversion from color output
      //* in the video device, but did not specify the type of device 
      if (is_color_device)
	{
	  //* There is no device specified, but there is a color
	  //* conversion requested, so this preprocessor 
	  //* switch tries to create a default color video constructor
	  //* if one has been compiled 

	  //* Preprocessor directives compile either this code...
  #ifndef DEFAULT_COLOR_DECLARATION
	cerr << "You have tried to convert color video input to "
	     << "a different representation but have not compiled "
	     << "to use a DEFAULT Video class that supports color."
	     << "look in the site_config.h file" << endl;
      exit(-1);
  #else
      //* ...or this code.
      cerr << "Video device not specified. "
	   << "Initializing with DEFAULT_COLOR_DECLARATION.  "
	   << "To override specify device name or image stream with -dev, -mpeg, or -image_sequence flags" << endl;      	  
      v = new DEFAULT_COLOR_DECLARATION;		
#endif 
	}
      else
	{
	  //* we just create the default video device, since the k	
	  //* user specified neither a device type nor a color conversion
	  cerr << "Video device not specified.  "
	       << "Initializing with DEFAULT_DECLARATION.  "
	       << "To override specify device name or image "  
	       << "stream with -dev, -mpeg, or -image_sequence flags" 
	       << endl;      	  
	  v = new DEFAULT_DECLARATION;
	};
    }
     
  //* Try to set the output type of the color video device.  If it exists,
  //* this call will convert the color information from a video feed into
  //* the requested format.  If the the video device pointed to by 'v'
  //* doesn't support color but the flag is set, (i.e. the user
  //* specified a both a color conversion and a monochrome device) 
  //* the call will fail and generate an error message.
  if (is_color_device)
    v->set_grab_type(in_color);
    

  //* Create a pipeline of whatever type you want
  Pipeline pl(v);
  Pipe_Params pp;

  pp.threshold = 50;
  pp.width = 7;
  pp.sigma_x = 2;

  XConsole c(0);

  // Smooth the image a bit
  pl = PL_Gaussian(1,pl,pp.width,pp.sigma_x);

  //  Pipeline pl1 = PL_Dx(-1,pl);
  //  Pipeline pl2 = PL_Dy(-1,pl);
  //  pl1.display_output(c);
  //  pl2.display_output(c);

  // Take the magnitude of the derivative

  //  pl = PL_Sqrt(-1,PL_Sq(-1,PL_Dx(-1,pl)) +
  //  	                 PL_Sq(-1,PL_Dy(-1,pl)));

  //  pl = PL_Sqrt(-1,PL_Sq(-1,pl1) + PL_Sq(-1,pl2));
 
  //  pl = PL_Sobel(-1,pl);
  Pipeline pl1 = PL_Abs(-1,PL_Dx(-1,pl));
  pl1.display_output(c); 

  Pipeline pl2 = PL_Abs(-1,PL_Dy(-1,pl));
  pl2.display_output(c);
    
  pl = PL_Sqrt(-1,PL_Sq(-1,pl1) + PL_Sq(-1,pl2));

			    //pl = pl-PL_Delay(-1,pl);
  //  pl = PL_Threshold(1,,pl),175);
  //pl = PL_Threshold(1,pl,pp.threshold);

  Pipeline_Video pl_v(&pl,pl.get_height(),pl.get_width());

  //* If these parameters have not been set, we set them now, to the default dimensions 
  //* of the video device.  Note that the x and y location of the widow are given with 
  //* respect to the center of the window, and NOT the uper left hand corner
  if (height < 0) 
    height = pl_v.height();
  if (width < 0)
    width = pl_v.width();
  if (xloc < 0)
    xloc = pl_v.width()/2.0;
  if (yloc < 0)
    yloc = pl_v.height()/2.0;

  //* Create a XConsole data structure of the appropriate type.
  //* XConsole only knows how to display color images of type 
  //* PIX_PACKED so it displays monochrome unless this type of 
  //* Video information has been requested
  int w_color = (pl_v.output_data_type() == PIX_PACKED);    
  //  XConsole c(w_color);
  if (w_color)
    cout << "Created color console" << endl;
  else
    cout << "Created monochrome console" << endl;
 
  //* Create a window to be managed by the XConsole and to display 
  //* the image in
  XWindow w(c);

  //* Create an image of the appropriate size.  Image is a stub 
  //* class derived from the XVImage<Txx> Template.  This template
  //* is not fully integrated into the package and should be used 
  //* with caution.
  Image im(width,height); 
  w.open(width,height,"XVision Display");

  TimingInfo timing_info(COMPUTE_RATE_EVERY_N_FRAMES, 
			1, "showlive");

  //* The main loop --- grab and display until the cows come home.
  i = 1;
  char lname[200];

  while (1) 
    {
      //* This is one way of grabbing data into an image from a video device.
      //* You may notice a different call is used in the grab_single demo.
      //* Here, im.data() gives you a pointer directly to the image data buffer
      //* but width and height of the image must be passed as arguments.
      if (!(pl_v.grab(im.data(),xloc,yloc,width,height,angle,ws,hs))) 
	{
	  cerr << "Desired image out of view." << endl;
	  if (angle == 0.0) 
	    exit(1);
	}
      else 
	{
	  //* inside the body of this if statement we compute rate information
	  //* normally, rate information is computed as part of the 
	  //* state information of a (top level) tracked feature.
	  //* as we are not tracking anything here, the loop has been
	  //* moved into the mainline
	  timing_info.tick();
	  
	  sprintf(lname,"XVision Display: %d", i++);
	  w.show(im,lname);

	  //* increments the angle at which we are grabbing
	  angle += delta_angle;
	  // test?
	  // w.rectangle(xloc,yloc,10,10);	
	}
    }
  
  //* technically, this will never happen, but we put it here anyway, 
  //* to show you how to get rid of windows if you want to be "clean" about it.
  w.close();
};





