/*
    Copyright (C) 1995 Kentaro Toyama and Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.
    */

//&&Section line_test
/** line_test demonstrates simple line tracking using the Edge class.
    

 **/


//&&Usage: line_test [OPTIONS]
//* -dev            use a specific device type for video input
//*                   [K2T_MONO K2T_COLOR IT_FG101 DT3155
//*                    METEOR INDYCAM_MONO INDYCAM_COLOR]
//* -mpeg           use a mpeg in place of video input. 
//*                    [filename]          
//* -image_sequence use an image sequence in place of video input
//*                    [filename]         
//* -length         the length of the line to be tracked  
//*                    [integer]         
//* -searchwidth    the total width of the search window extended
//*                 perpendicular to the target line 
//*                    [integer]         
//* -lengthsamp     the sampling rate along the length of line
//*                    [integer]         
//* -widthsamp      the sampling rate along the width of the search window
//*                    [integer]         
//* -noprint        supresses printing of line information
//*                    []         
//* -nodisplay      supresses target display on monitor 
//*                    []         
//* -rate           calculates frame rate every n cycles, where n > 0,
//*                 and prints the framerate. The default is 
//*                 0 (no computation or display)
//*                    [integer]  
//&&endUsage

#include "site_config.h"
#include "Tracker.hh"
#include "XConsole.hh"
#include "Pattern.hh"
#include "Line.hh"
#include "Devices.hh"
#include "Acquire.hh"
#include <sys/time.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "deviceparse.hh"

//&&stopdoc
char * usage_string = "\n" 
"-----------------------------USAGE------------------------------------\n"
"line_test:\n"   
"     -dev            [K2T_MONO K2T_COLOR IT_FG101 \n"
"                      DT3155 METEOR INDYCAM_MONO INDYCAM_COLOR]\n" 
"     -mpeg           [filename]         \n" 
"     -image_sequence [filename]        \n" 
"     -length         [length (in pixels) of line to track]\n" 
"     -searchwidth    [width (in pixels) of search window around line]\n" 
"     -lengthsamp     [sampling rate (in pixels) along length of line]\n"
"     -widthsamp      [sampling rate (in pixels) perpendicular to line]\n"
"     -noprint         -- supresses printing of line information] \n"
"     -nodisplay       -- supresses target display on monitor \n" 
"     -rate           [number of cycles used to calculate frame rate]\n" ; 
//&&startdoc


main(int argc, char *argv[]) {
  //  e.transition_type() = lightdark;

  int i;
  int length = 30;
  int width = 20;
  int lsamp = 1;
  int wsamp = 1;
  int p = 1;
  int d = 1;
  int r = 1;
  int COMPUTE_RATE_EVERY_N_FRAMES = 0;
  int is_color_device = 0;
  Video *v;

  for (i=1; i<argc ; i++) {
    if (strcmp(argv[i], "-length") == 0) {
      length = atoi(argv[++i]);
    } 
    else if (strcmp(argv[i], "-searchwidth") == 0) {
      width = atoi(argv[++i]);
    } 
    else if (strcmp(argv[i], "-widthsamp") == 0) {
      wsamp = atoi(argv[++i]);
    } 
    else if (strcmp(argv[i], "-lengthsamp") == 0) {
      lsamp = atoi(argv[++i]);
    } 
    else if (strcmp(argv[i], "-noprint") == 0) {
      p = 0;
    } 
    else if (strcmp(argv[i], "-nodisplay") == 0) {
      d = 0;
    }
    else if (strcmp(argv[i], "-dev") == 0) {
      if (argc == i+1)
	{
	  printf("\n%s\n", usage_string);
	  exit(1);
	};
      v = make_device(&is_color_device, argv[i+1]); 
      cerr << "Warning: using color device - grabbing at orientation not"
	   << " supported on color devices " << endl;
      if (v == NULL) {     
	printf("Unknown device type %s encountered, and ignored \n", 
	       argv[i+1]);      
      };
    }
    else if ((strcmp(argv[i], "-mpeg") == 0)) {
      cerr << "Opening MPEG video stream for file " 
	   << argv[i+1] << endl;
      v = new MPEG(argv[i+1]);
    }
    else if ((strcmp(argv[i], "-image_sequence") == 0)) {
      cerr << "Opening !Monochrome ImageSequence starting with file "
	   << argv[i+1] << "0 " <<  endl;
      v = new ImageSequence_Mono(argv[i+1]);
    }
    else if (strcmp(argv[i], "-rate") == 0) {		  
      if (argc == i+1)
	{
	  printf("\n%s\n", usage_string);
	  exit(1);
	};
      COMPUTE_RATE_EVERY_N_FRAMES = atoi(argv[++i]);
      if (COMPUTE_RATE_EVERY_N_FRAMES <= 0)
	{
	  printf("Cycle count for computing frame rate must be >= 0\n");
	  exit(1);
	};
    } else
      {		    
	printf("\n%s\n", usage_string);
	exit(1);
      };
  }
      
  XConsole c;
  XWindow w(c);
  XWindow w2(c);
  Edge e;

  if (v == NULL)
    {
      cerr << "Video device not specified.  "
	   << "Initializing with DEFAULT_DECLARATION.  "
	   << "To override specify device name or image "  
	   << "stream with -dev, -mpeg, or -image_sequence flags" 
	   << endl;      	  
      v = new DEFAULT_DECLARATION();
    }
  else
    {
      if (is_color_device)
	v->set_grab_type(PIX_LUMINANCE);
    };

  printf("width=%d, height=%d\n", v->width(), v->height());
  puts("constructing line...");
  Line l1 (&e,v,length,width);
  l1.length_sampling() = lsamp;
  l1.width_sampling() = wsamp;
  Image *iml = l1.image();

  puts("initializing line...");
  l1.interactive_init (w);
  w.close();

  w2.open(iml->width(),iml->height(),"Line Display");

  if (p) cout << l1;

  i = 1;
  static timeval oldtv = {0,0};  
  while (1) {
    l1.track();
    if (d) w2.show(*iml);
    if (p) cout << l1 << endl;
    if (COMPUTE_RATE_EVERY_N_FRAMES > 0) 
      {
	i++;
	i %= COMPUTE_RATE_EVERY_N_FRAMES;
	if (!(i))
	  {	
	    timeval newtv;
	    
	    gettimeofday(&newtv, NULL);
	    
	    int dsec = newtv.tv_sec - oldtv.tv_sec;
	    int dusec = newtv.tv_usec - oldtv.tv_usec;
	    
	    double msecs = dsec  * 1000 + dusec / 1000;
	    msecs /= COMPUTE_RATE_EVERY_N_FRAMES;  /* this gets average time 
						      per frame computation */
	    oldtv = newtv;

	    printf("RATE: %4.1f Hz \n", (1000 / msecs));
	    
	  };
      };
  }
  w2.close();
}

