
/*
    Copyright (C) 1995 Gregory D. Hager and Kentaro Toyama (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/


//&&Section cornertest
/** 
  cornertest demonstrates how composite feature trackers can be composed of simple 
  feature trackers -- in this case two line trackers are combined to track a corner.
   
 **/

//&&Usage: cornertest [OPTIONS]
//* -dev            use a specific device type for video input
//*                   [K2T_MONO K2T_COLOR IT_FG101 DT3155
//*                    METEOR_COLOR24 METEOR_COLOR16 METEOR_MONO
//*                    INDYCAM_MONO INDYCAM_COLOR]
//* -mpeg           use a mpeg in place of video input. 
//*                    [filename]          
//* -image_sequence use an image sequence in place of video input
//*                    [filename]         
//* -length         the length of the line to be tracked  
//*                    [integer]         
//* -searchwidth    the total width of the search window extended
//*                 perpendicular to the target line 
//*                    [integer]         
//* -lengthsamp     the sampling rate along the length of line
//*                    [integer]         
//* -widthsamp      the sampling rate along the width of the search window
//*                    [integer]         
//* -noprint        supresses printing of line information
//*                    []         
//* -nodisplay      supresses target display on monitor 
//*                    []         
//* -showgrab       shows the area that the tracker is grabbing in
//*                 the context of the interactive init window
//*                    []
//* -rate           calculates frame rate every n cycles, where n > 0,
//*                 and prints the framerate. The default is 
//*                 0 (no computation or display)
//*                    [integer]  
//&&endUsage
 

//&&stopdoc
char * usage_string = "\n" 
"-----------------------------USAGE------------------------------------\n"
"cornertest:\n"   
"     -dev            [K2T_MONO K2T_COLOR IT_FG101 \n"
"                      DT3155 METEOR_COLOR24 METEOR_COLOR16 METEOR_MONO \n"
"                      INDYCAM_MONO INDYCAM_COLOR]\n" 
"     -mpeg           [filename]         \n" 
"     -image_sequence [filename]        \n"   
"     -length         [length (in pixels) of line to track]\n" 
"     -searchwidth    [width (in pixels) of search window around line]\n" 
"     -lengthsamp     [sampling rate (in pixels) along length of line]\n"
"     -widthsamp      [sampling rate (in pixels) perpendicular to line]\n"
"     -noprint         -- supresses printing of line information] \n"
"     -nodisplay       -- supresses target display on monitor \n" 
"     -showgrab        -- shows the area that the tracker is grabbing in\n"
"                         the context of the interactive init window\n" 
"     -rate           [number of cycles used to calculate frame rate]\n" 
"     -follow         [follow edge to corner when initializing] \n"  
"     -linedir        [toggle direction of line from forward to backward -- \n"
"                      relevant for follow init]\n"
"     -cornerdir      [toggle direction of corner from right to left -- \n"
"                      relevant for follow init]\n";
//&&startdoc
#include "Tracker.hh"
#include "XConsole.hh"
#include "Pattern.hh"
#include "Line.hh"
#include "GILine.hh"
#include "Devices.hh"
#include "MPEG.hh"
#include <stdlib.h>
#include <stream.h>
#include "deviceparse.hh"
 
//&&Subsection Annotated Main Program
main(int argc, char *argv[]) {
  int i;
  int length=40;
  int width=40;
  int lsamp=1;
  int wsamp=1;
  int p = 1;
  int d = 1;
  int showgrab = 0;
  int follow_search = 0;
  int line_dir = 1;          // These defaults make the search go forward 
  int corner_dir = 1;        //   and look for an corner that turns right.
  int is_color_device = 0;
  int COMPUTE_RATE_EVERY_N_FRAMES = 0;
  Video *v = NULL;

  //* Lots of argument parsing happens here....
  for (i=1; i < argc ; i++) {
    if (strcmp(argv[i], "-length") == 0) {
      length = atoi(argv[++i]);
    } 
    else if (strcmp(argv[i], "-width") == 0) {
      width = atoi(argv[++i]);
    }
    else if (strcmp(argv[i], "-widthsamp") == 0) {
      wsamp = atoi(argv[++i]);
    } 
    else if (strcmp(argv[i], "-lengthsamp") == 0) {
      lsamp = atoi(argv[++i]);
    } 
    else if (strcmp(argv[i], "-noprint") == 0) {
      p = 0;
    } 
    else if (strcmp(argv[i], "-showgrab") == 0) {
      showgrab = 1;
    } 
    else if (strcmp(argv[i], "-nodisplay") == 0) {
      d = 0;
    }
    else if (strcmp(argv[i], "-follow") == 0) {
      follow_search = 1;
    }
    else if (strcmp(argv[i], "-cornerdir") == 0) {      
      corner_dir *= -1;
    }
    else if (strcmp(argv[i], "-linedir") == 0) {
      line_dir *= -1;
    }
    else if (strcmp(argv[i], "-dev") == 0) {
      if (argc == i+1)
	{
	  printf("\n%s\n", usage_string);
	  exit(1);
	};
      //* a special purpose routine located in "deviceparse.cc", 
      //* 'make_device' takes advantage of the C preprocessor to compile 
      //* appropriate Video class constructors based on what the user 
      //* specifies in "site_config.h"
      v = make_device(&is_color_device, argv[++i]);      
      //* If the device type is unknown, the flag and 
      //* its argument are ignored
      if (v == NULL) {     
	printf("Unknown device type %s encountered, and ignored \n", 
	       argv[i]);      
      }
    }
    else if ((strcmp(argv[i], "-mpeg") == 0)) {
      cerr << "Opening MPEG video stream for file ";
      fflush(stderr);
      cerr << argv[i+1] << endl;
      v = new MPEG(argv[++i]);
      v->set_synchronous();
    }
    else if ((strcmp(argv[i], "-image_sequence") == 0)) {
      cerr << "Opening !Monochrome ImageSequence starting with file ";
      fflush (stderr);
      cerr << argv[i+1] <<  endl;
      v = new ImageSequence_Mono(argv[++i]);
    }
    else if (strcmp(argv[i], "-rate") == 0) {		  
      if (argc == i+1)
	{
	  printf("\n%s\n", usage_string);
	  exit(1);
	};
      COMPUTE_RATE_EVERY_N_FRAMES = atoi(argv[++i]);
      if (COMPUTE_RATE_EVERY_N_FRAMES <= 0)
	{
	  printf("Cycle count for computing frame rate must be > 0\n");
	  exit(1);
	};
    } else
      {		    
	printf("\n%s\n", usage_string);
	exit(1);
      };
  }


  //* if the Video device is still unspecified we can try to create it with 
  //* the DEFAULT_DECLARATION macro
  if (v == NULL)
    {
      cerr << "Video device not specified.  "
	   << "Initializing with DEFAULT_DECLARATION.  "
	   << "To override specify device name or image "  
	   << "stream with -dev, -mpeg, or -image_sequence flags" 
	   << endl;      	  
      v = new DEFAULT_DECLARATION;
    }
  else
    {
      cout << "checking if video device is color" << endl;
      if (is_color_device)
	{
	  v->set_grab_type(PIX_LUMINANCE);
	  cout << "conversion to luminance values set in grab function" << endl;
	}
    };


  //* These are used in the tracking display s
  XConsole c(0);
  XWindow w(c);
  XWindow w2(c);
  Image im(width,length);
 
  //* This is the Pattern used for detection of a Line, it is simply a 
  //* intensity gradient in the region of interest
  Edge e;

  //* these are the component trackers that combine to make a composite feature 
  Line l1(&e,v,length,width);
  Line l2(&e,v,length,width);
  l1.width_sampling() = l2.width_sampling() = wsamp;
  l1.length_sampling() = l2.length_sampling() = lsamp;

  //* This builds the corner tracker out of two lines.
  //* The corner tracker is a type of GILine (General Intersecting Line)
  Corner c1 (l1,l2);

  //* one way to initialize is by growing the first line in one direction
  //* until the end is reached at a place where the other line can be 
  //* initialized 
  if (follow_search) 
    {
      l1.interactive_init(w);
      c1.set_state(l1.x(),l1.y(),l1.orientation(),l1.orientation()+1.57);
      c1.search_init (line_dir, corner_dir);
    } 
  //* the other way to initialize the corner is the intuitive way
  else 
    {
      c1.interactive_init (w);
    }
  
  //* clean up by closing window if we dont need it
  if (!showgrab)
    w.close();

  //* If we want to time the performance of tracking, 
  //* this initializes the timing properties of our 
  //* top-level feature (in our case, the corner).
  if (COMPUTE_RATE_EVERY_N_FRAMES)
    {
      cout << "getting rate information every " << COMPUTE_RATE_EVERY_N_FRAMES
	   << " frames" << endl;
      //* if we are going to be printing Line information, we will get rate
      //* information when we ask for a print out 
      if (p) 
	c1.init_timing(COMPUTE_RATE_EVERY_N_FRAMES, 0);
      //* otherwise, the feature will give rate information 
      //* independently as it computes it
      else	
	c1.init_timing(COMPUTE_RATE_EVERY_N_FRAMES, 1);
    }

  
  //* this is the main loop, tracking occurs here
  while (1) 
    {
      c1.track();
      
      if (d) 
	{
	  v->grab(im,int(c1.x()),int(c1.y()),0);
	  w2.show(im);
	};
      
      if (showgrab) 
	w.show_grab(v);
      
      if (p) 
	cout << c1;
    }  
}

