/*
    Copyright (C) 1991 Gregory D. Hager and Sidd Puri (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/

//&&Section SSDtest
/** 
  SSDtest demonstrates region tracking using the SSD class.   
  The SSD tracker is a gradient descent algorithm that uses
  precomputed bases, or templates, to solve a system of linear
  equations to minimize the error between a rectified 
  region of interest and the template it is supposed to match. 
  The result is a set of parameters describing the 
  change in the ROI in terms of the transformations which have been computed,
  with which the tracking algorithm then updates its state.
 **/


//&&Usage: SSDtest [OPTIONS]
//* -r              tries to account for changes in region rotation about Z axis
//*                    []
//* -s              tries to account for scale changes in region of interest(ROI)
//*                    []
//* -a              tries to account for changes in the aspect ratio of the ROI
//*                    []
//* -o              computes outliers in region of interest, to minimize 
//*                 their affect on tracking
//*                    []
//* -m              computes residuals
//*                    []
//* -h              tries to account for shearing motion in the ROI
//*                    []
//* -p              print tracker information to stdout  
//*                    []
//* -D              use a specific device type for video input
//*                   [K2T_MONO K2T_COLOR IT_FG101 DT3155
//*                    METEOR_COLOR24 METEOR_COLOR16 METEOR_MONO
//*                    INDYCAM_MONO INDYCAM_COLOR]
//* -M              use a mpeg in place of video input. 
//*                    [filename]          
//* -Q              use an image sequence in place of video input
//*                    [filename]       
//* -R              reduction factor.  the factor by which the ROI is reduced
//*                 to reduce the effects of noise
//*                    [int]
//* -W              width of the ROI in pixels
//*                    [int]
//* -H              height of the ROI in pixels 
//*                    [int]
//* -T              time the performance of the algorithm in Hz, report every 
//*                 n iterations, where n is the integer argument ot he flag
//*                    [int]
//* -V              change the value of the variance from the default (5.0), 
//*                 for use in the statistical computation of outliers
//*                    [float]
//* -O              change the value of the outlier threshold from the default 
//*                 (5.0), for use in the statistical computation of outliers
//*                    [float]
//* -I              changes the number of iterations used in the outlier 
//*                 compensation scheme (defaults to 1) 
//*                    [int]
//* -K              changes the number of iterations per image
//*                 (defaults to 1) 
//*                 this only matters if the framegrabber is synchronous.
//*                    [int]
//* -E              changes the number of iterations used in the outlier 
//*                 compensation scheme (defaults to 1) 
//*                    [int]
//* -P              changes the number of iterations used in the outlier 
//*                 compensation scheme (defaults to 1)        
//*                    [int]
//* -C              if the device is color, you can convert from the PIX_PACKED
//*                 data representation (default color input) to pull out the 
//*                 red, blue, or green band, or alternately, compute 
//*                 Hue, Saturation, or Intensity.  By default the device 
//*                 converts to PIX_LUMINANCE, which is the default output type of
//*                 a Monochrome device.
//*                    [R(ed) r(ed) B(lue) b(lue) G(reen) g(reen) H(ue) h(ue)
//*                     S(aturation) s(aturation) I(ntensity) i(ntensity)]
//* -S              controls the display of various information involved in the
//*                 SSD computation. The GRAB option keeps the interactive 
//*                 initialization window open, and shows the ROI being 
//*                 grabbed from the framebuffer.  The LIVE option shows the 
//*                 current reference image. The BASIS option shows a window
//*                 displaying the set of bases representing the 
//*                 various precomputed templates.  The OUTLIERS option shows 
//*                 a window displaying where the outliers are.  The RESIDUALS
//*                 option shows what is going on with the computation of
//*                 residuals.
//*                    [B(asis) b(asis) L(ive) l(ive) G(rab) g(rab) 
//*                     O(utliers) o(utliers) R(esiduals) r(esiduals)]
//&&endUsage
#include "Tracker.hh"
#include "XConsole.hh"
#include "Devices.hh"
#include "SSD.hh"
#include <time.h>
#include <signal.h>
#include <stdlib.h>
#include <fstream.h>
#include "deviceparse.hh"


//&&stopdoc
char OPTION_STRING[] = "rsahompD:M:Q:R:W:H:S:T:C:U:L:V:O:I:K:E:P:";
char * usage_string = "\n" 
"-----------------------------USAGE------------------------------------\n"
"SSDtest:\n"   
"   -D   (device)         [K2T_MONO K2T_COLOR IT_FG101 \n"
"                          DT3155 METEOR_COLOR24 METEOR_COLOR16 \n"
"                          METEOR_MONO INDYCAM_MONO INDYCAM_COLOR I1394]\n"  
"   -M   (mpeg)           [filename]\n" 
"   -Q   (image sequence) [filename]\n" 
"   -R   (reduction)      [reduction factor in pixels]\n"
"   -H   (height)         [height (in pixels) of region of interest window]\n"   
"   -W   (width)          [width (in pixels) of region of interest window]\n"   
"   -T   (timing)         [number of cycles used to calculate frame rate]\n"
"   -V   (variance)       [defaults to 5.0, value used in outlier detection]\n"
"   -O   (outlier thresh) [defaults to 5.0, threshold used in outlier detection]\n"
"   -I   (iterations)     [number iterations used by outlier detection]\n"
"   -K   (iterations)     [number iterations per image]\n"
"   -E   (iterations)     [number iterations used by outlier detection]\n"
"   -P   (iterations)     [number iterations used by outlier detection]\n"
"   -C   (convert)        [R(ed) r(ed) B(lue) b(lue) G(reen) g(reen) H(ue) h(ue)\n"
"                          S(aturation) s(aturation) I(ntensity) i(ntensity)]\n"
"   -S   (show)           [B(asis) b(asis) L(ive) l(ive) G(rab) g(rab) \n" 
"                          O(utliers) o(utliers) R(esiduals) r(esiduals)]\n"
"   -p   (print SSD tracker information)\n"  
"   -r   (compute rotation)\n" 
"   -s   (compute scale)\n" 
"   -a   (compute aspect ratio)\n" 
"   -h   (compute shear)\n" 
"   -m   (compute residuals)\n" 
"   -o   (compute outliers)\n" ;

//&&startdoc
//&&Subsection Annotated Main Program

//* The dimensions for the tracker
int width         = 70;
int height        = 70;

//* smoothing parameter 
int reduction     = 4;

//* boolean variables
int outliers      = 0;
int residuals      = 0;
int rotation      = 0; 
int cscale         = 0;
int aspect        = 0;
int shear         = 0;
int do_timing     = 0;
int show_basis    = 0;
int show_live     = 0;
int show_residual = 0;
int show_outliers = 0;
int show_grab     = 0;
int print_state   = 0;
int is_color_device = 0;

//* pointer to the video device
Video *v = NULL;

//* if you are using a color framegrabber, 
//* this data representation can be changed 
Device_output_type type          = PIX_LUMINANCE;

//* some parameters for SSD statistics, other varibles
float variance    = 5.0;
float outlier_t   = 5.0;
int i_iters     = 1;
int o_iters     = 1;
int e_iters     = 1;
int d_iters     = 1;
int timing_iterations = 0;
char *basis = NULL;

char *myopts[] = {
     #define LIVE        0
                             "live",
     #define BASIS       1
                             "basis",
     #define RESIDUAL    2
                             "residual",
     #define OUTLIERS    3
                             "outliers",
     #define HAT         4
                             "hat",
                             NULL}; 

//&&stopdoc
void
parse_args (int argc, char **argv)
{
  int c;
  extern char *optarg;
  char *options,*value;
  extern int optind;
  int aflg = 0;
  int bflg = 0;
  int errflag = 0;



  while ((c = getopt(argc, argv, OPTION_STRING)) != EOF)
    switch (c) {
    case 'r':
      rotation = 1;
      break;

    case 's':
      cscale = 1;
      break;

    case 'a':
      aspect = 1;
      break;

    case 'h':
      shear = 1;
      break;
 
    case 'o':
      outliers = 1;
      break;

    case 'm':
      residuals = 1;
      break;

    case 'p':
      print_state = 1;
      break;

    case 'R':
      reduction = atoi(optarg);
      break;

    case 'T':
      do_timing = 1;
      timing_iterations = atoi(optarg);
      break;

    case 'W':
      width = atoi(optarg);
      break;

    case 'H':
      height = atoi(optarg);
      break;

    case 'V':
      variance = atof(optarg);
      break;

    case 'I':
      i_iters = atoi(optarg);
      break;

    case 'K':
      o_iters = atoi(optarg);
      break;

    case 'E':
      e_iters = atoi(optarg);
      break;

    case 'P':
      d_iters = atoi(optarg);
      break;

    case 'O':
      outlier_t = atof(optarg);
      break;

    case 'D':
      v = make_device(&is_color_device, optarg); 
      if (v == NULL) {     
	printf("Unknown device type %s encountered, and ignored \n", 
	       optarg);
      }
      break;
      
    case 'M':
      cerr << "Opening MPEG video stream for file " 
	   << optarg << endl;
      v = new MPEG(optarg);
      v->set_synchronous(1);
      break;

    case 'Q':
      cerr << "Opening !Monochrome ImageSequence starting with file "
	   << optarg << "0 " <<  endl;
      v = new ImageSequence_Mono(optarg);
      break;

    case 'C':
      switch (optarg[0]) {

      case 'R':
      case 'r':
	type = PIX_RED;
	break;

      case 'G':
      case 'g':
	type = PIX_GREEN;
	break;

      case 'B':
      case 'b':
	type = PIX_BLUE;
	break;

      case 'h':
      case 'H':
	type = PIX_HUE;
	break;

      case 's':
      case 'S':
	type = PIX_SATURATION;
	break;

      case 'i':
      case 'I':
	type = PIX_INTENSITY;
	break;

      default:
	type = PIX_LUMINANCE;
      }
      break;

    case 'S':
      switch(*optarg) {

      case 'B' :
      case 'b':
	show_basis = 1;
	break;

    case 'L' :
    case 'l' :
      show_live = 1;
      break;

    case 'R' :
    case 'r' :
      show_residual = 1;
      break;
      
      case 'O' :
      case 'o' :
	show_outliers = 1;
	break;
	
      case 'G':
      case 'g':
	show_grab = 1;
	break;
             
    default:
      //* process unknown token 
      errflag++;
      break;
      }
  break;
    
    case '?':
      errflag++;
    }
     
  if (errflag) {
    (void)fprintf(stderr,
		  usage_string);
    exit (2);
  }
}
//&&startdoc


void sighndl(int ws) {
    cerr << "sig called" << endl;
    if(v) delete v;
    exit(0);
}

//* The mainline.
main(int argc, char *argv[]) 
{

  signal(SIGINT, sighndl);
  //* parses the command line arguments
  parse_args(argc,argv);

  //* creates an XConsole to manage display of images...
  XConsole c;
 
  //* interactive initialization happens in this window
  XWindow init_win(c); 

  //* live display of the tracked region happens in this window
  XWindow win_live(c); 

  //* live display of the residuals of the tracked region
  XWindow win_residual(c); 
  
  //* window to show the various (static) basis for the tracker  
  XWindow win_basis(c); 

  //* window to show the outliers
  XWindow win_outliers(c); 

  //* If no device has been declared, try defaults
  if (v == NULL)
    {
      if (type == PIX_LUMINANCE)
	{
	  v = new DEFAULT_DECLARATION;
	}
      else
	{
#ifdef DEFAULT_COLOR_DECLARATION
	  v = new DEFAULT_COLOR_DECLARATION;
	  is_color_device = 1;
#else
	  v = new DEFAULT_DECLARATION;
#endif
	}
    };

  //* grab type is set to PIX_LUMINANCE by default in the 
  //* any Video class deriving from MonoVideo.  This just checks
  //* to see if the user has specified a color conversion but is using 
  //* a monochrome device.  It also insures that we never try to 
  //* reset output type for a monochrome device 
  if (is_color_device)
    v->set_grab_type(type);	
  else
    if (type != PIX_LUMINANCE)
      cerr << "Cannot convert data representation in monochrome device" << endl;
  
  //* initializes the SSD tracker
  SSD* t = new SSD(v,width,height,reduction);

  t->set_oiter(o_iters); 

  //* outliers require ananalysis of the outliers
  //* which in turn requires that othe constants be set,
  //* i.e. thresholds for outlier detection and variance be set
  if (outliers) {
    t->check_residue(1);
    //* this sets the threshold for outlier detection
    t->set_threshold(outlier_t);
    t->set_pixel_sd(variance);
    t->set_iiter(i_iters); 
    t->set_morph(e_iters,d_iters);
  }

  //* computes the residuals from SSD
  if (residuals)
    t->compute_residue(1);

  //* accomodates for changes in rotation about Z axis 
  if (rotation)
    {
      t->compute_rotation(1);
#ifdef ODL 
     if (show_grab)
	{
	  cerr << "Because SSD does its own grabbing at an angle, "
	       << "the rotation option is not available with the "
	       << "Show GRAB option" << endl;
	  show_grab = 0;
	};
#endif
    };



  //* tries to accomodate for changes in scale 
  if (cscale)
    t->compute_scale(1);

  //* this option kills the tracker pretty fast....
  if (aspect)
    t->compute_aspect_ratio(1);

  //* tracker tries to account for shearing motion 
  if (shear)
    t->compute_shear(1);

  //* shows the Trackers reconstructed (magnified after reduction) reference image 
  //* this feature seems to work on and off
  if (show_live)
    t->show_live(&win_live);

  //* shows the various precomputed bases side by side in one window
  //* this display does not change, as the bases are precomputed templates
  //* which remain the same in each iteration the Tracking algorithm
  if (show_basis)
    t->show_basis(&win_basis);

  //* this option does not seem to display anything
  if (show_residual && residuals)
    t->show_residual(&win_residual);
  else 
    if (show_residual)
      cerr << "Must compute residuals to show them" << endl;

  //* if the outliers are being computed, this displays them
  //* at the resolution used by the SSD tracker
  if (show_outliers && outliers)
    t->show_outliers(&win_outliers);
  else
    if (show_outliers)
      cerr << "Must compute outliers to show them" << endl;

  //* this routine allows you to interactively initialize the tracking window
  t->interactive_init (init_win);
    
  //* closes the initialization window, keeping it open only if
  //* we have chosen to show the grabbing being done by the video device
  //* if it remains open it will be used as a display for the Tracker
  if (!show_grab)
    init_win.close();
  
  //* if timing information has been requested, this handles rate information
  //* using the generic interface for trackable objects
  if (do_timing) 
    {
      if (print_state) 
	//* if we are going to be printing tracker information, we will get rate
	//* information when we ask for a print out 
	t->init_timing(timing_iterations, 0);
      else	
	//* otherwise, the feature will give rate information 
	//* independently as it computes it	
	t->init_timing(timing_iterations, 1);
    }
  
  //* main loop.  just makes calls to the SSD tracker, which takes care of its own
  //* display and video interface.  printing of the tracker state and display
  //* of the region of interest in the framebuffer is controlled from the mainline
  //* as they are general features extended to other Tracking interfaces.  The rest
  //* of the interaction   

  int itercount = 0;
  Image grabim(width,height);

  while (    t->track() ) {
    //    if (show_grab) {
    
      //      init_win.show_grab(v,100.0);  // Update live image every 10th of asecond
      //      init_win.flush();
      v->grab_aligned(grabim,(int)t->x(),(int)t->y());
      init_win.clear(0);
      init_win.show(grabim,(int)t->x(),(int)t->y());
      init_win.flush();
      //    }
    if (print_state)
      cout << itercount++ << ". " << (*t) << endl;
    
    //    t->live_image().write_pgm("testim.pgm");
  }
  exit(1);

}



