/*
    Copyright (C) 1997 Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.
*/

//&&Section ColorBlobtest
/** 
  ColorBlobtest demonstrates region tracking using a derivative of BlobColor class.
  The generic Blob tracker, from which BlobColor is subclassed,  uses the center 
  of mass of a region of points that are satisfy a membership constraint to
  track a "Blob" region.  The BlobColor tracker, then, is responsible  for
  administering this membership test.  In order to do this,
  it creates a statistical model of the region of interest,
  where every pixel is a 3 dimensional vector in RGB color space.  It computes
  the mean and variance of the pixel-vectors in the user-initialized template,   
  and then inverts the co-variance matrix which can then be used to test pixel 
  membership in the region, by a standard deviance threshold (usually between 1 
  and 3) to cut out pixels that deviate too much from the original distribution.
 **/

//&&Usage: ColorBlobtest [OPTIONS]
//* -dev            use a specific device type for video input
//*                   [K2T_COLOR 
//*                    METEOR_COLOR24 
//*                    INDYCAM_COLOR]
//* -satthresh      any pixel too close to the edges of the color cube (RGB)
//*                 can disrupt the statistical model.  Thus, if the R, G or B
//*                 component of any pixel (vector) is above this threshold 
//*                 the pixel is discarded and unused in the computation.
//*                 the default value is 230.  Valid values range from 0 to 255.
//*                    [float]
//* -sdthresh       the blob membership is determined by this threshold,
//*                 which represents the amount of standard deviance which will 
//*                 be tolerated in the current region of interest before pixels
//*                 will not be considered members of the user-initialized 
//*                 template distribution.  The default alue is 2.0
//*                    [float]
//* -minpercent     tracking of the region hinges on whether the blob 
//*                 region of interest contains a certain percentage of pixels
//*                 that satisfy membership requirements this flag lets us 
//*                 vary that percentage, where (0.0 <= percentage <= 1.0)  
//*                    [float]
//* -width          width of the ROI in pixels
//*                    [int]
//* -height         height of the ROI in pixels 
//*                    [int]
//* -widthsamp      width sampling in window (in pixels)  
//*                    [integer]          
//* -heightsamp     height sampling in window (in pixels)
//*                    [integer]          
//* -noprint        supresses printing of line information
//*                    []         
//* -nodisplay      supresses target display on monitor 
//*                    []    
//* -showgrab       shows the area that the tracker is grabbing in
//*                 the context of the interactive init window
//*                    []
//* -rate           calculates frame rate every n cycles, where n > 0,
//*                 and prints the framerate. The default is 
//*                 0 (no computation or display)
//*                    [integer]  
//&&endUsage

#include "Tracker.hh"
#include "XConsole.hh"
#include "Devices.hh"
#include "BlobColor.hh"
#include <XConsole.hh>
#include <time.h>
#include "deviceparse.hh"

//&&stopdoc
char * usage_string = "\n" 
"-----------------------------USAGE------------------------------------\n"
"Blobtest:\n"   
"     -dev            [METEOR_COLOR24 K2T_COLOR INDYCAM_COLOR]\n"
"     -satthresh      [band threshold above which related pixel is determined\n"
"                      saturated and ignored in the computation so as \n" 
"                      not to disrupt the statistical model] \n" 
"     -sdthresh       [cutoff standard deviation after which pixel \n"
"                      membership in the Blob region is denied] \n"
"     -minpercent     [number of pixels that must be members in region \n"
"                      to continue tracking Blob region centroid]\n"
"     -height         [height of image window in pixels]\n" 
"     -width          [width of image window in pixels]\n" 
"     -widthsamp      [sampling rate (in pixels) along x axis]\n"
"     -heightsamp     [sampling rate (in pixels) along y axis]\n"
"     -rate           [number of cycles used to calculate frame rate]\n"   
"     -noprint         -- supresses printing of line information  \n"
"     -nodisplay       -- supresses target display on monitor \n"
"     -showgrab        -- shows the area that the tracker is grabbing \n"
"                         in the interactive init window \n" ;

//&&startdoc
//&&Subsection Annotated Main Program
main(int argc, char *argv[]) 
{
  int width = 40;
  int height = 40;
  int ws  = 1;
  int hs = 1;
  Device_output_type in_color = PIX_LUMINANCE;
  int is_color_device = 0;
  char filenamebuf[256];
  int display = 1;
  int show_grab = 0;
  int do_timing = 0;
  int timing_iterations = 0;
  int print_state = 1;
  sprintf(filenamebuf, "");
  float min_percent = 0.1;
  float saturation_threshold = 230;
  float sd_threshold = 2.0;

  Video *v = NULL;
  for (int i=1; i < argc ; i++) {
    if (strcmp(argv[i], "-dev") == 0) {
      //* a special purpose routine located in "deviceparse.cc", 
      //* 'make_device' takes advantage of the C preprocessor to compile 
      //* appropriate Video class constructors based on what the user 
      //* specifies in "site_config.h"
      v = make_device(&is_color_device, argv[++i]);      
      //* If the device type is unknown, the flag and 
      //* its argument are ignored
      if (!is_color_device)
	{
	  printf("Cannot run program without a color Video Device\n");
	  exit(1);
	};	    
      if (v == NULL) {     
	printf("Unknown device type %s encountered, and ignored \n", 
	       argv[i]);      
      }
    } 
    else if (strcmp(argv[i], "-nodisplay") == 0) {
      display = 0;
    }
    else  if (strcmp(argv[i], "-height") == 0) {
      height = atoi(argv[++i]);
    }
    else if (strcmp(argv[i], "-width") == 0) {
      width = atoi(argv[++i]);
    }
    else if (strcmp(argv[i], "-widthsamp") == 0) {
      ws = atoi(argv[++i]);
    }
    else if ((strcmp(argv[i], "-heightsamp") == 0)) {
      hs = atoi(argv[++i]);
    }
    else if (strcmp(argv[i], "-rate") == 0) {	
      do_timing = 1;
      timing_iterations = atoi(argv[++i]);
      if (timing_iterations <= 0)
	{
	  printf("Cycle count for computing frame rate must be >= 0\n");
	  exit(1);
	};
    }
    else if (strcmp(argv[i], "-showgrab") == 0) {		  
      show_grab = 1;
    }
    else if (strcmp(argv[i], "-noprint") == 0) {		  
      print_state = 0;
    }
    else if (strcmp(argv[i], "-satthresh") == 0) {
      saturation_threshold  = atof(argv[++i]);
    }
    else if (strcmp(argv[i], "-minpercent") == 0) {
      min_percent  = atof(argv[++i]);
    }
    else if (strcmp(argv[i], "-sdthresh") == 0) {
      sd_threshold  = atof(argv[++i]);
    }
    else {
      cerr << endl << endl;
      cerr << "---------" << endl ;
      cerr << "Unknown flag " << argv[i] << endl;
      cerr << usage_string << endl;
      exit(-1);
    }
  }
 
  //* If a Video device has not been specified, we create it.  
  if (v == NULL)
    {
      //* Preprocessor directives compile either this code...
#ifndef DEFAULT_COLOR_DECLARATION
      cerr << "You have tried to convert color video input to "
	   << "a different representation but have not compiled "
	   << "to use a DEFAULT Video class that supports color."
	     << "look in the site_config.h file" << endl;
      exit(-1);
#else
      //* ...or this code.
      cerr << "Video device not specified. "
	   << "Initializing with DEFAULT_COLOR_DECLARATION.  "     
	   << endl;
      v = new DEFAULT_COLOR_DECLARATION;		
      v->set_grab_type(PIX_PACKED);
#endif
    };

  //* must use a color console here
  XConsole c(1);

  //* one window will do interactive initialization and "showgrab",
  //* the other will show the blob's image buffer.
  XWindow w(c);
  XWindow w1(c);
  
  //* initializes a color blob tracker 
  BlobColor t(v,height,width,
	      sd_threshold,
	      ws,hs, 
	      min_percent, saturation_threshold);
  
  //* interactive initialization occurs here.  
  //* the window is closed only if we are not going to use it to
  //* show where in the framebuffer we are grabbing
  t.interactive_init (w);
  if (!show_grab)
    w.close();
  
  if (do_timing) 
    {
      if (print_state) 
	//* if we are going to be printing tracker information, we will get rate
	//* information when we ask for a print out 
	t->init_timing(timing_iterations, 0);
      else	
	//* otherwise, the feature will give rate information 
	//* independently as it computes it	
	t->init_timing(timing_iterations, 1);
    }
  
  //* this is the main loop which does the tracking
  //* and display.
  while (1) {
    t.track();
    if (show_grab)
      w.show_grab(v);
    if (display)
      t.show(w1);
    if (print_state)
      cout << t;
  }
} 








