/* Copyright (C) 1997 Gregory D. Hager (Yale Computer Science Robotics
    and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.

*/

//&&Section Blobtest
/** 
  Blobtest demonstrates simple region tracking using the Blob class.
  A blob tracker thresholds each pixel value in the region of interest, 
  and then uses the center of mass of those points that are above the threshold to 
  track the region.  To get good performance out of this demonstration of 
  Blob tracking, it is important to know how to initialize the blob tracker
  so that it will work the way that you want.  The blob tracker is initialized  
  with a membership function that tests pixels for membership in the blob based
  on whether or not the pixel is between the lower and upper thresholds.  
  these thresholds default to 0 and 100, respectively.  Furthermore, the 
  blob tracking only occurs if a minimum percentage of pixels in the region
  satisfies the membership requirement.  This minimum percentage defaults to .1.
  The best performance will occur if you are tracking a black dot on a white 
  background, for example.  If you wanted to track a white dot on a black 
  background, you could move the lower threshold to 150 and the upper threshold to 
  255, for example.  The minimum percentage should probably be kept fairly low.
  No matter where you initialize the blobtracker, it will try to center itself on
  a region containing 100% of the points that satisfy the membership.  For this
  reason, the best performance occurs when the blob region to be tracked 
  can be contained entirely in the region of interest. 
 **/


//&&Usage: Blobtest [OPTIONS]
//* -dev            use a specific device type for video input
//*                   [K2T_MONO K2T_COLOR IT_FG101 DT3155
//*                    METEOR_COLOR24 METEOR_COLOR16 METEOR_MONO
//*                    INDYCAM_MONO INDYCAM_COLOR]
//* -mpeg           use a mpeg in place of video input. 
//*                    [filename]          
//* -image_sequence use an image sequence in place of video input
//*                    [filename]         
//* -color_convert  project color video to a monochrome image of type
//*                    [R]ed,[G]reen,[B]lue,
//*                    [L]uminance,[S]aturation,[H]ue,[I]ntensity
//* -upperthresh    pixels are tested for membership in the blob tracker 
//*                 region of interest.  This is an upper threshold against which 
//*                 the pixel value is tested to determine membership 
//*                    [integer]
//* -lowerthresh    as above, lower threshold used to determine pixel 
//*                 membership in blob region
//*                    [integer]
//* -minpercent     tracking of the region hinges on whether the blob 
//*                 region of interest contains a certain percentage of pixels
//*                 that satisfy membership requirements this flag lets us 
//*                 vary that percentage, where (0.0 <= percentage <= 1.0)  
//*                    [float]
//* -width          width of the ROI in pixels
//*                    [int]
//* -height         height of the ROI in pixels 
//*                    [int]
//* -widthsamp      width sampling in window (in pixels)  
//*                    [integer]          
//* -heightsamp     height sampling in window (in pixels)
//*                    [integer]          
//* -noprint        supresses printing of line information
//*                    []         
//* -nodisplay      supresses target display on monitor 
//*                    []    
//* -showgrab       shows the area that the tracker is grabbing in
//*                 the context of the interactive init window
//*                    []
//* -rate           calculates frame rate every n cycles, where n > 0,
//*                 and prints the framerate. The default is 
//*                 0 (no computation or display)
//*                    [integer]  
//&&endUsage


#include "site_config.h"
#include "Tracker.hh"
#include "XConsole.hh"
#include "Devices.hh"
#include "BlobThresh.hh"
#include <XConsole.hh>
#include <XVImage.hh>
#include <time.h>
#include "deviceparse.hh"

//&&stopdoc
char * usage_string = "\n" 
"-----------------------------USAGE------------------------------------\n"
"Blobtest:\n"   
"     -dev            [K2T_MONO K2T_COLOR IT_FG101 DT3155\n"
"                      METEOR_MONO METEOR_COLOR24 METEOR_COLOR16\n"
"                      INDYCAM_MONO INDYCAM_COLOR]\n" 
"     -mpeg           [filename]         \n" 
"     -image_sequence [filename]        \n" 
"     -color_convert  [R G B L S H I]       \n" 
"     -upperthresh    [upper threshold used to determine pixel \n"
"                      membership in blob region]\n" 
"     -lowerthresh    [lower threshold used to determine pixel \n"
"                      membership in blob region]\n" 
"     -minpercent     [number of pixels that must be members in region \n"
"                      to continue tracking Blob region centroid]\n"
"     -height         [height of image window in pixels]\n" 
"     -width          [width of image window in pixels]\n" 
"     -widthsamp      [sampling rate (in pixels) along x axis]\n"
"     -heightsamp     [sampling rate (in pixels) along y axis]\n"
"     -rate           [number of cycles used to calculate frame rate]\n"   
"     -noprint         -- supresses printing of line information  \n"
"     -nodisplay       -- supresses target display on monitor \n"
"     -showgrab        -- shows the area that the tracker is grabbing \n"
"                         in the interactive init window \n" 
;

//&&startdoc


//&&Subsection Annotated Main Program

main(int argc, char *argv[]) 
{
  float min_percent = 0.1;
  int uthresh = 130;
  int lthresh = 0;
  int do_timing = 0;
  int timing_iterations = 0;
  int print_state = 1;
  int show_grab = 0;
  int ws = 1;
  int hs = 1;
  int height = 40;
  int width = 40;
  Device_output_type in_color = PIX_LUMINANCE;
  int is_color_device = 0;
  char filenamebuf[256];
  int display = 1;
  int grow_blob = 0;
  sprintf(filenamebuf, "");


  Video *v = NULL;
  for (int i=1; i < argc ; i++) {
    if (strcmp(argv[i], "-dev") == 0) {
      //* a special purpose routine located in "deviceparse.cc", 
      //* 'make_device' takes advantage of the C preprocessor to compile 
      //* appropriate Video class constructors based on what the user 
      //* specifies in "site_config.h"
      v = make_device(&is_color_device, argv[++i]);      
      //* If the device type is unknown, the flag and 
      //* its argument are ignored
      if (v == NULL) {     
	printf("Unknown device type %s encountered, and ignored \n", 
	       argv[i]);      
      }
    } 

    //* Rudimentary argument parsing
    else if (strcmp(argv[i], "-color_convert") == 0) {
      
      //* If the user specifies that they want a color conversion,
      //* we assume that they have provided a device capable of color 
      //* and set this flag

      is_color_device = 1;
      switch (argv[++i][0]) {
	
      case 'R':
	cerr << "Video device will convert color images to a PIX_RED representation" << endl; 
	in_color = PIX_RED;
	break;
	
      case 'G':
	cerr << "Video device will convert color images to a PIX_GREEN representation" << endl; 
	in_color = PIX_GREEN;
	break;

      case 'B':
	cerr << "Video device will convert color images to a PIX_BLUE representation" << endl; 
	in_color = PIX_BLUE;
	break;

      case 'I':
	cerr << "Video device will convert color images to a PIX_INTENSITY representation" << endl; 
	in_color = PIX_INTENSITY;
	break;

      case 'H':
	cerr << "Video device will convert color images to a PIX_HUE representation" << endl; 
	in_color = PIX_HUE;
	break;

      case 'S':
	cerr << "Video device will convert color images to a PIX_SATURATION representation" << endl; 
	in_color = PIX_SATURATION;
	break;
      
      default:
	cerr << "UNKNOWN FLAG: " << (argv[i]) << " ignored" << endl;
	cerr << "Video device will convert color images to a PIX_LUMINANCE representation" << endl; 	
	in_color = PIX_LUMINANCE;
	break;
      }
    }
    else if (strcmp(argv[i], "-nodisplay") == 0) {
      display = 0;
    }
    else if (strcmp(argv[i], "-growblob") == 0) {
      grow_blob = 1;
    }
    else  if (strcmp(argv[i], "-height") == 0) {
      height = atoi(argv[++i]);
    }
    else if (strcmp(argv[i], "-width") == 0) {
      width = atoi(argv[++i]);
    }
    else if (strcmp(argv[i], "-widthsamp") == 0) {
      ws = atoi(argv[++i]);
    }
    else if ((strcmp(argv[i], "-heightsamp") == 0)) {
      hs = atoi(argv[++i]);
    }
    else  if (strcmp(argv[i], "-lowerthresh") == 0) {
      lthresh = atoi(argv[++i]);
    }
    else if (strcmp(argv[i], "-upperthresh") == 0) {
      uthresh  = atoi(argv[++i]);
    }
    else if (strcmp(argv[i], "-minpercent") == 0) {
      min_percent  = atof(argv[++i]);
    }
    else if ((strcmp(argv[i], "-mpeg") == 0)) {
      cerr << "Opening MPEG video stream for file " 
	   << argv[++i] << endl;
      v = new MPEG(argv[i]);
    }
    else if ((strcmp(argv[i], "-image_sequence") == 0)) {
      cerr << "Opening !Monochrome ImageSequence starting with file "
	   << argv[++i] << "0 " <<  endl;
      ImageSequence_Mono* bla = new ImageSequence_Mono(argv[i]);
      bla->auto_step(1);
      v = (Video*)bla;
    }
    else if (strcmp(argv[i], "-rate") == 0) {	
      do_timing = 1;
      timing_iterations = atoi(argv[++i]);
      if (timing_iterations <= 0)
	{
	  printf("Cycle count for computing frame rate must be >= 0\n");
	  exit(1);
	};
    }
    else if (strcmp(argv[i], "-showgrab") == 0) {		  
      show_grab = 1;
    }
    else if (strcmp(argv[i], "-noprint") == 0) {		  
      print_state = 0;
    }
    else {
      cerr << endl << endl;
      cerr << "---------" << endl ;
      cerr << "Unknown flag " << argv[i] << endl;
      cerr << usage_string << endl;
      exit(-1);
    }
  }

  //* If a Video device has not been specified, we create it.  
  if (v == NULL)
    {
      //* Since the pointer is NULL, we want to know whether the 
      //* user specified a display conversion from color output
      //* in the video device, but did not specify the type of device 
      if (is_color_device)
	{
	  //* There is no device specified, but there is a color
	  //* conversion requested, so this preprocessor 
	  //* switch tries to create a default color video constructor
	  //* if one has been compiled 

	  //* Preprocessor directives compile either this code...
  #ifndef DEFAULT_COLOR_DECLARATION
	cerr << "You have tried to convert color video input to "
	     << "a different representation but have not compiled "
	     << "to use a DEFAULT Video class that supports color."
	     << "look in the site_config.h file" << endl;
      exit(-1);
  #else
      //* ...or this code.
      cerr << "Video device not specified. "
	   << "Initializing with DEFAULT_COLOR_DECLARATION.  "
	   << "To override specify device name or image stream with -dev, -mpeg, or -image_sequence flags" << endl;      	  
      v = new DEFAULT_COLOR_DECLARATION;		
#endif 
	}
      else
	{
	  //* we just create the default video device, since the k	
	  //* user specified neither a device type nor a color conversion
	  cerr << "Video device not specified.  "
	       << "Initializing with DEFAULT_DECLARATION.  "
	       << "To override specify device name or image "  
	       << "stream with -dev, -mpeg, or -image_sequence flags" 
	       << endl;      	  
	  v = new DEFAULT_DECLARATION;
	};
    }


  //* Try to set the output type of the color video device.  If it exists,
  //* this call will convert the color information from a video feed into
  //* the requested format.  If the the video device pointed to by 'v'
  //* doesn't support color but the flag is set, (i.e. the user
  //* specified a both a color conversion and a monochrome device) 
  //* the call will fail and generate an error message.
  if (is_color_device)
    v->set_grab_type(in_color);
  int w_color = (v->output_data_type() == PIX_PACKED);    
  XConsole c(w_color);
  XWindow w(c);
  XWindow w1(c);

  //* Initializes the BlobThreshold, which is an instance of BasicBlob.
  //* This will compute the center of mass of the pixels in the 
  //* region of interest (ROI) whose vales lie between the lower and upper
  //* thresholds, and then center the tracking ROI on that centroid 
  BlobThreshold t(v,height,width,lthresh,uthresh,ws,hs, min_percent);

  //* interactive initialization occurs here.  
  //* the window is closed only if we are not going to use it to
  //* show where in the framebuffer we are grabbing
  t.interactive_init (w);
  if (!show_grab)
    w.close();

  if (do_timing) 
    {
      if (print_state) 
	//* if we are going to be printing tracker information, we will get rate
	//* information when we ask for a print out 
	t.init_timing(timing_iterations, 0);
      else	
	//* otherwise, the feature will give rate information 
	//* independently as it computes it	
	t.init_timing(timing_iterations, 1);
    }

  //* This here is something we are still developing.
  //* The strategy as of yet does not rely on second order characteristics,
  //* and is somewhat unstable.  If you want to use this, type -growblob
  //* as a flag with the Blobtest command. This optionis not listed in usage 
  //* because it is experimental  If you want to change any of 
  //* the parameters below, you have to recompile. the blob tries to maintain
  //* membership between min_membership and max_membership by growing by 
  //* a percentage based on the amount of pixels currently
  //* satisfying membership in the region.
  float min_membership = 0.8;
  float max_membership = 0.83;
  float growth_factor = 1;
  float shrink_factor = .1;
  if (grow_blob)
    {
      t.autoresize(max_membership,
		    min_membership,
		    shrink_factor,
		    growth_factor);
    };
  
  //* this is the main loop which does the tracking
  //* and display.
  while (1) {
    t.track();
    if (print_state) 
      cout << t;
    if (display)
      t.show(w1);
    if (show_grab)
      w.show_grab(v);
  }
} 








