/*
    Copyright (C) 1998 Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.
*/

//&&Section pipe_test
/** pipe_test performs "live" video display from an input device.  
    Various parameters such as input device type, position, orientation, and 
    sampling rate can be modified as described below.  The XConsole and 
    XWindow classes are demonstrated, as well as the Video class.    
 **/

//&&Usage: pipe_test [OPTIONS]
//*     -height         height (number of rows) in window
//*                       [integer]          
//*     -width          width (number of columns) in window         
//*                       [integer]          
//*     -startx        x coordinate of corner of window
//*                       [integer]          
//*     -starty        y coordinate of corner of window
//*                       [integer]          
//&&endUsage


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fstream.h>
#include <math.h>
#include <rate.hh>
#include <site_config.h>
#include <Tracker.hh>
#include <XConsole.hh>
#include <Devices.hh>
#include <Image.hh>
#include "deviceparse.hh"
#include <Pipeline.hh>

//* This is just the number of pixel locations you
//* want to find in the end.  We should probably
//* do a connected components test but...

#define MAX_FACTOR 0.90 // locations within this percentage of max reported
#define MAX_LOCATIONS 2000
#define ERODE_SIZE 9
#define COLOR_THRESHOLD_VALUE 990      // PL_Project normalizes to 1000, so this is 0.99
#define INITIAL_COLOR_VECTOR 170,140,60  // Where the color thresholding starts

// Some pipe labels ...

#define COLOR_THRESHOLD    5
#define COLOR_PROJECTION   10
#define FINAL_THRESHOLD    20
#define COMBINED_VARIANCE  30

//&&stopdoc
char * usage_string = "\n" 
"-----------------------------USAGE------------------------------------\n"
"pipe_test:\n"   
"     -height         [height of image window in pixels]\n" 
"     -width          [width of image window in pixels]\n" 
"     -startx        [center of image window along x axis]\n"
"     -starty        [center of image window along y axis]\n"
"     -rate           [number of cycles used to calculate frame rate]\n"  
 ; 
//&&startdoc


//&&Subsection Annotated Main Program
void
main(int argc, char *argv[]) {

  //* parameters for grabbing from video input, and displaying image
  int height = -1;
  int width = -1;
  int xloc = -1;
  int yloc = -1;

  //* information used to compute rate (program performance)
  //* Usualy, rate information is computed by tracker elements, but
  //* this demonstration program does not perform tracking.
  int COMPUTE_RATE_EVERY_N_FRAMES = 0; 

  //* these variables hold information about the color capabilities 
  //* of the input device
  int is_color_device = 0;

  //* sampling rates for height, and width
  int ws = 1;
  int hs = 1;
  int do_display = 1;

  //* used in for loops
  int i,j;

  //* A black and white display

  XConsole c(0);

  // This is the structure where you set parameters for
  // the modules with modifiable parameters

  Pipe_Params pp;

  //* A pointer to a video device, the software abstraction for video input.

  Video *v = make_device(&is_color_device, "METEOR_COLOR24"); 

  //* all arguments for this program come in pairs of two
  for (i = 1; i < argc; i += 2) {
    if (strcmp(argv[i], "-height") == 0) {
      height = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-width") == 0) {
      width = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-startx") == 0) {
      xloc = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-starty") == 0) {
      yloc = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-nodisplay") == 0) {
      do_display = 0;i--;
    }
    else if (strcmp(argv[i], "-rate") == 0) {		  
      if (argc == i+1)
	{
	  printf("\n%s\n", usage_string);
	  exit(1);
	};
      COMPUTE_RATE_EVERY_N_FRAMES = atoi(argv[i + 1]);
      if (COMPUTE_RATE_EVERY_N_FRAMES <= 0)
	{
	  printf("Cycle count for computing frame rate must be > 0\n");
	  exit(1);
	};
    }
    else {
      cerr << "Unknown flag " << argv[i] << endl;
      cerr << usage_string << endl;
      exit(-1);
    }

  }   

  // We need a color device for all of this to work
  // This should be default, but it never hurts to make sure

  v->set_grab_type(PIX_PACKED);

  //* Here is the initial color pipeline attached
  //* to the input video stream 

  Pipeline pl(v);

  //* If these parameters have not been set, we set them now, to the default dimensions 
  //* of the video device.  Note that the x and y location of the widow are given with 
  //* respect to the center of the window, and NOT the uper left hand corner
  //* If they have been set, then insert a subwindowing into the pipeline

  if ((height > 0) || (width > 0)) {
    if (xloc < 0)
      xloc = 0;
    if (yloc < 0)
      yloc = 0;

    // Make sure the height and width aren't too large

    if (height < 0) 
      height = v->height()-yloc;
    else
      height = min2(v->height()-yloc,height);
    if (width < 0)
      width = v->width() - xloc;
    else
      width = min2(v->width()-xloc,width);

    // Create a subimage pipeline
    pl = PL_Subimage(pl, xloc, yloc, xloc + width-1, yloc+height-1);
  }

  // Here is the gray-scale pipeline

  Pipeline plg = PL_Gray(pl);
  if (do_display)
    plg.display_output(c,0);

  // Compute x and y variances

  Pipeline plvx = PL_Gaussian(PL_Sq(plg),11,3,0,1) -
			  PL_Sq(PL_Gaussian(plg,11,3,0,1));
  //  plvx.display_output(c,1,"X Variance");

  Pipeline plvy = PL_Gaussian(PL_Sq(plg),0,1,11,3) -
			  PL_Sq(PL_Gaussian(plg,0,1,11,3));
  //  plvy.display_output(c,1,"Y Variance");

  // This is the color part of the pipeline.  First project
  // onto an initial color guess --- this is adapted later

  Pipeline ppl = PL_Project(COLOR_PROJECTION,pl,INITIAL_COLOR_VECTOR,1);
  ppl = PL_Threshold(COLOR_THRESHOLD,ppl,COLOR_THRESHOLD_VALUE,1);
  if (do_display)
    ppl.display_output(c,1);

  // Now, do a little directional erosion on the
  // result --- the idea is to reward areas
  // which have yellow on both sides.

  Image erode_maskx(ERODE_SIZE,1);
  Image erode_masky(1,ERODE_SIZE);
  for (int itmp=0;itmp<ERODE_SIZE;itmp++)
    erode_masky[itmp][0] = 
      erode_maskx[0][itmp] = (abs(itmp-(ERODE_SIZE-1)/2.0) >= ERODE_SIZE/6.0);
  cout << (XVImage<int>)erode_maskx;

  Pipeline plex = PL_Erode(ppl,erode_maskx);
  Pipeline pley = PL_Erode(ppl,erode_masky);

  // Now, mask the variances with the output
  // of erode and then superimpose the results

  plvx = PL_Gaussian(COMBINED_VARIANCE,plvx*plex + plvy*pley,5,2);
  if (do_display)
    plvx.display_output(c,1,"Result");

  Pipeline_Video pl_v(&plvx,plvx.get_height(),plvx.get_width());

  // This is a branch used to adapt the color
  // In the main loop these projections are
  // used to pick new color coefficients.

  Pipeline adaptpl = ppl * pl;

  Pipeline rpl = PL_Project(20,adaptpl,1,0,0,0);
  Pipeline gpl = PL_Project(21,adaptpl,0,1,0,0);
  Pipeline bpl = PL_Project(22,adaptpl,0,0,1,0);

  //* Create a window to be managed by the XConsole and to display 
  //* the image in
  XWindow w(c);

  //* Create an image of the appropriate size.  Image is a stub 
  //* class derived from the XVImage<Txx> Template.  This template
  //* is not fully integrated into the package and should be used 
  //* with caution.

  // Get an image and a window for display of final output
  Image im(plvx.get_width()-3,plvx.get_height()-3);
  if (do_display)
    w.open(im.width(),im.height(),"XVision Display");

  TimingInfo timing_info(COMPUTE_RATE_EVERY_N_FRAMES, 
			1, "showlive");

  //* First, adapt the color for a couple of iterations; note
  //* that i is incremented only when a decent amount of color
  //* is found

  double diver;
  for (i=0;i<3;) {

    // Pull the color adaptation stuff through "by hand"

    rpl[20]->main_update();
    gpl[21]->main_update();
    bpl[22]->main_update();

    // Do some adaptation on the color projection segment
    // by first getting the segementation

    Image &xx = ppl[COLOR_THRESHOLD]->get_output();
    diver = xx.sum();
    cout << "Percent Yellow = " 
	 << diver/(xx.width()*xx.height()) << endl;

    // Computing the central moment of the rgb vector

    pp.rc = ((rpl[20]->get_output()).sum())/diver;
    pp.bc = ((bpl[22]->get_output()).sum())/diver;
    pp.gc = ((gpl[21]->get_output()).sum())/diver;
    pp.normalize = 1;

    // And then resetting the color.  Note that this
    // will always occur with a delay of one frame

    if (diver > 0.2) { // 20 % 
      cout << "r = " << pp.rc << " g = " << pp.gc << " b = " << pp.bc << endl;
      adaptpl[COLOR_PROJECTION]->modify_params(pp);
      i++;
    }
    else   {
      pp.threshold -= 5;
      adaptpl[COLOR_THRESHOLD]->modify_params(pp);
    }
  }

  //* Now, pull through the image

  int coords[MAX_LOCATIONS][2];
  int count = 0;

  if (!(pl_v.grab_region(im,1,1)))
    {
      cerr << "Desired image out of view." << endl;
      exit(1);
    }
  else 
    {
      Image imscale = im.scale();
      if (do_display)
	w.show(imscale,"Result");
      imscale.write_pgm("result.pgm");
      Image imout(30,30);
      char buff[210];

      // Now, find the values with a given percentage of 
      // the actual maximum;

      int maxval = im.max();
      maxval *= MAX_FACTOR;
      
      ofstream of("Coordinates");
      for (i=0;i<im.height();i++)
	for (j=0;j<im.width();j++)
	  if (im[i][j] >= maxval) {
	    cout << "(" << i << ", " << j << ")" << endl;
	    of << j << " " << i << endl;
	    coords[count][0] = i;
	    coords[count][1] = j;
	    if (v->grab(imout,j,i)) {
	      sprintf(buff,"location.%d.%d.ppm",i,j);
	      imout.write_ppm(buff);
	    }
	    count++;

	    if (count > MAX_LOCATIONS) {
	      cerr << "Not enough room to store locations" << endl;
	      exit(1);
	    }
	  }
    }

  cout << "Found " << count << " locations" << endl;
  //  w.close();
};





