/*
    Copyright (C) 1996,1997 Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.
*/

//&&Section pipe_test
/** pipe_test performs "live" video display from an input device.  
    Various parameters such as input device type, position, orientation, and 
    sampling rate can be modified as described below.  The XConsole and 
    XWindow classes are demonstrated, as well as the Video class.    
 **/

//&&Usage: pipe_test [OPTIONS]
//*     -dev            use a specific device type for video input
//*                       [K2T_MONO K2T_COLOR IT_FG101 DT3155
//*                        METEOR_MONO METEOR_COLOR24 METEOR_COLOR16
//*                        INDYCAM_MONO INDYCAM_COLOR]
//*     -mpeg           use a mpeg in place of video input. 
//*                       [filename]          
//*     -color_convert  project color video to a monochrome image of type
//*                       [R]ed,[G]reen,[B]lue,
//*                       [L]uminance,[S]aturation,[H]ue,[I]ntensity
//*     -height         height (number of rows) in window
//*                       [integer]          
//*     -width          width (number of columns) in window         
//*                       [integer]          
//*     -centerx        x coordinate of center of window
//*                       [integer]          
//*     -centery        y coordinate of center of window
//*                       [integer]          
//*     -widthsamp      width sampling in window (in pixels)  
//*                       [integer]          
//*     -heightsamp     height sampling in window (in pixels)
//*                       [integer]          
//&&endUsage


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <rate.hh>
#include <site_config.h>
#include <Tracker.hh>
#include <XConsole.hh>
#include <Devices.hh>
#include <Image.hh>
#include "deviceparse.hh"
#include <Pipeline.hh>

//* This is just the number of pixel locations you
//* want to find in the end.  We should probably
//* do a connected components test but...

#define DESIRED_STELLEN 25
#define ERODE_SIZE 9
#define COLOR_THRESHOLD 990      // PL_Project normalizes to 1000, so this is 0.99
#define INITIAL_COLOR_VECTOR 170,140,60  // Where the color thresholding starts

// Some pipe labels ...

#define COLOR_PROJECTION   10
#define FINAL_THRESHOLD    20
#define COMBINED_VARIANCE  30

//&&stopdoc
char * usage_string = "\n" 
"-----------------------------USAGE------------------------------------\n"
"pipe_test:\n"   
"     -dev            [K2T_MONO K2T_COLOR IT_FG101 DT3155\n"
"                      METEOR_MONO METEOR_COLOR24 METEOR_COLOR16\n"
"                      INDYCAM_MONO INDYCAM_COLOR]\n" 
"     -mpeg           [filename]         \n" 
"     -height         [height of image window in pixels]\n" 
"     -width          [width of image window in pixels]\n" 
"     -centerx        [center of image window along x axis]\n"
"     -centery        [center of image window along y axis]\n"
"     -widthsamp      [sampling rate (in pixels) along x axis]\n"
"     -heightsamp     [sampling rate (in pixels) along y axis]\n"
"     -rate           [number of cycles used to calculate frame rate]\n"  
 ; 
//&&startdoc


//&&Subsection Annotated Main Program
void
main(int argc, char *argv[]) {
  //* buffer to hold name used to initialize MPEG or image sequence 
  char filenamebuf[256];
  sprintf(filenamebuf, "");

  //* parameters for grabbing from video input, and displaying image
  int height = -1;
  int width = -1;
  int xloc = -1;
  int yloc = -1;

  //* information used to compute rate (program performance)
  //* Usualy, rate information is computed by tracker elements, but
  //* this demonstration program does not perform tracking.
  int COMPUTE_RATE_EVERY_N_FRAMES = 0; 

  //* these variables hold information about the color capabilities 
  //* of the input device
  int is_color_device = 0;

  //* sampling rates for height, and width
  int ws = 1;
  int hs = 1;

  //* used in for loops
  int i,j;

  //* A pointer to a video device, the software abstraction for video input.
  //* First, we check to see if the user fores a specific type of video device.
  //* If not, then we decide what kind of video device we have here and
  //* set the appropriate default color mapping.  These values
  //* are inherited from files included in the site_config.h file.
  Video *v = NULL;    

  //* all arguments for this program come in pairs of two
  for (i = 1; i < argc; i += 2) {
    if ((i+1) == argc) {
      cerr << "Warning, please check that you"
	   << " are including an argument to every flag, if you "
	   << "don't the program will probably segfault" << endl;
      //* just a warning in case user inputs an odd number of arguments 
    };


    if (strcmp(argv[i], "-dev") == 0) {
      //* a special purpose routine located in "deviceparse.cc", 
      //* 'make_device' takes advantage of the C preprocessor to compile 
      //* appropriate Video class constructors based on what the user 
      //* specifies in "site_config.h"
      v = make_device(&is_color_device, argv[i+1]); 
      //* If the device type is unknown, the flag and 
      //* its argument are ignored
      if (v == NULL) {     
	printf("Unknown device type %s encountered, and ignored \n", 
	       argv[i+1]);      
      }
    } 
    else  if (strcmp(argv[i], "-height") == 0) {
      height = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-width") == 0) {
      width = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-centerx") == 0) {
      xloc = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-centery") == 0) {
      yloc = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-widthsamp") == 0) {
      ws = atoi(argv[i+1]);
    }
    else if ((strcmp(argv[i], "-heightsamp") == 0)) {
      hs = atoi(argv[i+1]);
    }
    else if ((strcmp(argv[i], "-mpeg") == 0)) {
      cerr << "Opening MPEG video stream for file " 
	   << argv[i+1] << endl;
      v = new MPEG(argv[i+1]);
    }
    else if (strcmp(argv[i], "-rate") == 0) {		  
      if (argc == i+1)
	{
	  printf("\n%s\n", usage_string);
	  exit(1);
	};
      COMPUTE_RATE_EVERY_N_FRAMES = atoi(argv[i + 1]);
      if (COMPUTE_RATE_EVERY_N_FRAMES <= 0)
	{
	  printf("Cycle count for computing frame rate must be > 0\n");
	  exit(1);
	};
    }
    else {
      cerr << "Unknown flag " << argv[i] << endl;
      cerr << usage_string << endl;
      exit(-1);
    }

  }   

  //* If a Video device has not been specified, we create it.  
  if (v == NULL)
    {
      //* Since the pointer is NULL, we want to know whether the 
      //* user specified a display conversion from color output
      //* in the video device, but did not specify the type of device 
      if (is_color_device)
	{
	  //* There is no device specified, but there is a color
	  //* conversion requested, so this preprocessor 
	  //* switch tries to create a default color video constructor
	  //* if one has been compiled 

	  //* Preprocessor directives compile either this code...
  #ifndef DEFAULT_COLOR_DECLARATION
	cerr << "You have tried to convert color video input to "
	     << "a different representation but have not compiled "
	     << "to use a DEFAULT Video class that supports color."
	     << "look in the site_config.h file" << endl;
      exit(-1);
  #else
      //* ...or this code.
      cerr << "Video device not specified. "
	   << "Initializing with DEFAULT_COLOR_DECLARATION.  "
	   << "To override specify device name or image stream with -dev, -mpeg, or -image_sequence flags" << endl;      	  
      v = new DEFAULT_COLOR_DECLARATION;		
#endif 
	}
      else
	{
	  //* we just create the default video device, since the k	
	  //* user specified neither a device type nor a color conversion
	  cerr << "Video device not specified.  "
	       << "Initializing with DEFAULT_DECLARATION.  "
	       << "To override specify device name or image "  
	       << "stream with -dev, -mpeg, or -image_sequence flags" 
	       << endl;      	  
	  v = new DEFAULT_DECLARATION;
	};
    }
     
  // We need a color device for all of this to work

  v->set_grab_type(PIX_PACKED);

  // This is the structure where you set parameters for
  // the modules with modifiable parameters

  Pipe_Params pp;

  // A black and white display

  XConsole c(0);

  //* Here is the initial color pipeline attached
  //* to the input video stream 

  Pipeline pl(v);

  // Here is the gray-scale pipeline

  Pipeline plg = PL_Gray(pl);
  plg.display_output(c,0);

  // Compute x and y variances

  Pipeline plvx = PL_Gaussian(PL_Sq(plg),11,3,0,1) -
			  PL_Sq(PL_Gaussian(plg,11,3,0,1));
  plvx.display_output(c,1,"X Variance");

  Pipeline plvy = PL_Gaussian(PL_Sq(plg),0,1,11,3) -
			  PL_Sq(PL_Gaussian(plg,0,1,11,3));
  plvy.display_output(c,1,"Y Variance");

  // This is the color part of the pipeline.  First project
  // onto an initial color guess --- this is adapted later

  Pipeline ppl = PL_Project(COLOR_PROJECTION,pl,INITIAL_COLOR_VECTOR,1);
  ppl = PL_Threshold(1,ppl,COLOR_THRESHOLD,1);
  ppl.display_output(c,1);

  // Now, do a little directional erosion on the
  // result --- the idea is to reward areas
  // which have yellow on both sides.

  Image erode_maskx(ERODE_SIZE,1);
  Image erode_masky(1,ERODE_SIZE);
  for (int itmp=0;itmp<ERODE_SIZE;itmp++)
    erode_masky[itmp][0] = 
      erode_maskx[0][itmp] = (abs(itmp-(ERODE_SIZE-1)/2.0) >= ERODE_SIZE/6.0);
  cout << (XVImage<int>)erode_maskx;

  Pipeline plex = PL_Erode(ppl,erode_maskx);
  Pipeline pley = PL_Erode(ppl,erode_masky);

  // Now, mask the variances with the output
  // of erode and then superimpose the results

  plvx = PL_Gaussian(COMBINED_VARIANCE,plvx*plex + plvy*pley,5,2);
  plvx.display_output(c,1,"Result");

  // Threshold the result

  Pipeline plvxt = PL_Threshold(FINAL_THRESHOLD,plvx,0,1);

  // Erode it a bit to get rid of randomness

  Image erode_mask(3,3);
  erode_mask = 1;
  plvxt = PL_Erode(plvxt,erode_mask);

  // Make the final video output

  Pipeline_Video pl_v(&plvxt,plvxt.get_height(),plvxt.get_width());

  // This is a branch used to adapt the color
  // In the main loop these projections are
  // used to pick new color coefficients.

  Pipeline adaptpl = ppl * pl;

  Pipeline rpl = PL_Project(20,adaptpl,1,0,0,0);
  Pipeline gpl = PL_Project(21,adaptpl,0,1,0,0);
  Pipeline bpl = PL_Project(22,adaptpl,0,0,1,0);

  //* If these parameters have not been set, we set them now, to the default dimensions 
  //* of the video device.  Note that the x and y location of the widow are given with 
  //* respect to the center of the window, and NOT the uper left hand corner
  if (height < 0) 
    height = pl_v.height();
  if (width < 0)
    width = pl_v.width();
  if (xloc < 0)
    xloc = pl_v.width()/2.0;
  if (yloc < 0)
    yloc = pl_v.height()/2.0;

  //* Create a window to be managed by the XConsole and to display 
  //* the image in
  XWindow w(c);

  //* Create an image of the appropriate size.  Image is a stub 
  //* class derived from the XVImage<Txx> Template.  This template
  //* is not fully integrated into the package and should be used 
  //* with caution.

  // Get an image and a window for display of final output
  Image im(plvx.get_width(),plvx.get_height());
  w.open(im.width(),im.height(),"XVision Display");

  TimingInfo timing_info(COMPUTE_RATE_EVERY_N_FRAMES, 
			1, "showlive");

  //* The main loop --- grab and display until the cows come home.
  i = 1;
  char lname[200];

  double factor = 0.98;
  int done = 0;
  while (!done) 
    {
      //* This is one way of grabbing data into an image from a video device.
      //* You may notice a different call is used in the grab_single demo.
      //* Here, im.data() gives you a pointer directly to the image data buffer
      //* but width and height of the image must be passed as arguments.
      if (!(pl_v.grab(im,xloc,yloc,0.0,ws,hs))) 
	{
	  cerr << "Desired image out of view." << endl;
	    exit(1);
	}
      else 
	{
	  //* inside the body of this if statement we compute rate information
	  //* normally, rate information is computed as part of the 
	  //* state information of a (top level) tracked feature.
	  //* as we are not tracking anything here, the loop has been
	  //* moved into the mainline
	  timing_info.tick();

	  sprintf(lname,"%d", i++);
	  w.show(im.scale(),lname);
	  im.write_pgm("testoutput.pgm");

	  // See how many locations passed the threshold 

	  int found_locations = im.sum();
	  cout << "Found " <<  found_locations << " locations with factor " << factor << endl;
	  if (found_locations < DESIRED_STELLEN - 3)
	    factor -= .01;
	  else if (found_locations > DESIRED_STELLEN + 3)
	    factor += .01;
	  else
	    done = 1;

	  // Pull the color adaptation stuff through

	  rpl[20]->main_update();
	  gpl[21]->main_update();
	  bpl[22]->main_update();

	  // Do some adaptation on the color projection segment
	  
	  double diver;
	  cout << "Percent Yellow = " 
	       << (diver = (ppl[1]->get_output()).sum())/(im.width()*im.height()) << " " ;
	  
	  // Compute central moment of the rgb vector

	  pp.rc = ((rpl[20]->get_output()).sum())/diver;
	  pp.bc = ((bpl[22]->get_output()).sum())/diver;
	  pp.gc = ((gpl[21]->get_output()).sum())/diver;
	  pp.normalize = 1;

	  // Here is color and maximum adaptation.  Note these
	  // will always occur with a delay of one frame

	  if (diver > 0.2) { // 20 % 
	    cout << "r = " << pp.rc << " g = " << pp.gc << " b = " << pp.bc << endl;
	    adaptpl[COLOR_PROJECTION]->modify_params(pp);
	  }

	  pp.threshold = ((plvx[COMBINED_VARIANCE]->get_output()).max())*factor;
	  plvxt[FINAL_THRESHOLD]->modify_params(pp);
	}
    }

  // List the location which are above threshold

  for (i=0;i<im.height();i++)
    for (j=0;j<im.width();j++)
      if (im[i][j])
	cout << "(" << i << ", " << j << ")" << endl;

  
  //* technically, this will never happen, but we put it here anyway, 
  //* to show you how to get rid of windows if you want to be "clean" about it.
  w.close();
};





