/*
    Copyright (C) 1996,1997 Gregory D. Hager (Yale
    Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.
*/

//&&Section line_test
/** 
  line_test demonstrates simple line tracking.  Line definitions 
  include an Edge class that describes how the Tracker
  identifies intensity transitions in the image.  
 **/


//&&Usage: linetest [OPTIONS]
//* -dev            use a specific device type for video input
//*                   [K2T_MONO K2T_COLOR IT_FG101 DT3155
//*                    METEOR_COLOR24 METEOR_COLOR16 METEOR_MONO
//*                    INDYCAM_MONO INDYCAM_COLOR]
//* -mpeg           use a mpeg in place of video input. 
//*                    [filename]          
//* -image_sequence use an image sequence in place of video input
//*                    [filename]         
//* -length         the length of the line to be tracked  
//*                    [integer]         
//* -searchwidth    the total width of the search window extended
//*                 perpendicular to the target line 
//*                    [integer]         
//* -lengthsamp     the sampling rate along the length of line
//*                    [integer]         
//* -widthsamp      the sampling rate along the width of the search window
//*                    [integer]         
//* -noprint        supresses printing of line information
//*                    []         
//* -nodisplay      supresses target display on monitor 
//*                    []         
//* -showgrab       shows the area that the tracker is grabbing in
//*                 the context of the interactive init window
//*                    []
//* -rate           calculates frame rate every n cycles, where n > 0,
//*                 and prints the framerate. The default is 
//*                 0 (no computation or display)
//*                    [integer]  
//&&endUsage


#include "site_config.h"
#include "Tracker.hh"
#include "XConsole.hh"
#include "Pattern.hh"
#include "PLine.hh"
#include "Devices.hh"
#include "MPEG.hh"
#include "Acquire.hh"
#include <sys/time.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "deviceparse.hh"


//&&stopdoc
char * usage_string = "\n" 
"-----------------------------USAGE------------------------------------\n"
"linetest:\n"   
"     -dev            [K2T_MONO K2T_COLOR IT_FG101 \n"
"                      DT3155 METEOR_COLOR24 METEOR_COLOR16 METEOR_MONO \n"
"                      INDYCAM_MONO INDYCAM_COLOR]\n" 
"     -mpeg           [filename]         \n" 
"     -image_sequence [filename]        \n"   
"     -length         [length (in pixels) of line to track]\n" 
"     -searchwidth    [width (in pixels) of search window around line]\n" 
"     -lengthsamp     [sampling rate (in pixels) along length of line]\n"
"     -widthsamp      [sampling rate (in pixels) perpendicular to line]\n"
"     -noprint         -- supresses printing of line information] \n"
"     -nodisplay       -- supresses target display on monitor \n" 
"     -showgrab        -- shows the area that the tracker is grabbing in\n"
"                         the context of the interactive init window\n" 
"     -rate           [number of cycles used to calculate frame rate]\n" 
; 
//&&startdoc


//&&Subsection Annotated Main Program
main(int argc, char *argv[]) {

  int bounding_box_width;
  int bounding_box_height;
  int bounding_box_x1;
  int bounding_box_y1;
  int bounding_box_x2;
  int bounding_box_y2;
 
  int i;
  int length = 30;
  int width = 20;
  int lsamp = 1;
  int wsamp = 1;
  int printinfo = 1;
  int displayline = 1;
  int show_grab = 0;
  int COMPUTE_RATE_EVERY_N_FRAMES = 0;
  int is_color_device = 0;
  Video *v = NULL;

  for (i=1; i < argc ; i++) {    
    if (strcmp(argv[i], "-length") == 0) {
      length = atoi(argv[++i]);
    } 
    else if (strcmp(argv[i], "-searchwidth") == 0) {
      width = atoi(argv[++i]);
    } 
    else if (strcmp(argv[i], "-widthsamp") == 0) {
      wsamp = atoi(argv[++i]);
    } 
    else if (strcmp(argv[i], "-lengthsamp") == 0) {
      lsamp = atoi(argv[++i]);
    } 
    else if (strcmp(argv[i], "-noprint") == 0) {
      printinfo = 0;
    } 
    else if (strcmp(argv[i], "-showgrab") == 0) {
      show_grab = 1;
    } 
    else if (strcmp(argv[i], "-nodisplay") == 0) {
      displayline = 0;
    }
    else if (strcmp(argv[i], "-dev") == 0) {
      if (argc == i+1)
	{
	  printf("\n%s\n", usage_string);
	  exit(1);
	};
      //* a special purpose routine located in "deviceparse.cc", 
      //* 'make_device' takes advantage of the C preprocessor to compile 
      //* appropriate Video class constructors based on what the user 
      //* specifies in "site_config.h"
      v = make_device(&is_color_device, argv[++i]);      
      //* If the device type is unknown, the flag and 
      //* its argument are ignored
      if (v == NULL) {     
	printf("Unknown device type %s encountered, and ignored \n", 
	       argv[i]);      
      }
      if (is_color_device)
	{
	  cerr << "Warning: using color device - grabbing at orientation not"
	       << " supported on color devices " << endl;
	}
    }
    else if ((strcmp(argv[i], "-mpeg") == 0)) {
      cerr << "Opening MPEG video stream for file " 
	   << argv[i+1] << endl;
      v = new MPEG(argv[i+1]);
    }
    else if ((strcmp(argv[i], "-image_sequence") == 0)) {
      cerr << "Opening !Monochrome ImageSequence starting with file "
	   << argv[i+1] << "0 " <<  endl;
      v = new ImageSequence_Mono(argv[i+1]);
    }
    else if (strcmp(argv[i], "-rate") == 0) {		  
      if (argc == i+1)
	{
	  printf("\n%s\n", usage_string);
	  exit(1);
	};
      COMPUTE_RATE_EVERY_N_FRAMES = atoi(argv[++i]);
      if (COMPUTE_RATE_EVERY_N_FRAMES <= 0)
	{
	  printf("Cycle count for computing frame rate must be > 0\n");
	  exit(1);
	};
    } else
      {		    
	printf("\n%s\n", usage_string);
	exit(1);
      };
  }
      
  if (v == NULL)
    {
      cerr << "Video device not specified.  "
	   << "Initializing with DEFAULT_DECLARATION.  "
	   << "To override specify device name or image "  
	   << "stream with -dev, -mpeg, or -image_sequence flags" 
	   << endl;      	  
      v = new DEFAULT_DECLARATION;
    }
  else
    {
      cout << "checking if video device is color" << endl;
      if (is_color_device)
	{
	  v->set_grab_type(PIX_LUMINANCE);
	  cout << "checking if video device is color" << endl;
	  cout << "will convert to luminance values in grab function" << endl;
	}
      else
	{
	  cout << "checking if video device is not color" << endl;
	};
    };

  XConsole c(0);
  XWindow w(c);
  XWindow w2(c);
  Edge e;
    
  printf("width=%d, height=%d\n", v->width(), v->height());
  puts("constructing line...");
  PLines l1(&e,v,length,width);
  //* pointer to the image that the line tracker uses.
  //* this image will be grabbed at an angle to have upright orientation

  puts("initializing line...");
  l1.interactive_init (w);
  
  if (!show_grab)
    w.close();

  //* If we want to time the performance of tracking, 
  //* this initializes the timing properties of our 
  //* top-level feature (in our case, a line).
  if (COMPUTE_RATE_EVERY_N_FRAMES)
    {
      //* if we are going to be printing Line information, we will get rate
      //* information when we ask for a print out 
      if (printinfo) 
	l1.init_timing(COMPUTE_RATE_EVERY_N_FRAMES, 0);
      //* otherwise, the feature will give rate information 
      //* independently as it computes it
      else	
	l1.init_timing(COMPUTE_RATE_EVERY_N_FRAMES, 1);
    }

  while (1) {
    //* this call uses the framework classes to perform one iteration of a 
    //* generic tracking loop with our Line tracker
    l1.track();

    //* The video device keeps track of the last window that it has grabbed in
    //* to memory.  This allows for a trick whereby you can display just that memory
    //* within the greater context of the Video window.  This makes for a neat
    //* graphics, but it slows down tracking performance considerably
    if (show_grab)
      w.show_grab(v);

    if (printinfo) cout << l1 << endl;
  };
}

