/*  Copyright (C) 1995, 1996, 1997 Kentaro Toyama and Gregory D. Hager
    (Yale Computer Science Robotics and Vision Laboratory)

    Permission is granted to any individual or institution to use, copy, 
    modify, and distribute this software, provided that this complete 
    copyright and permission notice is maintained, intact, in all copies 
    and supporting documentation.  Authors of papers that describe software 
    systems using this software package are asked to acknowledge such use
    by a brief statement in the paper.

    Gregory D. Hager provides this software "as is" without express or
    implied warranty.
*/

//&&Section image_acquire
/** 
  image_acquire can be used to acquire a single image from a video
  device to a file in pgm format .  By default, it acquires the
  complete image to the file "image.pgm."  To convert this image, 
  one can use an image editing tool such as XV. 
  Command line arguments can be used acquire a subsampled image, 
  or a window of the image.
  **/

//&&Usage:        image_acquire [OPTIONS] 
//*     -dev            use a specific device type for video input
//*                       [K2T_MONO K2T_COLOR IT_FG101 DT3155
//*                        METEOR_MONO METEOR_COLOR24 METEOR_COLOR16
//*                        INDYCAM_MONO INDYCAM_COLOR]
//*     -mpeg           use a mpeg in place of video input. 
//*                       [filename]          
//*     -color_convert  project color video to a monochrome image of type
//*                       [R]ed,[G]reen,[B]lue,
//*                       [L]uminance,[S]aturation,[H]ue,[I]ntensity
//*     -interactive    get the window location interactively
//*                       using the mouse
//*     -number         number of images to acquire
//*     -wait           wait for a CR to acquire image
//*     -height         height (number of rows) in window
//*                       [integer]          
//*     -width          width (number of columns) in window         
//*                       [integer]          
//*     -centerx        x coordinate of center of window
//*                       [integer]          
//*     -centery        y coordinate of center of window
//*                       [integer]          
//*     -widthsamp      width sampling in window (in pixels)  
//*                       [integer]          
//*     -heightsamp     height sampling in window (in pixels)
//*                       [integer]          
//*     -angle          orientation of window (in degrees)
//*                       [float]          
//*     -output         name of output file (when multiple image, 
//*                      the number of the image is concatenated)
//*                       [filename]          
//&&endUsage


#include "site_config.h"
#include "MPEG.hh"
#include "Devices.hh"
#include "Image.hh"
#include "deviceparse.hh"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <stream.h>
//#include <sgtty.h>
#include <fcntl.h>
#include <signal.h>
#include <termios.h>


//&&stopdoc
char * usage_string = "\n" 
"-----------------------------USAGE------------------------------------\n"
"image_acquire:\n"   
"     -dev            [K2T_MONO K2T_COLOR IT_FG101 DT3155\n"
"                      METEOR_MONO METEOR_COLOR24 METEOR_COLOR16\n"
"                      INDYCAM_MONO INDYCAM_COLOR]\n" 
"     -color_convert  [R G B L S H I]       \n" 
"     -mpeg           [filename]         \n" 
"     -interactive     \n" 
"     -number         [number of images to acquire] \n" 
"     -waitr          [wait for CR to acquire] \n" 
"     -height         [height of image window in pixels]\n" 
"     -width          [width of image window in pixels]\n" 
"     -centerx        [center of image window along x axis]\n"
"     -centery        [center of image window along y axis]\n"
"     -widthsamp      [sampling rate (in pixels) along x axis]\n"
"     -heightsamp     [sampling rate (in pixels) along y axis]\n"
"     -angle          [orientation (in degrees) at which to grab image]\n"
"     -output         [filename to write image to]\n"  
; 

// Some stuff to help the interactive part

void set_up_term(int fd)
{
  struct termios tio;
  ioctl(fd, TCGETS, &tio);
  tio.c_lflag = tio.c_lflag & ~(ICANON|ECHO);  /* |ECHO */
  ioctl(fd, TCSETS, &tio);

  // Make non-blocking

  fcntl(fd, F_SETFL, O_NONBLOCK);   


}

void reset_term(int fd)
{
  struct termios tio;
  ioctl(fd, TCGETS, &tio);
  tio.c_lflag = tio.c_lflag | ICANON| ECHO;  /* |ECHO */
  ioctl(fd, TCSETS, &tio);
  fcntl(fd, F_SETFL, 0);

}

void waitforkey()
{
  set_up_term(fileno(stdin));
  while (getchar() == EOF);
  reset_term(fileno(stdin));
}

void pexit(void *xx)
{
  reset_term(fileno(stdin));
  exit(1);
}

//&&startdoc
//&&Subsection Annotated Main Program

main(int argc, char *argv[]) 
{
  //* buffer to hold name used to initialize MPEG or image sequence 
  char filenamebuf[256];
  sprintf(filenamebuf, "");

  //* The following variables get set during argument processing. 
  
  //* This is a null pointer to the software abstraction for video input, 
  //* which can be any class instantiated from a subclass of the Video class.
  Video *v = NULL;

  //* Number of images and so forth
  int number = 1;
  int wait   = 0;
  int interactive = 0;

  //* dimensions for the grabbing of input from the Video device, and for  
  //* recording this information to a .pgm file
  int height = -1;
  int width = -1;
  float xloc = -1;
  float yloc = -1;

  //* sampling ratio for width and height
  int ws = 1;
  int hs = 1;

  //* angle to grab at
  float angle = 0;
  
  //* output file name, defaults to "image.pgm"
  char *fn = "image";

  //* these variables manage color information about the device and output 
  //* data representation
  int color_device = 0;
  Device_output_type in_color = PIX_PACKED;
  for (int i=1; i < argc ; i+=2) {
    if ((i+1) == argc) {
      cerr << "Warning, please check that you"
	   << " are including an argument to every flag, if you "
	   << "don't the program will probably segfault" << endl;
    };
    if ((strcmp(argv[i], "-mpeg") == 0)) {
      cerr << "Opening MPEG video stream for file " 
	   << argv[i+1] << endl;
      v = new MPEG(argv[i+1]);
      color_device = (v->output_data_type() == PIX_PACKED);    
    }
    else  if (strcmp(argv[i], "-dev") == 0) {
      //* a special purpose routine located in "deviceparse.cc", 
      //* 'make_device' takes advantage of the C preprocessor to compile 
      //* appropriate Video class constructors based on what the user 
      //* specifies in "site_config.h"
      v = make_device(&color_device, argv[i+1]);      
      //* If the device type is unknown, the flag and 
      //* its argument are ignored
      if (v == NULL) {     
	printf("Unknown device type %s encountered, and ignored \n", 
	       argv[i+1]);      
      }
    }     
    else if (strcmp(argv[i], "-color_convert") == 0) {

      //* If the user specifies that they want a color conversion,
      //* we assume that they have provided a device capable of color 
      //* and set this flag

      color_device = 1;
      switch (argv[i+1][0]) {
      case 'L':
	cerr << "Video device will convert color images to a PIX_LUMINANCE representation" << endl; 
	in_color = PIX_LUMINANCE;
	break;
	
      case 'R':
	cerr << "Video device will convert color images to a PIX_RED representation" << endl; 
	in_color = PIX_RED;
	break;
	
      case 'G':
	cerr << "Video device will convert color images to a PIX_GREEN representation" << endl; 
	in_color = PIX_GREEN;
	break;

      case 'B':
	cerr << "Video device will convert color images to a PIX_BLUE representation" << endl; 
	in_color = PIX_BLUE;
	break;

      case 'I':
	cerr << "Video device will convert color images to a PIX_INTENSITY representation" << endl; 
	in_color = PIX_INTENSITY;
	break;

      case 'H':
	cerr << "Video device will convert color images to a PIX_HUE representation" << endl; 
	in_color = PIX_HUE;
	break;

      case 'S':
	cerr << "Video device will convert color images to a PIX_SATURATION representation" << endl; 
	in_color = PIX_SATURATION;
	break;

      default:
	printf("Unknown color mode %c \n", argv[i+1][0]);
	break;
      }
    }
    else  if (strcmp(argv[i], "-height") == 0) {
      height = atoi(argv[i+1]);
    }
    else  if (strcmp(argv[i], "-interactive") == 0) {
      interactive = 1;i--;
    }
    else  if (strcmp(argv[i], "-wait") == 0) {
      wait = 1;i--;
    }
    else  if (strcmp(argv[i], "-number") == 0) {
      number = atoi(argv[i+1]);
    }
    else  if (strcmp(argv[i], "-height") == 0) {
      height = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-width") == 0) {
      width = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-centerx") == 0) {
      xloc = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-centery") == 0) {
      yloc = atoi(argv[i+1]);
    }
    else if (strcmp(argv[i], "-widthsamp") == 0) {
      ws = atoi(argv[i+1]);
    }
    else if ((strcmp(argv[i], "-heightsamp") == 0)) {
      hs = atoi(argv[i+1]);
    }
    else if ((strcmp(argv[i], "-angle") == 0)) {
      angle = atof(argv[i+1])*M_PI/180.0;
    }
    else if ((strcmp(argv[i], "-output") == 0)) {
      fn = argv[i+1];
    }
    else {
      cerr << "Unknown flag " << argv[i] << endl;
      cerr << usage_string << endl;
      exit(-1);
    }
  }
  //* If a Video device has not been specified, we create it.  
  if (v == NULL)
    {
      //* Since the pointer is NULL, we want to know whether the 
      //* user specified a display conversion from color output
      //* in the video device, but did not specify the type of device 
      if (color_device)
	{
	  //* There is no device specified, but there is a color
	  //* conversion requested, so this preprocessor 
	  //* switch tries to create a default color video constructor
	  //* if one has been compiled 

	  //* Preprocessor directives compile either this code...
#ifndef DEFAULT_COLOR_DECLARATION
	  cerr << "You have tried to convert color video input to "
	       << "a different representation but have not compiled "
	       << "to use a DEFAULT Video class that supports color."
	     << "look in the site_config.h file" << endl;
	  exit(-1);
#else
	  //* ...or this code.
	  cerr << "Video device not specified. "
	       << "Initializing with DEFAULT_COLOR_DECLARATION.  "
	       << "To override specify device name or image stream with -dev, -mpeg, or -image_sequence flags" << endl;      	  
	  v = new DEFAULT_COLOR_DECLARATION;		
#endif 
	}
      else
	{
	  //* we just create the default video device, since the k	
	  //* user specified neither a device type nor a color conversion
	  cerr << "Video device not specified.  "
	       << "Initializing with DEFAULT_DECLARATION.  "
	       << "To override specify device name or image "  
	       << "stream with -dev, -mpeg, or -image_sequence flags" 
	       << endl;      	  
	  v = new DEFAULT_DECLARATION;
	};
    }
     
  //* Try to set the output type of the color video device.  If it exists,
  //* this call will convert the color information from a video feed into
  //* the requested format.  If the the video device pointed to by 'v'
  //* doesn't support color but the flag is set, (i.e. the user
  //* specified a both a color conversion and a monochrome device) 
  //* the call will fail and generate an error message.
  if (color_device)
    {
      v->set_grab_type(in_color);
    }

  //
  int output_in_color = (v->output_data_type() == PIX_PACKED);    

  //* If these parameters didn't get set, set them now to the video device defaults.
  //* note that the x and y location of the window refers to the location of the 
  //* center of the window rather than the upper left hand corner
  if (height < 0) 
    height = v->height();
  if (width < 0)
    width = v->width();
  if (xloc < 0)
    xloc = v->width()/2.0;
  if (yloc < 0)
    yloc = v->height()/2.0;

  //* Create an image to hold the acquired region.
  Image im(width,height);

  char buff[80];
  XConsole c(output_in_color);
  XWindow w(c);
  for (int i=0;i<number;i++) {
    if (wait) {
      //* Wait until the signal ...
      printf ("Press any key when ready...\n");
      set_up_term(fileno(stdin));
      while (getchar() == EOF) {
	v->grab(im,xloc,yloc,angle,ws,hs);
	w.show(im);
	w.flush();  /// Just to force XWindow updates
      }
      reset_term(fileno(stdin));
    }
    //* ....grab the image...
    v->grab(im,xloc,yloc,angle,ws,hs);

    //* ....and write it out.
    if (output_in_color) {
      sprintf(buff,"%s%d.ppm",fn,i);
      im.write_ppm(buff);
    }
    else {
      sprintf(buff,"%s%d.pgm",fn,i);
      im.write_pgm(buff);
    }
    cout << "Wrote " << buff << endl;
  }
}



