#include <assert.h>
#define MAX_INPUTS 10

// Basic Handle is a pointer to an untyped chunk of storage.
// We use it to build the dataflow graph without worrying about
// types.

class BasicHandle {
  
  void *value;       // Value I store

public:

  BasicHandle (void *vin) {value = vin;}
  void *getval() const {return value;}
  const BasicHandle *getme() const {return this;}

};

// PipeModule is the basic unit of the graph;
// BasicHandle is used to manage it.  In order to make it
// easy to traverse the graph, the predecessors (of type BasicHandle)
// are stored here

template <class Tout>
class PipeModule {

  // For storing the graph

  int input_count;
  const BasicHandle *previous[MAX_INPUTS];

  // The value last computed

  Tout value;

  // This'll be filled in later when we lift things
  // For now, its a placeholder for constants

  virtual Tout compute_local_fn(int) {return value;};

protected:

  // For bookeeping what data has passed through

  int frame_count;

  void add_prec(const BasicHandle *x) {
    previous[input_count] = x;
    input_count++;
  };

  PipeModule<Tout> & operator[](int i) {
    assert(i < input_count);
    return *((PipeModule<Tout> *)(previous[i]->getval()));
    }

public:

  PipeModule() {frame_count = -1; input_count = 0;};
  PipeModule(Tout x) {value = x;};

  // This is the generic updating algorithm ...

  virtual Tout get_value(int fcin) {

    if (fcin > frame_count) {
      value = compute_local_fn(fcin);
      frame_count = fcin;
    }
    return value;
  }

};

// Pipes are really Handles, but they carry types and so
// act as a cosmetic sugar on Handles; they also allow automatically
// lifting constants

template <class Tout>
class Pipe : public BasicHandle {

public:

  Pipe() : BasicHandle((void *)0) {};
  Pipe(PipeModule<Tout> *v) : BasicHandle(v)  {};
  Pipe(const Tout v) : BasicHandle(new PipeModule<Tout>(v)) {};

  // Ideally, I'd like an automatic cast to the stored PipeModule ..
  // the two overloads I have are too weird ...

  PipeModule<Tout> *getval() const 
    {return (PipeModule<Tout> *)BasicHandle::getval();}
  PipeModule<Tout> *operator ->() const {return getval();}

};

template <class T>
class PipeDelay: public PipeModule<T> {
  
  int beencalled;
  T val;

public:

  PipeDelay() {beencalled = 0;};

  Pipe<T> &operator () (Pipe<T> &x,T firstval = 0)  
    {
      PipeDelay<T> *temp = new PipeDelay<T>;
      Pipe <T> *tempp = new Pipe<T>(temp);
      temp->val = firstval;
      temp->add_prec(x.getme());
      return *tempp;
    }

  T compute_local_fn(int fc) 
    {
      if (!beencalled) {
	beencalled = 1;
	return val;
      }
      else
	return ((*this)[0]).get_value(fc-1);
    }
};

// The remainder of these are "lifting" operations
// on functions

template <class Tout>
class PipeModuleLift0: public PipeModule<Tout> {
  
  Tout (*fn)();


public:

  PipeModuleLift0(Tout (*fnin)()): PipeModule<Tout>(), fn(fnin) {};

  Pipe<Tout> &operator () ()
    {
      return *(new Pipe<Tout>(new PipeModuleLift0(fn)));
    }

  Tout compute_local_fn(int fc) {return (*fn)();};
};

template <class Tin,class Tout>
class PipeModuleLift1: public PipeModule<Tout> {
  
  Tout (*fn)(const Tin&);   

public:

  PipeModuleLift1(Tout (*fnin)(const Tin&)): fn(fnin) {};

  Pipe<Tout> &operator () (const Pipe<Tin> &x)
    {
      PipeModuleLift1 *temp = new PipeModuleLift1<Tin, Tout>(fn);
      Pipe<Tout> *tempp = new Pipe<Tout>(temp);
      temp->add_prec(x.getme());
      return *tempp;
    }

  Tout compute_local_fn(int fc) 
    {
	return (*fn)((*this)[0].get_value(fc));
    }

};

template <class Tin1, class Tin2, class Tout>
class PipeModuleLift2: public PipeModule<Tout> {
  
  Tout (*fn)(const Tin1&, const Tin2&);   

public:

  PipeModuleLift2(Tout (*fnin)(const Tin1&, const Tin2&)): fn(fnin) {};

  Pipe<Tout> &operator () (const Pipe<Tin1> &x, const Pipe<Tin2> &y)  
    {
      PipeModuleLift2 *temp = new PipeModuleLift2<Tin1, Tin2, Tout>(fn);
      Pipe<Tout> *tempp = new Pipe<Tout>(temp);
      temp->add_prec(x.getme());
      temp->add_prec(y.getme());
      return *tempp;
    }

  Tout compute_local_fn(int fc) 
    {
	return (*fn)((*this)[0].get_value(fc),
		     (*this)[1].get_value(fc));
    }
};
