/*
Copyright (c) 2018, Michael Kazhdan, Alex Baden, and Keenan Crane
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution.

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <float.h>
#include "SphericalGrid.h"

template< class Real > RotationGrid< Real >::RotationGrid( void ) : res(0) , values(NULL) { ; }
template< class Real > RotationGrid< Real >::RotationGrid( int resolution ) : res(0) , values(NULL) { resize( resolution ); }
template< class Real > RotationGrid< Real >::~RotationGrid( void ) { if( values ) resize(0); }
template<class Real>
int RotationGrid<Real>::read(const char* fileName){
	FILE* fp=fopen(fileName,"rb");
	if(!fp){return 0;}
	int r=read(fp);
	fclose(fp);
	return r;
}
template<class Real>
int RotationGrid<Real>::write(const char* fileName) const{
	FILE* fp=fopen(fileName,"wb");
	if(!fp){return 0;}
	int w=write(fp);
	fclose(fp);
	return w;
}
template<class Real>
int RotationGrid<Real>::read(FILE* fp){
	int io,r;
	io=int(fread(&r,sizeof(int),1,fp));
	if(!io){return 0;}
	resize(r);
	io=int(fread(values,sizeof(Real),res*res*res,fp));
	if(io==res*res*res){return 1;}
	else{return 0;}
}
template<class Real>
int RotationGrid<Real>::write(FILE* fp) const {
	int io;
	io=int(fwrite(&res,sizeof(int),1,fp));
	if(!io){return 0;}
	io=int(fwrite(values,sizeof(Real),res*res*res,fp));
	if(io==res*res*res){return 1;}
	else{return 0;}
}
template<class Real>
int RotationGrid<Real>::resolution(void) const{return res;}
template<class Real>
int RotationGrid<Real>::resize( int r )
{
	if(r<0){return 0;}
	else{
		if(values){delete[] values;}
		values=NULL;
		res=0;
		if(r){
			values=new Real[r*r*r];
			if(!values){return 0;}
			else{res=r;}
		}
		clear();
		return 1;
	}
}
template<class Real>
void RotationGrid<Real>::clear(void){if(res){memset(values,0,sizeof(Real)*res*res*res);}}
template<class Real>
void RotationGrid<Real>::SetCoordinates( Real theta , Real phi , Real psi , Real matrix[3][3] )
{
	Real ci, cj, ch, si, sj, sh, cc, cs, sc, ss;
    ci = cos(theta);	cj = cos(phi);	ch = cos(psi);
    si = sin(theta);	sj = sin(phi);	sh = sin(psi);

    cc = ci*ch;		cs = ci*sh;		sc = si*ch;		ss = si*sh;

	matrix[1][1] =  cj;		matrix[1][2] =  sj*si;		matrix[1][0] =  sj*ci;
	matrix[2][1] =  sj*sh;	matrix[2][2] = -cj*ss+cc;	matrix[2][0] = -cj*cs-sc;
	matrix[0][1] = -sj*ch;	matrix[0][2] =  cj*sc+cs;	matrix[0][0] =  cj*cc-ss;
}
template<class Real>
void RotationGrid<Real>::SetCoordinates( const Real matrix[3][3] , Real& theta , Real& phi , Real& psi )
{
	Real sy;
	sy=Real(sqrt(matrix[1][2]*matrix[1][2]+matrix[1][0]*matrix[1][0]));
	if(sy>16*FLT_EPSILON){
		theta=Real( atan2(matrix[1][2],matrix[1][0]));
		phi=Real( atan2(sy,matrix[1][1]));
		psi=Real( atan2(matrix[2][1],-matrix[0][1]));
	}
	else{
		theta=Real( atan2(-matrix[2][0],matrix[2][2]));
		phi=Real(atan2(sy,matrix[1][1]));
		psi=0;
	}
}

template<class Real>
void RotationGrid<Real>::SetCoordinates( const Real axis[3] , Real angle , Real matrix[3][3] )
{
	Real m1[3][3],m2[3][3];
	Real theta,phi;
	SphericalGrid<Real>::SetCoordinates(axis,theta,phi);
	SetCoordinates(0,-phi,-theta,m1);
	SetCoordinates(theta,phi,angle,m2);

	for(int i=0;i<3;i++){
		for(int j=0;j<3;j++){
			matrix[i][j]=0;
			for(int k=0;k<3;k++){
				matrix[i][j]+=m2[k][j]*m1[i][k];
			}
		}
	}
}

template<class Real>
void RotationGrid<Real>::setCoordinates( Real i , Real j , Real k , Real matrix[3][3] ) const
{
	Real theta,phi,psi;
	theta=Real(2.0*i*PI/res);
	phi=Real(PI*(2.0*j+1)/(2.0*res));
	psi=Real(2.0*k*PI/res);

	SetCoordinates(theta,phi,psi,matrix);
}
template<class Real>
void RotationGrid<Real>::setCoordinates( const Real matrix[3][3] , Real& i , Real& j , Real& k ) const
{
	Real theta,phi,psi;
	SetCoordinates(matrix,theta,phi,psi);
	i=(theta*res)/Real(2.0*PI);
	j=Real(2.0*res*phi-PI)/Real(2*PI);
	k=(psi*res)/Real(2.0*PI);
}
template<class Real>
Real& RotationGrid<Real>::operator() ( int i , int j , int k )
{
	int x=i+res/2;
	int y=j;
	int z=k+res/2;

	if(y<0){y=2*res-((-y)%(2*res));}
	y%=2*res;
	if(y>=res){
		y=2*res-y-1;
		x+=res/2;
		z+=res/2;
	}
	if(x<0){x=res-((-x)%res);}
	x=x%res;
	if(z<0){z=res-((-z)%res);}
	z=z%res;
	return values[x*res*res+y*res+z];
}
template<class Real>
Real RotationGrid<Real>::operator() ( int i , int j , int k ) const
{
	int x=i+res/2;
	int y=j;
	int z=k+res/2;

	if(y<0){y=2*res-((-y)%(2*res));}
	y%=2*res;
	if(y>=res){
		y=2*res-y-1;
		x+=res/2;
		z+=res/2;
	}
	if(x<0){x=res-((-x)%res);}
	x=x%res;
	if(z<0){z=res-((-z)%res);}
	z=z%res;
	return values[x*res*res+y*res+z];
}
template<class Real>
Real RotationGrid<Real>::operator() ( double x , double y , double z )
{
	int x1,y1,z1;
	double dx,dy,dz;

	if(x<0){x1=-int(-x)-1;}
	else{x1=int(x);}
	if(y<0){y1=-int(-y)-1;}
	else{y1=int(y);}
	if(z<0){z1=-int(-z)-1;}
	else{z1=int(z);}

	dx=x-x1;
	dy=y-y1;
	dz=z-z1;
	return 
		(*this)(x1  ,y1  ,z1  )*(Real(1.0)-dx)*(Real(1.0)-dy)*(Real(1.0)-dz)+
		(*this)(x1+1,y1  ,z1  )*(          dx)*(Real(1.0)-dy)*(Real(1.0)-dz)+
		(*this)(x1  ,y1+1,z1  )*(Real(1.0)-dx)*(          dy)*(Real(1.0)-dz)+
		(*this)(x1+1,y1+1,z1  )*(          dx)*(          dy)*(Real(1.0)-dz)+
		(*this)(x1  ,y1  ,z1+1)*(Real(1.0)-dx)*(Real(1.0)-dy)*(          dz)+
		(*this)(x1+1,y1  ,z1+1)*(          dx)*(Real(1.0)-dy)*(          dz)+
		(*this)(x1  ,y1+1,z1+1)*(Real(1.0)-dx)*(          dy)*(          dz)+
		(*this)(x1+1,y1+1,z1+1)*(          dx)*(          dy)*(          dz);
}
template<class Real>
Real RotationGrid<Real>::operator() ( double x , double y , double z ) const
{
	int x1,y1,z1;
	double dx,dy,dz;

	if(x<0){x1=-int(-x)-1;}
	else{x1=int(x);}
	if(y<0){y1=-int(-y)-1;}
	else{y1=int(y);}
	if(z<0){z1=-int(-z)-1;}
	else{z1=int(z);}

	dx=x-x1;
	dy=y-y1;
	dz=z-z1;
	return 
		(*this)(x1  ,y1  ,z1  )*(Real(1.0)-dx)*(Real(1.0)-dy)*(Real(1.0)-dz)+
		(*this)(x1+1,y1  ,z1  )*(          dx)*(Real(1.0)-dy)*(Real(1.0)-dz)+
		(*this)(x1  ,y1+1,z1  )*(Real(1.0)-dx)*(          dy)*(Real(1.0)-dz)+
		(*this)(x1+1,y1+1,z1  )*(          dx)*(          dy)*(Real(1.0)-dz)+
		(*this)(x1  ,y1  ,z1+1)*(Real(1.0)-dx)*(Real(1.0)-dy)*(          dz)+
		(*this)(x1+1,y1  ,z1+1)*(          dx)*(Real(1.0)-dy)*(          dz)+
		(*this)(x1  ,y1+1,z1+1)*(Real(1.0)-dx)*(          dy)*(          dz)+
		(*this)(x1+1,y1+1,z1+1)*(          dx)*(          dy)*(          dz);
}
template< class Real > Real RotationGrid< Real >::squareL2Norm( void ) const { return Dot( *this , *this ); }
template< class Real > Real RotationGrid< Real >::l1Norm( void ) const
{
	double l1 = 0 , c = 1;
	for( int i=0 ; i<res ; i++ )
	{
		double t1 = cos(PI*(2.0*i+2)/(2.0*res));
		double t2 = ((c-t1)/(2.0*res)/(2.0*res));
		c = t1;
		for( int j=0 ; j<res ; j++ ) for( int k=0 ; k<res ; k++ ) l1 += fabs( values[j*res*res+i*res+k] ) * t2;
	}
	return Real( l1 * 4.0/3.0 * PI );
}

template<class Real>
Real RotationGrid<Real>::SquareL2Difference(const RotationGrid& g1,const RotationGrid& g2){return g1.squareL2Norm()+g2.squareL2Norm()-2*Dot(g1,g2);}
template<class Real>
Real RotationGrid<Real>::L1Difference(const RotationGrid& g1,const RotationGrid& g2){
	double d=0,c=1;
	if(g1.res != g2.res){
		fprintf(stderr,"Could not compare arrays of different sizes: %d != %d\n",g1.res,g2.res);
		exit(0);
	}
	for( int i=0 ; i<g1.res ; i++ )
	{
		double t1 =cos(PI*(2.0*i+2)/(2.0*g1.res));
		double t2=((c-t1)/(2.0*g1.res)/(2.0*g1.res));
		c=t1;
		for( int j=0 ; j<g1.res ; j++ ) for( int k=0 ; k<g1.res ; k++ ) d += (Real)fabs( g1.values[j*g1.res*g1.res+i*g1.res+k] - g2.values[j*g1.res*g1.res+i*g1.res+k] ) * t2;
	}
	return Real(d*4.0/3.0*PI);
}
template<class Real>
Real RotationGrid<Real>::Dot(const RotationGrid& g1,const RotationGrid& g2){
	double d=0,c=1;
	if(g1.res != g2.res){
		fprintf(stderr,"Could not compare arrays of different sizes: %d != %d\n",g1.res,g2.res);
		exit(0);
	}
	for(int i=0;i<g1.res;i++){
		double t1 =cos(PI*(2.0*i+2)/(2.0*g1.res));
		double t2=((c-t1)/(2.0*g1.res)/(2.0*g1.res));
		c=t1;
		for(int j=0;j<g1.res;j++){for(int k=0;k<g1.res;k++){d+=g1.values[j*g1.res*g1.res+i*g1.res+k]*g2.values[j*g1.res*g1.res+i*g1.res+k]*t2;}}
	}
	return Real(d*4.0/3.0*PI);
}
