#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <soft_fftw.h>
#include <utils_so3.h>
#include <math.h>
#include "fftw3.h"

///////////////////
// FourierKeySO3 //
///////////////////
template<class Real> int FourierKeySO3<Real>::Entries(const int& bw){return (4*bw*bw*bw+3*bw*bw-bw)/6;}
template<class Real> FourierKeySO3<Real>::FourierKeySO3(void){
	bw=0;
	values=NULL;
}
template<class Real> FourierKeySO3<Real>::~FourierKeySO3(void){
	if(values){delete[] values;}
	values=NULL;
	bw=0;
}
template<class Real> int FourierKeySO3<Real>::read(const char* fileName){
	FILE* fp=fopen(fileName,"rb");
	if(!fp){return 0;}
	int r=read(fp);
	fclose(fp);
	return r;
}
template<class Real> int FourierKeySO3<Real>::write(const char* fileName) const{
	FILE* fp=fopen(fileName,"wb");
	if(!fp){return 0;}
	int w=write(fp);
	fclose(fp);
	return w;
}
template<class Real> int FourierKeySO3<Real>::read(FILE* fp){
	int b,r;
	r=int(fread(&b,sizeof(int),1,fp));
	if(!r){return 0;}
	resize(b);
	r=int(fread(values,sizeof(Complex<Real>),Entries(bw),fp));
	if(r==Entries(bw)){return 1;}
	else{return 0;}
}
template<class Real> int FourierKeySO3<Real>::write(FILE* fp) const {
	int w;
	w=int(fwrite(&bw,sizeof(int),1,fp));
	if(!w){return 0;}
	w=int(fwrite(values,sizeof(Complex<Real>),Entries(bw),fp));
	if(w==Entries(bw)){return 1;}
	else{return 0;}
}
template<class Real> int FourierKeySO3<Real>::bandWidth(void) const{return bw;}
template<class Real> int FourierKeySO3<Real>::resolution(void) const {return bw*2;}
template<class Real> int FourierKeySO3<Real>::resize(const int& resolution,const int& clr){
	int b=resolution>>1;
	if(b<0){return 0;}
	else if(b!=bw){
		if(values){delete[] values;}
		values=NULL;
		bw=0;
		if(b){
			values=new Complex<Real>[Entries(b)];
			if(!values){return 0;}
			else{bw=b;}
		}
	}
	if(clr){clear();}
	return 1;
}
template<class Real> void FourierKeySO3<Real>::clear(void){if(bw){memset(values,0,sizeof(Complex<Real>)*Entries(bw));}}
template<class Real> Complex<Real>& FourierKeySO3<Real>::operator() (const int& i,const int& j,const int& k){
	return values[so3CoefLoc(j,k,i,bw)];
}
template<class Real> Complex<Real> FourierKeySO3<Real>::operator() (const int& i,const int& j,const int& k) const {
	if(j<0){
		int jj=-j;
		int kk=-k;
		return values[so3CoefLoc(jj,kk,i,bw)].conjugate();
	}
	else{return values[so3CoefLoc(j,k,i,bw)];}
}
template<class Real> Real FourierKeySO3<Real>::squareNorm(void) const{return Dot(*this,*this).r;}
template<class Real> Real FourierKeySO3<Real>::SquareDifference(const FourierKeySO3& g1,const FourierKeySO3& g2){return g1.squareNorm()+g2.squareNorm()-2*Dot(g1,g2).r;}
template<class Real> Complex<Real> FourierKeySO3<Real>::Dot(const FourierKeySO3& g1,const FourierKeySO3& g2){
	Complex<Real> d;
	int idx=0;
	if(g1.bw != g2.bw){
		fprintf(stderr,"Could not compare arrays of different sizes: %d != %d\n",g1.bw,g2.bw);
		exit(0);
	}
	for(int i=0;i<g1.bw;i++){
		for(int j=0;j<=i;j++){
			d+=g1(i,j,0)*g2(i,j,0).conjugate();
			for(int k=1;k<=i;k++){
				d+=g1(i,j,k)*g2(i,j,k).conjugate()*2;
				d+=g1(i,j,-k)*g2(i,j,-k).conjugate()*2;
			}
		}
	}
	return d;
}
///////////////////////////////////
// WignerTransform::ScratchSpace //
///////////////////////////////////
template<class Real>
WignerTransform<Real>::ScratchSpace::ScratchSpace(void){
	bw=0;
	data=coeffs=workspace_cx=workspace_cx2=NULL;
	workspace_re=NULL;
	p=0;
}
template<class Real>
WignerTransform<Real>::ScratchSpace::~ScratchSpace(void){resize(0);}
template<class Real>
void WignerTransform<Real>::ScratchSpace::resize(const int& b){
	if(b!=bw){
		int size=b*2;
		if(data)					{fftw_free(data);}
		if(coeffs)					{fftw_free(coeffs);}
		if(workspace_cx)			{fftw_free(workspace_cx);}
		if(workspace_cx2)			{fftw_free(workspace_cx2);}
		if(workspace_re)			{fftw_free(workspace_re);}
		if(p)						{fftw_destroy_plan(p);}

		bw=0;
		data=coeffs=workspace_cx=workspace_cx2=NULL;
		workspace_re=NULL;
		p=0;

		if(b>0){
			bw=b;
			data=(fftw_complex*)fftw_malloc(sizeof(fftw_complex)*size*size*size);
			coeffs=(fftw_complex*)fftw_malloc(sizeof(fftw_complex)*(4*bw*bw*bw-3*bw)/3);
			workspace_cx=(fftw_complex*)fftw_malloc(sizeof(fftw_complex)*size*size*size);
			workspace_cx2=(fftw_complex*)fftw_malloc(sizeof(fftw_complex)*size);
			workspace_re=(double*)fftw_malloc(sizeof(double)*(12*size+size*bw));

			int na[2], inembed[2], onembed[2] ;
			int rank, howmany, istride, idist, ostride, odist ;
			howmany = size*size ;
			idist = size ;
			odist = size ;
			rank = 2 ;
			inembed[0] = size ;
			inembed[1] = size*size ;
			onembed[0] = size ;
			onembed[1] = size*size ;
			istride = 1 ;
			ostride = 1 ;
			na[0] = 1 ;
			na[1] = size ;

			p = fftw_plan_many_dft( rank, na, howmany,
				workspace_cx, inembed,
				istride, idist,
				data, onembed,
				ostride, odist,
				FFTW_FORWARD, FFTW_ESTIMATE );
		}
	}
}
/////////////////////
// WignerTransform //
/////////////////////
template<class Real>
void WignerTransform<Real>::resize(const int& resolution){scratch.resize(resolution>>1);}
template<class Real>
int WignerTransform<Real>::InverseFourier(FourierKeySO3<Real>& key,RotationGrid<Real>& g){
	if(key.resolution()!=g.resolution()){g.resize(key.resolution());}
	int bw=key.bandWidth(),sz=g.resolution();
	scratch.resize(bw);
	int i,j,k,m,h,idx=0;
	double n,temp;
	for(i=0;i<bw;i++){
		for(j=0;j<bw;j++){
			temp=1;
			if(i>j){
				if((i-j)%2){temp*=-1;}
				m=i;
			}
			else{
				if((j-i)%2){temp*=-1;}
				m=j;
			}
			h=bw-m;
			for(k=0;k<h;k++){
				n=sqrt(8.0*PI*PI/(2*(m+k)+1));
				scratch.coeffs[idx][0]=key(m+k,i,j).r*n*temp;
				scratch.coeffs[idx][1]=key(m+k,i,j).i*n*temp;
				idx++;
			}
		}
		for(j=bw-1;j>0;j--){
			if(j%2==0){temp=1;}
			else{temp=-1;}
			if(i>j){
				if((i-j)%2){temp*=-1;}
				m=i;
			}
			else{
				if((j-i)%2){temp*=-1;}
				m=j;
			}
			h=bw-m;
			for(k=0;k<h;k++){
				n=sqrt(8.0*PI*PI/(2*(m+k)+1));
				scratch.coeffs[idx][0]=key(m+k,i,-j).r*n*temp;
				scratch.coeffs[idx][1]=key(m+k,i,-j).i*n*temp;
				idx++;
			}
		}
	}
	for(i=bw-1;i>0;i--){
		for(j=0;j<bw;j++){
			if(i%2==0){temp=1;}
			else{temp=-1;}
			if(i>j){
				if((i-j)%2){temp*=-1;}
				m=i;
			}
			else{
				if((j-i)%2){temp*=-1;}
				m=j;
			}
			h=bw-m;
			for(k=0;k<h;k++){
				n=sqrt(8.0*PI*PI/(2*(m+k)+1));
				scratch.coeffs[idx][0]= key(m+k,i,-j).r*n*temp;
				scratch.coeffs[idx][1]=-key(m+k,i,-j).i*n*temp;
				idx++;
			}
		}
		for(j=bw-1;j>0;j--){
			if((i+j)%2==0){temp=1;}
			else{temp=-1;}
			if(i>j){
				if((i-j)%2){temp*=-1;}
				m=i;
			}
			else{
				if((j-i)%2){temp*=-1;}
				m=j;
			}
			h=bw-m;
			for(k=0;k<h;k++){
				n=sqrt(8.0*PI*PI/(2*(m+k)+1));
				scratch.coeffs[idx][0]= key(m+k,i,j).r*n*temp;
				scratch.coeffs[idx][1]=-key(m+k,i,j).i*n*temp;
				idx++;
			}
		}
	}
	Inverse_SO3_Naive_fftw(bw,scratch.coeffs,scratch.data,scratch.workspace_cx,scratch.workspace_cx2,scratch.workspace_re,&scratch.p,1);
	idx=0;
	for(int i=0;i<2*bw;i++){
		for(int j=0;j<2*bw;j++){
			for(int k=0;k<2*bw;k++){
				g.values[idx]=Real(scratch.data[2*bw*i+4*bw*bw*j+k][0]);
				idx++;
			}
		}
	}
	return 1;
}
