/*
Copyright (c) 2009, Michael Kazhdan
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

#include "MemoryUsage.h"
#include "CmdLineParser.h"
#include "LaplacianMatrix1D.h"
#include "TriangleIntegrals.h"
#include "Util.h"

//////////////////////
// MeshTreeNodeData //
//////////////////////
template< class Real >
MeshTreeNodeData< Real >::MeshTreeNodeData (void)
{
	index = tempIndex = -1;
	tStart = tEnd = 0;
}
template< class Real > MeshTreeNodeData< Real >::~MeshTreeNodeData(void)	{	;	}


////////////////
// MeshOctree //
////////////////
template< class Real , bool Primal >
double MeshOctree< Real , Primal >::maxMemoryUsage=0;

template< class Real , bool Primal >
double MeshOctree< Real , Primal >::MemoryUsage(void)
{
	double mem = double( MemoryInfo::Usage() ) / (1<<20);
	if(mem>maxMemoryUsage) maxMemoryUsage=mem;
	return mem;
}

template< class Real , bool Primal >
MeshOctree< Real , Primal >::MeshOctree(void)
{
	TreeOctNode::SetAllocator( MEMORY_ALLOCATOR_BLOCK_SIZE );
}
class DataSortIndex
{
public:
	int sortIndex , dataIndex;
	static int Compare( const void* v1 , const void* v2 )
	{
		DataSortIndex* si1 = (DataSortIndex*)v1;
		DataSortIndex* si2 = (DataSortIndex*)v2;
		return si1->sortIndex-si2->sortIndex;
	}
};

template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_addSamples( const std::vector< OrientedPoint3D< Real > >& samples , const std::vector< int >& indices , int depth , bool progress )
{
	ProgressBar* bar=NULL;
	int nodeCount=0;
	std::vector< DataSortIndex > sampleIndices;
	std::vector< int > pointStart;
	sampleIndices.resize( samples.size() );

	if( progress ) bar = new ProgressBar( 20 , samples.size()   , "Adding Samples" );
	for( int i=0 ; i<samples.size() ; i++ )
	{
		if( progress ) bar->update();
		Point3D< Real > p = samples[i].position;
		TreeOctNode* node = &tree;
		while( node->depth()<depth )
		{
			Point3D< Real > center;
			Real width;
			int x , y , z;
			x = y = z = 0;
			node->centerAndWidth( center , width );
			if( Primal )
			{
				center *= 2;
				width  *= 2;
			}
			if( p[0]>center[0] ) x=1;
			if( p[1]>center[1] ) y=1;
			if( p[2]>center[2] ) z=1;

			if( !node->children ) node->initChildren();
			node = node->children + Cube::CornerIndex( x , y , z );
		}
		if( node->nodeData.index==-1 ) node->nodeData.index = nodeCount++;
		sampleIndices[i].dataIndex = i;
		sampleIndices[i].sortIndex = node->nodeData.index;
	}
	if( progress ) delete bar;
	qsort( &sampleIndices[0] , sampleIndices.size() , sizeof( DataSortIndex ) , DataSortIndex::Compare );
	pointStart.resize( nodeCount+1 );
	nodeCount=-1;
	for( int i=0 ; i<samples.size() ; i++ ) while( sampleIndices[i].sortIndex!=nodeCount ) pointStart[++nodeCount] = i;
	pointStart[++nodeCount] = (int)samples.size();
	_samples.resize( samples.size() );
	_parents.resize( samples.size() );
	for( int i=0 ; i<samples.size() ; i++ )
	{
		_samples[i] = samples[ sampleIndices[i].dataIndex ];
		_parents[i] = indices[ sampleIndices[i].dataIndex ];
	}

	for( TreeOctNode* node = tree.nextNode() ; node ; node = tree.nextNode( node ) )
		if( node->nodeData.index!=-1 )
		{
			node->nodeData.tStart = pointStart[ node->nodeData.index   ];
			node->nodeData.tEnd   = pointStart[ node->nodeData.index+1 ];
			node->nodeData.index  = - 1;
		}
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_setTriangles( TreeOctNode* node , const std::vector< TriangleIndex >& triangles , const std::vector< int >& parents , int& offset )
{
	if( node->children )
	{
		for( int c=0 ; c<Cube::CORNERS ; c++ ) _setTriangles( node->children+c , triangles , parents , offset );
		node->nodeData.tStart = node->children[0].nodeData.tStart;
		node->nodeData.tEnd   = node->children[Cube::CORNERS-1].nodeData.tEnd;
	}
	else
	{
		for( int i=0 ; i<node->nodeData.tEnd-node->nodeData.tStart ; i++ )
		{
			_triangles[offset+i] = triangles[node->nodeData.tStart+i];
			_parents  [offset+i] = parents  [node->nodeData.tStart+i];
		}
		node->nodeData.tEnd = offset + node->nodeData.tEnd - node->nodeData.tStart;
		node->nodeData.tStart = offset;
		offset += node->nodeData.tEnd-node->nodeData.tStart;
	}
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_addTriangles( const std::vector< TriangleIndex >& triangles , int depth )
{
	class VoxelIndex
	{
	public:
		int idx[3];
		bool operator == ( const VoxelIndex& i ) const { return (idx[0]==i.idx[0]) && (idx[1]==i.idx[1]) && (idx[2]==i.idx[2]); }
	};

	int nodeCount=0;
	std::vector< DataSortIndex > triangleIndices;
	std::vector< int > triangleStart;
	triangleIndices.resize( triangles.size() );

	for( int i=0 ; i<triangles.size() ; i++ )
	{
		const TriangleIndex& tIndex = triangles[i];

		// Compute the finest depth at which the triangle is entirely contained within a node
		int tDepth;
		VoxelIndex idx[3];
		for( tDepth=0 ; tDepth<depth ; tDepth++ )
		{
			for( int j=0 ; j<3 ; j++ ) for( int c=0 ; c<3 ; c++ ) idx[j].idx[c] = int( _vertices[ tIndex[j] ][c] * (1<<(tDepth+1)) );
			if( !(idx[0]==idx[1]) || !(idx[0]==idx[2]) ) break;
		}

		// Add the triangle index to every node that completely contains the triangle
		for( int j=0 ; j<3 ; j++ ) for( int c=0 ; c<3 ; c++ ) idx[j].idx[c] = int( _vertices[ tIndex[j] ][c] * (1<<tDepth) );
		Point3D< Real > p;
		for( int c=0 ; c<3 ; c++ ) p[c] = ( Real(idx[0].idx[c])+Real( 0.5 ) ) / (1<<tDepth);
		TreeOctNode* node = &tree;
		if( Primal )
		{
			tDepth++;
			if( tDepth>depth ) tDepth = depth;
		}
		while( node->depth()<tDepth )
		{
			Point3D< Real > center;
			Real width;
			int x , y , z;
			x = y = z = 0;
			node->centerAndWidth( center , width );
			if( Primal )
			{
				center *= 2;
				width  *= 2;
			}
			if( p[0]>center[0] ) x=1;
			if( p[1]>center[1] ) y=1;
			if( p[2]>center[2] ) z=1;

			if( !node->children ) node->initChildren();
			node = node->children + Cube::CornerIndex( x , y , z );
		}
		if( node->nodeData.index==-1 ) node->nodeData.index = nodeCount++;
		triangleIndices[i].dataIndex = i;
		triangleIndices[i].sortIndex = node->nodeData.index;
	}
	qsort( &triangleIndices[0] , triangleIndices.size() , sizeof( DataSortIndex ) , DataSortIndex::Compare );

	triangleStart.resize( nodeCount+1 );
	nodeCount=-1;
	for( int i=0 ; i<triangles.size() ; i++ ) while( triangleIndices[i].sortIndex!=nodeCount ) triangleStart[++nodeCount] = i;
	triangleStart[++nodeCount] = (int) triangles.size();

	_triangles.resize( triangles.size() );
	for( int i=0 ; i<triangles.size() ; i++ ) _triangles[i] = triangles[ triangleIndices[i].dataIndex ];

	for( TreeOctNode* node = tree.nextNode() ; node ; node = tree.nextNode( node ) )
		if( node->nodeData.index!=-1 )
		{
			node->nodeData.tStart = triangleStart[ node->nodeData.index   ];
			node->nodeData.tEnd   = triangleStart[ node->nodeData.index+1 ];
			node->nodeData.index  = - 1;
		}

	std::vector< PolygonData > polygons;
	std::vector< TriangleIndex > splitTriangles;
	std::vector< int > splitParents;
	hash_map< long long , int > vTable;
	_addTriangles( &tree , _triangles , polygons , depth , vTable , splitTriangles , splitParents );

	_triangles.resize( splitTriangles.size() );
	_parents.resize  ( splitParents.size() );
	for( int i=0 ; i<splitTriangles.size() ; i++ ) _triangles[i] = splitTriangles[i];
	for( int i=0 ; i<splitParents.size()   ; i++ ) _parents  [i] = triangleIndices[ splitParents[i] ].dataIndex;
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_addTriangles( TreeOctNode* node , const std::vector< TriangleIndex >& triangles , std::vector< PolygonData >& polygons , int depth ,
												 hash_map< long long , int >& vTable , std::vector< TriangleIndex >& splitTriangles , std::vector< int >& splitParents )
{
	int start = node->nodeData.tStart;
	int end   = node->nodeData.tEnd;
	node->nodeData.tStart = node->nodeData.tEnd = 0;

	int d = node->depth();
	// If we're deep enough, return
	if( d>depth ) return;
	// If there is nothing to add, return
	if( !polygons.size() && start==end && !node->children ) return;

	// Add the triangles from the polygons
	if( d==depth )
	{
		node->nodeData.tStart = (int)splitTriangles.size();

		MinimalAreaTriangulation< Real > MAT;
		for( int i=0 ; i<polygons.size() ; i++ )
		{
			std::vector< Point3D< Real > > vertices;
			std::vector< TriangleIndex > triangles;
			vertices.resize( polygons[i].polygon.size() );
			for( int j=0 ; j<polygons[i].polygon.size() ; j++ ) vertices[j] = _vertices[ polygons[i].polygon[j] ];
			MAT.GetTriangulation( vertices , triangles );
			TriangleIndex tIndex;
			size_t sz = splitTriangles.size();
			splitTriangles.resize( sz+triangles.size() );
			splitParents.resize  ( sz+triangles.size() );

			for( int j=0 ; j<triangles.size() ; j++ )
			{
				tIndex[0] = polygons[i].polygon[ triangles[j][0] ];
				tIndex[1] = polygons[i].polygon[ triangles[j][1] ];
				tIndex[2] = polygons[i].polygon[ triangles[j][2] ];
				splitTriangles[sz+j] = tIndex;
				splitParents  [sz+j] = polygons[i].parent;
			}
		}
	}

	// Add the triangles stored with the node
	size_t sz=0;
	if( d==depth )
	{
		sz = splitTriangles.size();
		splitTriangles.resize( sz+(end-start) );
		splitParents.resize  ( sz+(end-start) );
	}

	for( int i=0 ; i<end-start ; i++ )
	{
		int idx = start+i;
		if( d<depth )
		{
			sz = polygons.size();
			polygons.resize( sz+1 );
			polygons[sz].polygon.resize( 3 );
			for( int j=0 ; j<3 ; j++ ) polygons[sz].polygon[j] = triangles[ idx ][j];
			polygons[sz].parent = idx;
		}
		if( d==depth )
		{
			splitTriangles[sz+i] = triangles[idx];
			splitParents  [sz+i] = idx;
		}
	}
	if( d==depth )
	{
		node->nodeData.tEnd = (int) splitTriangles.size();
		return;
	}

	// Split up the polygons and pass them on to the children

	Point3D< Real > axis[3] , center;
	Real width;
	node->centerAndWidth( center , width );
	if( Primal )
	{
		center *= 2;
		width  *= 2;
	}
	for( int i=0 ; i<3 ; i++ )
	{
		axis[i] *= 0;
		axis[i][i] = 1;
	}
	std::vector< PolygonData > subPolygons[ Cube::CORNERS ];

	if( !node->children ) node->initChildren();

	for( int i=0 ; i<polygons.size() ; i++ )
	{
		std::vector< int > xPolygons[2];
		SplitPolygon( _vertices , vTable , polygons[i].polygon , axis[0] , center[0] , xPolygons[0] , xPolygons[1] );
		for( int x=0 ; x<2 ; x++ )
		{
			std::vector< int > yPolygons[2];
			SplitPolygon( _vertices , vTable , xPolygons[x] , axis[1] , center[1] , yPolygons[0] , yPolygons[1] );
			for( int y=0 ; y<2 ; y++ )
			{
				std::vector< int > zPolygons[2];
				SplitPolygon( _vertices , vTable , yPolygons[y] , axis[2] , center[2] , zPolygons[0] , zPolygons[1] );
				for( int z=0 ; z<2 ; z++ )
				{
					int idx = Cube::CornerIndex( x , y , z );
					if( zPolygons[z].size() )
					{
						size_t sz = subPolygons[idx].size();
						subPolygons[idx].resize( sz+1 );
						subPolygons[idx][sz].parent = polygons[i].parent;
						subPolygons[idx][sz].polygon.resize( zPolygons[z].size() );
						for( int i=0 ; i<zPolygons[z].size() ; i++ ) subPolygons[idx][sz].polygon[i] = zPolygons[z][i];
					}
				}
			}
		}
	}
	for( int c=0 ; c<Cube::CORNERS ; c++ ) _addTriangles( node->children+c , triangles , subPolygons[c] , depth , vTable , splitTriangles , splitParents );
}

template< class Real , bool Primal >
int MeshOctree< Real , Primal >::dimension( int depth ) const
{
	if( Primal ) return _dimension[depth+1];
	else         return _dimension[depth  ];
}

template< class Real , bool Primal >
int MeshOctree< Real , Primal >::setTree( const std::vector< OrientedPoint3D< Real > >& samples , const std::vector< int >& indices , int depth , double cutOff , bool progress )
{
	_fromSamples = true;
	if( Primal ) _addSamples( samples , indices , depth+1 , progress );
	else         _addSamples( samples , indices , depth   , progress );
	if( Primal ) return _setPrimalTree( depth+1 , cutOff );
	else         return _setDualTree  ( depth   , cutOff );
}
template< class Real , bool Primal >
template<class OrientedVertex>
int MeshOctree< Real , Primal >::setTree( const std::vector< OrientedVertex >& vertices , const std::vector< TriangleIndex >& triangles , int depth , double cutOff , bool progress )
{
	_fromSamples = false;
	_baseNormals.resize( triangles.size() );
	for( int i=0 ; i<triangles.size() ; i++ )
	{
		TriangleIndex tIndex = triangles[ i ];
		Point3D< double > v1 = Point3D< double >( vertices[ tIndex[0] ] ) , v2 = Point3D< double >( vertices[ tIndex[1] ] ) , v3 = Point3D< double >( vertices[ tIndex[2] ] );
		Point3D< double > normal = Point3D< double >::CrossProduct( v2-v1 , v3-v1 ) ;
		_baseNormals[ i ] = Point3D< Real >( normal / Length( normal ) );
	}
	_vertices.resize( vertices.size() );
	for( int i=0 ; i<vertices.size() ; i++ ) _vertices[i] = Point3D< Real >( vertices[i].point );
	if( Primal ) _addTriangles( triangles , depth+1 );
	else         _addTriangles( triangles , depth   );

	if( Primal ) return _setPrimalTree( depth+1 , cutOff );
	else         return _setDualTree  ( depth   , cutOff );
}

template< class Real , bool Primal >
void MeshOctree< Real , Primal >::setNoiseCoefficients( const double& persistence , std::vector< Point3D< Real > >& coefficients ) const
{
	_setNoiseCoefficients< Point3D< Real > >( persistence , coefficients );
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::setNoiseCoefficients( const int& minDepth , const int& maxDepth , const double& persistence , std::vector< Point3D< Real > >& coefficients ) const
{
	_setNoiseCoefficients< Point3D< Real > >( minDepth , maxDepth , persistence , coefficients );
}


template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::_setNoiseCoefficients( const double& persistence , std::vector< C >& coefficients ) const
{
	_setNoiseCoefficients( 1 , tree.maxDepth() , persistence , coefficients );
}
template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::_setNoiseCoefficients( int minDepth , int maxDepth , const double& persistence , std::vector< C >& coefficients ) const
{
	if( Primal )
	{
		minDepth++;
		maxDepth++;
	}
	std::vector< C > _coefficients;
	double p = persistence;
	if( p<0 ) p = 1;
	_coefficients.resize( _dimension[minDepth] );
	for( int d=minDepth ; d<=maxDepth ; d++ )
	{
		double amplitude=pow( p , d );
		coefficients.resize( _dimension[d] );
		for( const TreeOctNode* node=tree.nextNode() ; node ; node=tree.nextNode(node) )
		{
			if( node->depth()!=d || node->nodeData.index<0 ) continue;
			for( int c=0 ; c<3 ; c++ )	coefficients[node->nodeData.index][c] = _coefficients[node->nodeData.index][c] + Real( amplitude*(Random2<double>()*2.0-1.0) );
		}
		if( d<maxDepth ) upSample( coefficients , d , _coefficients );
	}
	for( int d=maxDepth ; d<tree.maxDepth() ; d++ )
	{
		_coefficients.resize( coefficients.size() );
		for( int i=0 ; i<coefficients.size() ; i++ ) _coefficients[i] = coefficients[i];
		upSample( _coefficients , d , coefficients );
	}
	Real max = 0;
	for( int i=0 ; i<coefficients.size() ; i++ ) for( int c=0 ; c<3 ; c++ ) if( fabs( coefficients[i][c] ) > max ) max = fabs( coefficients[i][c] );
	for( int i=0 ; i<coefficients.size() ; i++ ) for( int c=0 ; c<3 ; c++ ) coefficients[i][c] = coefficients[i][c] / max;

}

template< class Real , bool Primal >
void MeshOctree< Real , Primal >::setCheckerBoardCoefficients( int depth , std::vector< Point3D< Real > >& coefficients ) const
{
	if( Primal ) depth++;

	int maxDepth = tree.maxDepth();
	coefficients.resize( _dimension[maxDepth] );
	for( const TreeOctNode* node=tree.nextNode() ; node ; node=tree.nextNode(node) )
	{
		if( node->depth()==maxDepth && node->nodeData.index!=-1)
		{
			int d , off[3];
			node->depthAndOffset(d,off);
			for( int c=0 ; c<3 ; c++ )
			{
				off[c] >>= (maxDepth-depth);
				if( off[c]&1 )	coefficients[node->nodeData.index][c] = 1;
				else			coefficients[node->nodeData.index][c] = 0;
			}
		}
	}
}


template< class Real , bool Primal >
void MeshOctree< Real , Primal >::SetTangents( const Point3D< Real >& normal , Point3D< Real >& t1 , Point3D< Real >& t2 )
{
	double nLength = sqrt( Point3D< Real >::SquareNorm( normal ) );
	if( nLength==0 )
	{
		t1 *= 0;
		t2 *= 0;
		return false;
	}
	Point3D< Real > p , n = normal / nLength;
	p[0] = 1 , p[1] = 0 , p[2] = 0;
	p = Point3D< Real >::CrossProduct( n , p );
	if( Point3D< Real >::SquareNorm( p ) < 0.00001 )
	{
		p[0] = 0 , p[1] = 1 , p[2] = 0;
		p = Point3D< Real >::CrossProduct( n , p );
	}
	t1  = p / sqrt( Point3D< Real >::SquareNorm( p ) );
	t2  = Point3D< Real >::CrossProduct( t1 , n );
	t2 /= sqrt( Point3D< Real >::SquareNorm( t2 ) );
}

template< class Real , bool Primal >
int MeshOctree< Real , Primal >::_setAdjacency( int depth , SparseMatrix< Real >& matrix ) const
{
	int max = 0;
	matrix.Resize( _dimension[depth] );

	for( const TreeOctNode* node=tree.nextNode() ; node ; node=tree.nextNode( node ) )
	{
		if( node->depth()!=depth || node->nodeData.index==-1 ) continue;
		int idx = node->nodeData.index;
		if( Primal )
		{
			matrix.SetGroupSize( idx , 3*3*3 );

			typename TreeOctNode::Neighbors3& neighbors = neighborKey3.getNeighbors( node );
			for( int i=0 ; i<3 ; i++ )
				for( int j=0 ; j<3 ; j++ )
					for( int k=0 ; k<3 ; k++ )
						if( neighbors.neighbors[i][j][k] && neighbors.neighbors[i][j][k]->nodeData.index!=-1 )
						{
							matrix[idx][i*3*3+j*3+k].N = neighbors.neighbors[i][j][k]->nodeData.index;
							matrix[idx][i*3*3+j*3+k].Value = 0;
						}
						else 
							matrix[idx][i*3*3+j*3+k].N = -1;
		}
		else
		{
			matrix.SetGroupSize( idx , 5*5*5 );

			typename TreeOctNode::Neighbors5& neighbors = neighborKey5.getNeighbors( node );
			for( int i=0 ; i<5 ; i++ )
				for( int j=0 ; j<5 ; j++ )
					for( int k=0 ; k<5 ; k++ )
						if( neighbors.neighbors[i][j][k] && neighbors.neighbors[i][j][k]->nodeData.index!=-1 )
						{
							matrix[idx][i*5*5+j*5+k].N = neighbors.neighbors[i][j][k]->nodeData.index;
							matrix[idx][i*5*5+j*5+k].Value = 0;
						}
						else
							matrix[idx][i*5*5+j*5+k].N = -1;
		}
		if( node->nodeData.tEnd-node->nodeData.tStart > max ) max = node->nodeData.tEnd - node->nodeData.tStart;
	}
	return max;
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_compressMatrix( SparseMatrix< Real >& matrix ) const
{
	if( Primal )
	{
		MatrixEntry< Real > entries[3][3][3];
		for( int idx=0 ; idx<matrix.groups ; idx++ )
		{
			int count = 0;
			for( int x=0 ; x<3 ; x++ ) for( int y=0 ; y<3 ; y++ ) for( int z=0 ; z<3 ; z++ )
				if( matrix[idx][x*3*3+y*3+z].N != -1 )
				{
					entries[x][y][z] = matrix[idx][x*3*3+y*3+z];
					count++;
				}
				else entries[x][y][z].N = -1;
				matrix.SetGroupSize( idx , count );
				matrix.groupSizes[idx] = 0;
				for( int x=0 ; x<3 ; x++ ) for( int y=0 ; y<3 ; y++ ) for( int z=0 ; z<3 ; z++ )
					if( entries[x][y][z].N != -1 ) matrix[idx][ matrix.groupSizes[idx]++ ] = entries[x][y][z];
		}
	}
	else
	{
		MatrixEntry< Real > entries[5][5][5];
		for( int idx=0 ; idx<matrix.groups ; idx++ )
		{
			int count = 0;
			for( int x=0 ; x<5 ; x++ ) for( int y=0 ; y<5 ; y++ ) for( int z=0 ; z<5 ; z++ )
				if( matrix[idx][x*5*5+y*5+z].N != -1 )
				{
					entries[x][y][z] = matrix[idx][x*5*5+y*5+z];
					count++;
				}
				else entries[x][y][z].N = -1;
				matrix.SetGroupSize( idx , count );
				matrix.groupSizes[idx] = 0;
				for( int x=0 ; x<5 ; x++ ) for( int y=0 ; y<5 ; y++ ) for( int z=0 ; z<5 ; z++ )
					if( entries[x][y][z].N != -1 ) matrix[idx][ matrix.groupSizes[idx]++ ] = entries[x][y][z];
		}
	}
}

template< class Real , bool Primal >
int MeshOctree< Real , Primal >::getDotProductMatrix( int depth , SparseMatrix<Real>& dotProduct , bool progress ) const
{
	if( Primal ) return _getPrimalDotProductMatrix( depth , dotProduct , progress );
	else         return _getDualDotProductMatrix( depth , dotProduct , progress );
}
template< class Real , bool Primal >
int MeshOctree< Real , Primal >::getLaplacianMatrix( int depth , SparseMatrix<Real>& laplacian , bool progress ) const
{
	if( Primal ) return _getPrimalLaplacianMatrix( depth , laplacian , progress );
	else         return _getDualLaplacianMatrix( depth , laplacian , progress );
}
template< class Real , bool Primal >
int MeshOctree< Real , Primal >::getConstrainedLaplacianMatrix( int depth , SparseMatrix<Real>& laplacian , Real dotWeight , Real lapWeight , bool progress ) const
{
	if( Primal ) return _getPrimalConstrainedLaplacianMatrix( depth , laplacian , dotWeight , lapWeight , progress );
	else         return _getDualConstrainedLaplacianMatrix( depth , laplacian , dotWeight , lapWeight , progress );
}
template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function >
void MeshOctree< Real , Primal >::getPreciseDotVector( const std::vector< Function >& functions , int depth , std::vector< V >& b , bool progress ) const
{
	if( Primal ) _getPrimalPreciseDotVector< V , HighPrecisionV , Function >( functions , depth , b , progress );
	else         _getDualPreciseDotVector  < V , HighPrecisionV , Function >( functions , depth , b , progress );
}

template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function >
int MeshOctree< Real , Primal >::getPreciseGradientDotGradientVector( const std::vector< Function >& gradientFunctions , int depth , std::vector< V >& b , bool progress , bool normalize ) const
{
	if( Primal ) return _getPrimalPreciseGradientDotGradientVector< V , HighPrecisionV , Function >( gradientFunctions , depth , b , progress , normalize );
	else         return _getDualPreciseGradientDotGradientVector  < V , HighPrecisionV , Function >( gradientFunctions , depth , b , progress , normalize );
}

template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function >
int MeshOctree< Real , Primal >::getPreciseConstrainedGradientDotGradientVector( const std::vector< Function >& functions , Real dotWeight , Real lapWeight , int depth , std::vector< V >& b , bool progress , bool normalize ) const
{
	if( Primal ) return _getPrimalPreciseConstrainedGradientDotGradientVector< V , HighPrecisionV , Function >( functions , dotWeight , lapWeight , depth , b , progress , normalize );
	else         return _getDualPreciseConstrainedGradientDotGradientVector  < V , HighPrecisionV , Function >( functions , dotWeight , lapWeight , depth , b , progress , normalize );
}

template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function >
V MeshOctree< Real , Primal >::getPreciseIntegral( const std::vector< Function >& functions ) const
{
	int depth = tree.maxDepth();
	HighPrecisionV integral;
	integral *= 0;
	if( _fromSamples )
		for( int i=0 ; i<_samples.size() ; i++ )
		{
			int idx = _parents[ i ];
			Point3D< double > p = Point3D< double >( _samples[i].position );
			Point3D< double > n = Point3D< double >( _samples[i].normal );
			integral += HighPrecisionV( functions[idx]( p ) ) * Length( n );
		}
	else
		for( int i=0 ; i<_triangles.size() ; i++ )
		{
			int idx = _parents[ i ];
			const TriangleIndex& tIndex = _triangles[ i ];
			TriangleIntegrator12< HighPrecisionV > ti( _vertices[ tIndex[0] ] , _vertices[ tIndex[1] ] , _vertices[ tIndex[2] ] );
			for( int s=0 ; s<TriangleIntegrator12< V >::SampleNum ; s++ ) ti.values[s] = HighPrecisionV( functions[ idx ]( ti[s] ) );
			integral += ti.integrate();
		}
	return V( integral );
}
template< class Real , bool Primal >
template< class V , class HighPrecisionV >
V MeshOctree< Real , Primal >::getPreciseCoefficientIntegral( const std::vector< V >& coefficients, int depth ) const
{
	if( Primal ) return _getPrimalPreciseCoefficientIntegral< V , HighPrecisionV >( coefficients , depth );
	else         return _getDualPreciseCoefficientIntegral  < V , HighPrecisionV >( coefficients , depth );
}

template< class Real , bool Primal >
Real MeshOctree< Real , Primal >::getArea( void ) const
{
	double area = 0;
	if( _fromSamples ) for( int i=0 ; i<_samples.size() ; i++ ) area += Length( Point3D< double >( _samples[i].normal ) );
	else
		for( int i=0 ; i<_triangles.size() ; i++ )
		{
			const TriangleIndex& tIndex = _triangles[ i ];
			area += Length( Point3D< Real >::CrossProduct( _vertices[ tIndex[1] ] - _vertices[ tIndex[0] ] , _vertices[ tIndex[2] ] - _vertices[ tIndex[0] ] ) );
		}
	return Real( area );
}
template< class Real , bool Primal >
const OctNode< MeshTreeNodeData< Real > , Real >* MeshOctree< Real , Primal >::getLeafNode( const Point3D< Real >& position ) const
{
	Point3D< Real > myCenter;
	Real myWidth;

	if( Primal )
	{
		myCenter.coords[0]=myCenter.coords[1]=myCenter.coords[2]=Real(1.0);
		myWidth=Real(2.0);
	}
	else
	{
		myCenter.coords[0]=myCenter.coords[1]=myCenter.coords[2]=Real(0.5);
		myWidth=Real(1.0);
	}
	if( position.coords[0]<0 || position.coords[0]>1 ) return NULL;
	if( position.coords[1]<0 || position.coords[1]>1 ) return NULL;
	if( position.coords[2]<0 || position.coords[2]>1 ) return NULL;
	const TreeOctNode* node=&tree;

	while( 1 )
	{
		if( !node->children ) return node;

		int cIndex=TreeOctNode::CornerIndex( myCenter , position );
		node = &node->children[cIndex];
		myWidth/=2;
		if(cIndex&1)	myCenter.coords[0] += myWidth/2;
		else			myCenter.coords[0] -= myWidth/2;
		if(cIndex&2)	myCenter.coords[1] += myWidth/2;
		else			myCenter.coords[1] -= myWidth/2;
		if(cIndex&4)	myCenter.coords[2] += myWidth/2;
		else			myCenter.coords[2] -= myWidth/2;
	}
	return NULL;
}
template< class Real , bool Primal >
OctNode< MeshTreeNodeData< Real > , Real >* MeshOctree< Real , Primal >::getLeafNode( const Point3D< Real >& position , int maxDepth , bool init )
{
	Point3D< Real > myCenter;
	Real myWidth;

	if( Primal )
	{
		myCenter.coords[0]=myCenter.coords[1]=myCenter.coords[2]=Real(1.0);
		myWidth=Real(2.0);
	}
	else
	{
		myCenter.coords[0]=myCenter.coords[1]=myCenter.coords[2]=Real(0.5);
		myWidth=Real(1.0);
	}
	if(position.coords[0]<0 || position.coords[0]>1)	return NULL;
	if(position.coords[1]<0 || position.coords[1]>1)	return NULL;
	if(position.coords[2]<0 || position.coords[2]>1)	return NULL;
	TreeOctNode* node=&tree;
	int d=0;
	while(d<maxDepth)
	{
		if(!node->children)
		{
			if( init )	node->initChildren();
			else		return node;
		}
		int cIndex=TreeOctNode::CornerIndex( myCenter , position );
		node = &node->children[cIndex];
		myWidth/=2;
		if(cIndex&1)	myCenter.coords[0] += myWidth/2;
		else			myCenter.coords[0] -= myWidth/2;
		if(cIndex&2)	myCenter.coords[1] += myWidth/2;
		else			myCenter.coords[1] -= myWidth/2;
		if(cIndex&4)	myCenter.coords[2] += myWidth/2;
		else			myCenter.coords[2] -= myWidth/2;
		d++;
	}
	return node;
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_getPointOffset( const Point3D< Real >& position , int depth , int offset[3] ) const
{
	offset[0] = offset[1] = offset[2] = 0;
	Point3D< Real > myCenter;
	Real myWidth;

	if( Primal )
	{
		myCenter.coords[0]=myCenter.coords[1]=myCenter.coords[2]=Real(1.0);
		myWidth=Real(2.0);
	}
	else
	{
		myCenter.coords[0]=myCenter.coords[1]=myCenter.coords[2]=Real(0.5);
		myWidth=Real(1.0);
	}
	if(position.coords[0]<0 || position.coords[0]>1)	return;
	if(position.coords[1]<0 || position.coords[1]>1)	return;
	if(position.coords[2]<0 || position.coords[2]>1)	return;
	int d=0;
	while( d<depth )
	{
		offset[0] <<= 1;
		offset[1] <<= 1;
		offset[2] <<= 1;
		int cIndex=TreeOctNode::CornerIndex( myCenter , position );
		myWidth/=2;
		if(cIndex&1)	myCenter.coords[0] += myWidth/2 , offset[0] |= 1;
		else			myCenter.coords[0] -= myWidth/2;
		if(cIndex&2)	myCenter.coords[1] += myWidth/2 , offset[1] |= 1;
		else			myCenter.coords[1] -= myWidth/2;
		if(cIndex&4)	myCenter.coords[2] += myWidth/2 , offset[2] |= 1;
		else			myCenter.coords[2] -= myWidth/2;
		d++;
	}
}
template< class Real , bool Primal >
typename MeshOctree< Real , Primal >::TreeOctNode::Neighbors3 MeshOctree< Real , Primal >::_getFinestNodeNeighbors( const Point3D< Real >& point , int depth , int offset[3] )
{
	const TreeOctNode* node = getLeafNode( point );
	if( node->depth()==depth )
	{
		int depth;
		node->depthAndOffset( depth , offset );
		return neighborKey3.getNeighbors( node );
	}
	else
	{
		Real width;
		if( Primal ) width = Real( 2.0 ) / ( 1<<depth );
		else         width = Real( 1.0 ) / ( 1<<depth );
		typename TreeOctNode::Neighbors3 neighbors;
		Point3D< Real > p;
		for( int i=-1 ; i<=1 ; i++ )
		{
			p[0] = point[0] + (width * i);
			for( int j=-1 ; j<=1 ; j++ )
			{
				p[1] = point[1] + (width * j);
				for( int k=-1 ; k<=1 ; k++ )
				{
					p[2] = point[2] + (width * k);
					TreeOctNode* node = getLeafNode( p , depth , false );
					if( node->depth()==depth ) neighbors.neighbors[i+1][j+1][k+1] = node;
					else                       neighbors.neighbors[i+1][j+1][k+1] = NULL;
				}
			}
		}
		_getPointOffset( point , depth , offset );
		return neighbors;
	}
}


template< class Real , bool Primal >
template< class Vertex , class C >
void MeshOctree< Real , Primal >::getValues( const std::vector<Vertex>& points , const std::vector< C >& coefficients , std::vector< C >& pointValues )
{
	if( Primal ) _getPrimalValues( points , coefficients , pointValues );
	else         _getDualValues  ( points , coefficients , pointValues );
}
template< class Real , bool Primal >
template< class Vertex , class V >
void MeshOctree< Real , Primal >::getGradientValues ( const std::vector< Vertex >& points , const std::vector< V >& coefficients , std::vector< Gradient3D< V > >& pointGradients )
{
	if( Primal ) _getPrimalGradientValues( points , coefficients , pointGradients );
	else         _getDualGradientValues  ( points , coefficients , pointGradients );
}
template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::downSample(const Vector< C >& high,int depth , Vector< C >& low ) const
{
	if( Primal ) _primalDownSample( high , depth , low );
	else         _dualDownSample  ( high , depth , low );
}


template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::upSample( const std::vector< C >& low , int depth , std::vector< C >& high ) const
{
	if( Primal ) _primalUpSample( low , depth , high );
	else         _dualUpSample  ( low , depth , high );
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::downSample( const SparseMatrix< Real >& high , const DownSampleMatrix< Real >& highToLow , int depth , SparseMatrix< Real >& low ) const
{
	if( Primal ) _primalDownSample( high , highToLow , depth , low );
	else         _dualDownSample  ( high , highToLow , depth , low );
}

template< class Real , bool Primal >
void MeshOctree< Real , Primal >::downSample( const SparseMatrix<Real>& high , int depth , SparseMatrix<Real>& low ) const
{
	if( Primal ) _primalDownSample( high , depth , low );
	else         _dualDownSample  ( high , depth , low );
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::getDownSampleMatrix( int depth , DownSampleMatrix< Real > &highToLow ) const
{
	if( Primal ) _getPrimalDownSampleMatrix( depth , highToLow );
	else         _getDualDownSampleMatrix  ( depth , highToLow );
}

/////////////////
// Solver Code //
/////////////////

template<class Real>
int DiagonalCompare( const void* v1 , const void* v2 )
{
	const MatrixEntry< Real > *d1 = (MatrixEntry< Real >*)v1;
	const MatrixEntry< Real > *d2 = (MatrixEntry< Real >*)v2;
	Real t = d1->Value - d2->Value;
	if		( t < 0 )	return  1;
	else if	( t > 0 )	return -1;
	else				return  0;
}
template<class MType,class VType> Vector<VType> Multiply(const SparseMatrix<MType>& A,const Vector<VType>& x)
{
	Vector<VType> y = A*x;
	double average=0;
	for(int i=0;i<x.Dimensions();i++)	average+=x[i];
	average/=x.Dimensions();
	for(int i=0;i<x.Dimensions();i++)	y[i]+=average;
	return y;
}

template<class Real,int Dim>
void SolveCG(const SparseMatrix<Real>& M,const Vector<Point<Real,Dim> >& b,int iters,Vector<Point<Real,Dim> >& x)
{
	for(int d=0;d<Dim;d++)
	{
		int dim=M.groups;
		Vector<Real> tempX,tempB;
		tempX.Resize(dim);
		tempB.Resize(dim);
		for(int i=0;i<dim;i++)
		{
			tempX[i]=x[i][d];
			tempB[i]=b[i][d];
		}
		SolveConjugateGradient<Real,Real>(M,tempB,iters,tempX,Multiply<Real,Real>);
		for(int i=0;i<dim;i++)	x[i][d]=tempX[i];
	}
}

// Assuming that the entries are sorted from largest to smallest
template<class Real>
int GetMidPoint(const MatrixEntry< Real >* D,int count)
{
	double sum = 0;
	for(int i=0;i<count;i++) sum += D[i].Value;
	sum /= count;
	int idx;
	for(idx=0;idx<count;idx++) if(D[idx].Value<sum) break;
	return idx-1;
}
template<class Real>
void GetEndPoints( const MatrixEntry< Real>* D , int count , std::vector<int>& endPoints )
{
	endPoints.resize(1);
	endPoints[0] = count;
	for( int i=0 ; ; i++ )
	{
		int idx = GetMidPoint( &D[0],endPoints[i] );
		if( idx <= 1 ) break;
		endPoints.push_back(idx);
	}
}

template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_setDiagonal( const SparseMatrix< Real >& M , Vector< MatrixEntry< Real> >& D , bool verbose ) const
{
	D.Resize(M.groups);
	for(int j=0;j<M.groups;j++)
	{
		double l1Norm=0;
		for(int k=0;k<M.groupSizes[j];k++)
			if(j==M.m_ppElements[j][k].N)
			{
				l1Norm+=fabs(M.m_ppElements[j][k].Value);
				D[j].N=j;
				D[j].Value=M.m_ppElements[j][k].Value;
			}
		// WARNING!!! This is a hack that needs to be fixed.
		if(l1Norm==0)	D[j].N=-1;
	}
}

template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::_solveRestriction(int minDepth,int maxDepth,
										   const std::vector< SparseMatrix< Real >* >& M,				// The matrix
										   const std::vector< Vector< MatrixEntry< Real > > >& D,		// The diagonal entries
										   const std::vector< Vector< C >* >& B,						// The constraints
										   const std::vector<  DownSampleMatrix< Real >* >& DS,			// The restriction operators
										   const std::vector< Vector< C >* >& X,						// The solution
										   size_t iters , float sor , int verbose , double& solveTime , double& vectorSetTime , float iterMultiplier ) const
{
	double tt;
	Vector< C > R;
	float _myIters = float( iters );

	for(int d=maxDepth ; d >= minDepth ; d-- )
	{
		double inRNorm = 0 , outRNorm = 0 , inXNorm = 0 , outXNorm = 0;
		if(d<maxDepth)	X[d]->Resize(B[d]->Dimensions());
		if(verbose & FULL_VERBOSE)
		{
			for( size_t j=0 ; j<X[d]->Dimensions() ; j++ ) inXNorm+=C::SquareNorm( (*X[d])[j] );
			R=(*B[d])-(*M[d])*(*X[d]);
			for( size_t j=0 ; j<R.Dimensions() ; j++) inRNorm+=C::SquareNorm(R[j]);
		}
		if( _myIters>0 )
		{
			tt=Time();
			SparseMatrix<Real>::SolveGaussSeidel( *M[d] , D[d] , *B[d] , (size_t) ceil( _myIters ) , *X[d] , 0 , X[d]->Dimensions() , sor , false );
			solveTime+=Time()-tt;
		}

		tt=Time();
		// Compute the residual
		R=(*B[d])-(*M[d])*(*X[d]);

		// Down-sample the residual
		if(d>minDepth)
		{
			B[d-1]->Resize( _dimension[d-1] );
			DS[d]->Multiply( R , (*B[d-1]) );
		}
		vectorSetTime+=Time()-tt;

		if( verbose&FULL_VERBOSE && iters>0 )
			for( size_t j=0 ; j<R.Dimensions() ; j++ ) outRNorm+=C::SquareNorm(R[j]) , outXNorm += C::SquareNorm((*X[d])[j]);
		if( verbose&FULL_VERBOSE )
		{
			for( int dd=0 ; dd<d ; dd++ ) printf(" ");
			if( iters>0 ) printf( "R[%d][%d]:  %f -> %f     (%f -> %f)\n" , d , int( ceil(_myIters) ) , sqrt(inRNorm) , sqrt(outRNorm) , sqrt(inXNorm) , sqrt(outXNorm) );
			else		  printf( "B[%d]:  %f     (%f)\n" , d , sqrt(inRNorm) , sqrt(inXNorm) );
		}
		_myIters *= iterMultiplier;
		if( iters && !_myIters ) _myIters = 1;
	}
}

template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::_solveProlongation(int minDepth,int maxDepth,
										  const std::vector<  SparseMatrix< Real >* >& M,
										  const std::vector< Vector< MatrixEntry< Real > > >& D,
										  const std::vector< Vector< C >* >& B,
										  const std::vector<  DownSampleMatrix< Real >* >& DS,
										  const std::vector< Vector< C >* >& X,
										  size_t iters , float sor , int verbose, double& solveTime, double& vectorSetTime , float iterMultiplier ) const
{
	double tt;
	Vector< C > R;
	float _myIters;

	for( int d = minDepth ; d <= maxDepth ; d++ )
	{
		_myIters = float( iters );
		for( int dd = d ; dd<maxDepth ; dd++ ) _myIters *= iterMultiplier;

		double inRNorm = 0 , outRNorm = 0 , inXNorm = 0 , outXNorm = 0;
		if( d>0 )
		{
			tt=Time();
			// Up-sample the coarser solution and add it back to the earlier finer solution
			DS[d]->MultiplyTranspose( *X[d-1] , *X[d] , false );
			vectorSetTime+=Time()-tt;
			if(verbose&FULL_VERBOSE)
			{
				R=(*B[d])-(*M[d])*(*X[d]);
				for( size_t j=0 ; j<R.Dimensions() ; j++ ) inRNorm += C::SquareNorm(R[j]) , inXNorm += C::SquareNorm( (*X[d])[j] );
			}
		}
		tt=Time();
		SparseMatrix<Real>::SolveGaussSeidel( *M[d] , D[d] , *B[d] , (size_t)ceil( _myIters ) , *X[d] , 0 , X[d]->Dimensions() , sor , false );
	
		solveTime+=Time()-tt;
		if(verbose&FULL_VERBOSE)
		{
			R=(*B[d])-(*M[d])*(*X[d]);
			for( size_t j=0 ; j<R.Dimensions() ; j++ )		outRNorm += C::SquareNorm(R[j]);
			for( size_t j=0 ; j<X[d]->Dimensions() ; j++ )	outXNorm += C::SquareNorm((*X[d])[j]);
			for( int dd=0 ; dd<d ; dd++ ) printf(" ");
			printf("P[%d][%d]:  %f -> %f     (%f -> %f)\n" , d , int( ceil(_myIters) ) , sqrt(inRNorm) , sqrt(outRNorm) , sqrt(inXNorm) , sqrt(outXNorm) );
		}
	}
}
template< class C , class Real >
Vector< C > FullMultiply( const SparseMatrix< Real >& A,const Vector< C >& x ) { return A*x; }

template< class Real , bool Primal >
template< class C >
int MeshOctree< Real , Primal >::_solveConjugateGradient( const SparseMatrix< Real >& A , const Vector< C >& b , const size_t& iters , Vector< C >& x , size_t dim ) const
{
	double eps=1e-16;
	Vector< C > r = b - A*x;
	Vector< C > d = r;
	Real delta_new , delta_0;
	delta_new = 0;
	for( size_t i=0 ; i<dim ; i++ ) delta_new += C::SquareNorm( r[i] );
	delta_0 = delta_new;
	if( delta_new<eps ) return 0;
	int ii;
	for( ii=0; ii<iters && delta_new>eps*delta_0 ; ii++ )
	{
		Vector< C > q = A*d;
		double dDotQ = 0 , alpha = 0;
		for( size_t i=0 ; i<dim ; i++ ) dDotQ += C::Dot( d[i] , q[i] );
		alpha = delta_new / dDotQ;
		for( size_t i=0 ; i<dim ; i++ ) x[i] += d[i] * Real( alpha );
		if( !(ii%50) )	r = b - A*x;
		else			for( size_t i=0 ; i<dim ; i++ ) r[i] = r[i] - q[i] * Real( alpha );

		double delta_old = delta_new , beta;
		delta_new = 0;
		for( size_t i=0 ; i<dim ; i++ ) delta_new += C::SquareNorm( r[i] );
		beta = delta_new / delta_old;
		for( size_t i=0 ; i<dim ; i++ ) d[i] = r[i] + d[i] * Real( beta );
	}
	return ii;
}
template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::solveLinearSystem( SparseMatrix<Real>& m , Vector< C >& b , Vector< C >& x,
										    int minDepth , int iters , int vCycles , float sor , bool cascadic , int verbose , bool vCycle , bool solveBaseCG , float iterMultiplier ) const
{
	double normB , normR , normX;
	double vectorSetTime , matrixSetTime , solveTime;
	solveLinearSystem( m , b , x , minDepth , iters , vCycles , sor , cascadic , verbose , vCycle, solveBaseCG , normB , normR , normX , vectorSetTime , matrixSetTime , solveTime , iterMultiplier );
}
template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::solveLinearSystem( SparseMatrix<Real>& m , Vector< C >& b , Vector< C >& x,
										    int minDepth , int iters , int vCycles , float sor , bool cascadic , int verbose , bool vCycle , bool solveBaseCG ,
											double& normB , double& normR , double& normX , double& vectorSetTime , double& matrixSetTime , double& solveTime , 
											float iterMultiplier ) const
{
	double tt;
	int rIters = cascadic ? 0 : iters;
	int pIters = iters;

	solveTime = vectorSetTime = matrixSetTime = 0;

	// The systems: M[d]*X[d]=B[d] for all depths d
	std::vector< SparseMatrix< Real >* > M;
	std::vector< SparseMatrix< Real > > _M;
	std::vector< DownSampleMatrix< Real >* > DS;
	std::vector< DownSampleMatrix< Real > > _DS;
	std::vector< Vector< C >* > X , B;
	std::vector< Vector< C > > _X , _B;
	// For Gauss-Seidel solvers, we store the diagonal elements of each matrix separately
	std::vector< Vector< MatrixEntry< Real > > > D;

	// The maximum depth of the tree
	int maxDepth=tree.maxDepth();
	if( minDepth>maxDepth ) minDepth = maxDepth;

	_M.resize( maxDepth );
	_X.resize( maxDepth );
	_B.resize( maxDepth );
	M.resize( maxDepth+1 );
	X.resize( maxDepth+1 );
	B.resize( maxDepth+1 );
	_DS.resize( maxDepth );
	DS.resize( maxDepth+1 );

	M[maxDepth] = &m;
	X[maxDepth] = &x;
	B[maxDepth] = &b;

	for( int d = 0 ; d < maxDepth ; d++ ) M[d]=&_M[d] , X[d]=&_X[d] , B[d]=&_B[d];
	for( int d = 0 ; d <= maxDepth ; d++ ) X[d]->Resize( _dimension[d] );

	tt=Time();
	// Using the finest depth's Laplacian, compute the lower depth's Laplacians
	_DS.resize(maxDepth);
	DS.resize(maxDepth+1);
	DS[0] = NULL;
	for( int d = 1; d<= maxDepth ; d++ ) getDownSampleMatrix( d , _DS[d-1] ) , DS[d] = &_DS[d-1];
	for( int d = maxDepth-1 ; d >= 0 ; d-- ) downSample(*M[d+1] , *DS[d+1] ,  d+1 , *M[d] );

	D.resize( maxDepth+1 );
	for( int d = 0 ; d <= maxDepth ; d++ ) _setDiagonal( *M[d] , D[d] , verbose&VERBOSE );

	// Get the constraint vector at the finest resolution
	matrixSetTime+=Time()-tt;

	for( int vc=0 ; vc<vCycles ; vc++ )
	{
		// Perform multiple V/W-cycle of the multi-grid solver
		if(vCycle)
		{
			_solveRestriction ( minDepth , maxDepth , M , D , B , DS , X , rIters , sor , verbose , solveTime , vectorSetTime , iterMultiplier );
			if( solveBaseCG ) _solveConjugateGradient( *M[minDepth] , *B[minDepth] , 6*int( sqrt( M[minDepth]->groups+1.0 ) ) , *X[minDepth] , X[minDepth]->Dimensions() );
			_solveProlongation( minDepth , maxDepth , M , D , B , DS , X , pIters , sor ,  verbose , solveTime , vectorSetTime , iterMultiplier );
		}
		else
		{
			// Down-sample all the way down
			_solveRestriction( minDepth , maxDepth , M , D , B , DS , X , 0 , sor ,  verbose , solveTime , vectorSetTime , iterMultiplier );

			for( int d = minDepth ; d <= maxDepth ; d++ )
			{
				_solveRestriction ( minDepth , d , M , D , B , DS , X , rIters , sor ,  verbose , solveTime , vectorSetTime , iterMultiplier );
				if( solveBaseCG ) _solveConjugateGradient( *M[minDepth] , *B[minDepth] , 6*int( sqrt( M[minDepth]->groups+1.0 ) ) , *X[minDepth] , X[minDepth]->Dimensions() );
				_solveProlongation( minDepth , d , M , D , B , DS , X , pIters , sor ,  verbose , solveTime , vectorSetTime , iterMultiplier );
				if( d<maxDepth ) DS[d+1]->MultiplyTranspose( *X[d] , *X[d+1] , false );
			}
		}
	}
	MemoryUsage();
	if( verbose & VERBOSE )
	{
		Vector< C > r = m*x - b;
		normB = normR = normX = 0;
		for( size_t j=0 ; j<b.Dimensions() ; j++ )
		{
			normB += C::SquareNorm(b[j]);
			normR += C::SquareNorm(r[j]);
			normX += C::SquareNorm(x[j]);
		}
	}

}

template< class Real >
void SetTangents( const Point3D< Real >& normal , Point3D< Real >& t1 , Point3D< Real >& t2 )
{
	double nLength = sqrt( Point3D< Real >::SquareNorm( normal ) );
	Point3D< Real > p , n = normal / Real(nLength);
	p[0] = 1 , p[1] = 0 , p[2] = 0;
	p = Point3D< Real >::CrossProduct( n , p );
	if( Point3D< Real >::SquareNorm( p ) < 0.00001 )
	{
		p[0] = 0 , p[1] = 1 , p[2] = 0;
		p = Point3D< Real >::CrossProduct( n , p );
	}
	t1  = p / Real(sqrt( Point3D< Real >::SquareNorm( p ) ));
	t2  = Point3D< Real >::CrossProduct( t1 , n );
	t2 /= sqrt( Point3D< Real >::SquareNorm( t2 ) );
}
template< class Real >
void SetCurvatureMatrix( const Point3D< Real >& normal , const Gradient3D< Point3D< Real > >& normalGradient ,
						Matrix< Real , 3 , 3 >& c , Point3D< Real >& t1 , Point3D< Real > &t2 )
{
	Point3D< Real > column;
	double nLength = sqrt( Point3D< Real >::SquareNorm( normal ) );

	SetTangents( normal , t1 , t2 );

	for( int j=0 ; j<3 ; j++ )
	{
		for( int k=0 ; k<3 ; k++ ) column[k] = normalGradient[j][k];
		double dot = Point3D< Real >::Dot( column , normal ) / (nLength*nLength);
		for( int k=0 ; k<3 ; k++ ) c(j,k) = ( normalGradient[j][k] - dot * normal[k] ) / nLength;
	}
}


template< class Real , bool Primal > int MeshOctree< Real , Primal >::vertexCount( void ) const { return _vertices.size(); }
template< class Real , bool Primal > const Point3D< Real >&  MeshOctree< Real , Primal >::vertex( int idx ) const { return _vertices[idx]; }

template< class Real , bool Primal > int MeshOctree< Real , Primal >::triangleCount( void ) const { return _triangles.size(); }
template< class Real , bool Primal > const TriangleIndex& MeshOctree< Real , Primal >::triangle( int idx ) const { return _triangles[idx]; }
template< class Real , bool Primal > int MeshOctree< Real , Primal >::triangleParent( int idx ) const { return _parents[idx]; }

