/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
template< class Real >
void Spline< Real >::Weights( Real t, Real& w1 , Real& w2 , Real& w3 , Real& w4 , int type )
{
	Real t2,t3;
	switch( type ){
	case NEAREST:
		w1=w2=w3=w4=0;
		if(t<0.5){w2=1;}
		else{w3=1;}
		break;
	case LINEAR:
		w2=1.f-t;
		w3=t;
		w1=w4=0;
		break;
	case QUADRATIC:
		t2=t*t;
		w1=( t2-t    )/4;
		w2=(-t2-3*t+4)/4;
		w3=(-t2+5*t  )/4;
		w4=( t2-  t  )/4;
		break;
	case CUBIC:
		t2=t*t;
		t3=t2*t;
		w1=(-  t3+3*t2-2*t  )/6;
		w2=( 3*t3-6*t2-3*t+6)/6;
		w3=(-3*t3+3*t2+6*t  )/6;
		w4=(   t3-       t  )/6;
		break;
	case CATMULL_ROM:
		t2=t*t;
		t3=t2*t;
		w1=(-  t3+2*t2-  t  )/2;
		w2=( 3*t3-5*t2    +2)/2;
		w3=(-3*t3+4*t2+  t  )/2;
		w4=(   t3-  t2      )/2;
		break;
	case CUBIC_B_SPLINE:
		t2=t*t;
		t3=t2*t;
		w1=(-  t3+3*t2-3*t+1)/6;
		w2=( 3*t3-6*t2    +4)/6;
		w3=(-3*t3+3*t2+3*t+1)/6;
		w4=(   t3           )/6;
		break;
	}
}
template< class Real >
void Spline< Real >::DWeights( Real t , Real& w1 , Real& w2 , Real& w3 , Real& w4 , int type )
{
	Real t2;
	switch(type){
	case NEAREST:
		w1=w2=w3=w4=0;
		break;
	case LINEAR:
		w2=-1;
		w3= 1;
		w1=w4=0;
		break;
	case QUADRATIC:
		w1=( 2*t-1)/4;
		w2=(-2*t-3)/4;
		w3=(-2*t+5)/4;
		w4=( 2*t-1)/4;
		break;
	case CUBIC:
		t2=t*t;
		w1=(-3*t2+ 6*t-2)/6;
		w2=( 9*t2-12*t-3)/6;
		w3=(-9*t2+ 6*t+6)/6;
		w4=( 3*t2-     1)/6;
		break;
	case CATMULL_ROM:
		t2=t*t;
		w1=(-3*t2+ 4*t-1)/2;
		w2=( 9*t2-10*t  )/2;
		w3=(-9*t2+ 8*t+1)/2;
		w4=( 3*t2- 2*t  )/2;
		break;
	case CUBIC_B_SPLINE:
		t2=t*t;
		w1=(-3*t2+ 6*t-3)/6;
		w2=( 9*t2-12*t  )/6;
		w3=(-9*t2+ 6*t+3)/6;
		w4=( 3*t2       )/6;
		break;
	}
}
template< class Real >
void Spline< Real >::DDWeights( Real t , Real& w1 , Real& w2 , Real& w3 , Real& w4 , int type )
{
	switch(type){
	case NEAREST:
		w1=w2=w3=w4=0;
		break;
	case LINEAR:
		w1=w2=w3=w4=0;
		break;
	case QUADRATIC:
		w1=w4= 0.5f;
		w2=w3=-0.5f;
		break;
	case CUBIC:
		w1=(- 6*t+ 6)/6;
		w2=( 18*t-12)/6;
		w3=(-18*t+ 6)/6;
		w4=(  6*t   )/6;
		break;
	case CATMULL_ROM:
		w1=(- 6*t+ 4)/2;
		w2=( 18*t-10)/2;
		w3=(-18*t+ 8)/2;
		w4=(  6*t- 2)/2;
		break;
	case CUBIC_B_SPLINE:
		w1=(- 6*t+ 6)/6;
		w2=( 18*t-12)/6;
		w3=(-18*t+ 6)/6;
		w4=(  6*t   )/6;
		break;
	}
}
template< class Real >
Real NonUniformNonRationalBSpline< Real >::GetWeight( Real t , int k , int d , Real* knots )
{
	Real t1=0,t2=0;
	if(d==1)
		if(knots[k]<=t && t<knots[k+1]){return 1;}
		else{return 0;}

	if(knots[k+d-1]!=knots[k  ]) t1=-(knots[k  ]-t)/(knots[k+d-1]-knots[k  ])*GetWeight(t,k  ,d-1,knots);
	if(knots[k+d  ]!=knots[k+1]) t2= (knots[k+d]-t)/(knots[k+d  ]-knots[k+1])*GetWeight(t,k+1,d-1,knots);
	return t1+t2;
}
template< class Real >
Real UniformNonRationalBSpline< Real >::_GetWeight( Real t , int k , int d )
{
	// knots[k] = -0.5 + k
	Real t1 = 0 , t2 = 0;
	if( d==1 )
	{
		if( (-0.5+k)<=t && t<(0.5+k) )	return 1;
		else							return 0;
	}
	t1 = -((-0.5+k  )-t)/(d-1)*_GetWeight( t , k   , d-1 );
	t2 =  ((-0.5+k+d)-t)/(d-1)*_GetWeight( t , k+1 , d-1 );
	return t1+t2;
}
template< class Real >
Real UniformNonRationalBSpline< Real >::GetWeight( Real t , int d )
{
	return _GetWeight(t+0.5*(d-1),0,d);
}
