/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#ifndef SPARSE_MATRIX_INTERFACE_INCLUDED
#define SPARSE_MATRIX_INTERFACE_INCLUDED

#define FORCE_TWO_BYTE_ALIGNMENT 1
#include "Array.h"

#if FORCE_TWO_BYTE_ALIGNMENT
#pragma pack(push)
#pragma pack(2)
#endif // FORCE_TWO_BYTE_ALIGNMENT
template< class T , class IndexType >
struct MatrixEntry
{
	MatrixEntry( void )       { N =-1 , Value = 0; }
	MatrixEntry( int i )      { N = i , Value = 0; }
	MatrixEntry( int n , T v ){ N = n , Value = v; }
	IndexType N;
	T Value;
};
#if FORCE_TWO_BYTE_ALIGNMENT
#pragma pack(pop)
#endif // FORCE_TWO_BYTE_ALIGNMENT

/************************************************
 * A class supporting a parallel multigrid solver
 * Implementation based on the safe-zone ideas
 * described in Ofir et al. (ToG '08)
 ************************************************/
template< class T2 >
class ParallelSolution
{
	template< class T , class const_iterator > friend class SparseMatrixInterface;
	int _iters , _threads , _sliceDependence , _sliceCount;
	const int *_entriesPerSlice , *_slicesPerThread;
	std::vector< Pointer( T2 ) > _pSolution;
	std::vector< int > _startEntry;
	std::vector< std::pair< int , int > > _sliceBounds , _skirtBounds;
	void _clearSkirts( void );
	void _mergeSkirts( void );
	void _synchronize( void );
public:
	ParallelSolution( void );
	ParallelSolution( int iters , const std::vector< int >& entriesPerSlice , const std::vector< int >& slicesPerThread , int sliceDependence );
	~ParallelSolution( void );
	void init( int iters , const std::vector< int >& entriesPerSlice , const std::vector< int >& slicesPerThread , int sliceDependence );
	void SetFromArray( ConstPointer( T2 )  solution );
	void SetFromArray( const Vector< T2 >& solution ) { SetFromArray( solution+0 ); }
	void SetToArray  ( Pointer( T2 ) solution ) const;
	void SetToArray  ( Vector< T2 >& solution ) const { SetToArray( solution+0 ); }
	void clear( void );
	int threads( void ) const { return _threads; }
	int sliceStart( int t ) const { if( t<0 || t>=_threads ) return 0 ; return _sliceBounds[t].first;  }
	int sliceEnd  ( int t ) const { if( t<0 || t>=_threads ) return 0 ; return _sliceBounds[t].second; }
	int sliceSize( int t ) const { if( t<0 || t>=_threads ) return 0 ; return _sliceBounds[t].second - _sliceBounds[t].first; }
	int size( int t ) const { if( t<0 || t>=_threads ) return 0 ; return _startEntry[ _sliceBounds[t].second-1 ] + _entriesPerSlice[ _sliceBounds[t].second-1 ] - _startEntry[ _sliceBounds[t].first ]; }
	int size( void ) const { int sz=0 ; for( int i=0 ; i<_threads ; i++ ) sz += size( i ) ; return sz; }
	int offset( int t ) const { if( t<0 || t>_threads ) return 0 ; return _startEntry[ _sliceBounds[t].first ]; }
	void bounds( int t , int& start , int& end ) const { if( t<0 || t>=_threads ) start = end =0 ; else start = _startEntry[_skirtBounds[t].first] - _startEntry[_sliceBounds[t].first] , end = _startEntry[ _skirtBounds[t].second-1 ] + _entriesPerSlice[ _skirtBounds[t].second-1 ] - _startEntry[_sliceBounds[t].first]; }
	ConstPointer( T2 ) operator[] ( int t ) const { if( t<0 || t>=_threads ) return NullPointer< T2 >( ) ; return _pSolution[t]; }
	Pointer( T2 ) operator[] ( int t ) { if( t<0 || t>=_threads ) return NullPointer< T2 >( ) ; return _pSolution[t]; }
};

enum
{
	MULTIPLY_ADD ,
	MULTIPLY_SUBTRACT ,
	MULTIPLY_CLEAR
};

template< class T , class const_iterator > class SparseMatrixInterface
{
public:
	virtual const_iterator begin( int row ) const = 0;
	virtual const_iterator end  ( int row ) const = 0;
	virtual size_t Rows   ( void )          const = 0;
	virtual size_t RowSize( size_t idx )    const = 0;

	size_t Entries( void ) const;

	double SquareNorm( void ) const;
	double SquareASymmetricNorm( void ) const;
	double SquareASymmetricNorm( int& idx1 , int& idx2 ) const;

	template< class T2 > void Multiply           ( ConstPointer( T2 ) In , Pointer( T2 ) Out ) const;
	template< class T2 > void Multiply           (      Pointer( T2 ) In , Pointer( T2 ) Out ) const { return Multiply( ( ConstPointer( T2 ) ) In , Out ); }
	template< class T2 > void Multiply           ( const Vector< T2 >& In , Vector< T2 >& Out ) const { Multiply( In+0 , Out+0 ); }
	template< class T2 > void MultiplyParallel   ( ConstPointer( T2 )  In , Pointer( T2 ) Out , int threads , int multiplyFlag=MULTIPLY_ADD ) const;
	template< class T2 > void MultiplyParallel   (      Pointer( T2 )  In , Pointer( T2 ) Out , int threads , int multiplyFlag=MULTIPLY_ADD ) const { MultiplyParallel( ( ConstPointer(T2) )( In ) , Out , threads , multiplyFlag ); }
	template< class T2 > void MultiplyParallel   ( const Vector< T2 >& In , Vector< T2 >& Out , int threads , int multiplyFlag=MULTIPLY_ADD ) const { MultiplyParallel( In+0 , Out+0 , threads , multiplyFlag ); }

	template< class T2 > void MultiplyTranspose  ( ConstPointer( T2 ) In , Pointer( T2 ) Out ) const;
	template< class T2 > void MultiplyTranspose  (      Pointer( T2 ) In , Pointer( T2 ) Out ) const { return MultiplyTranspose( ( ConstPointer( T2 ) ) In , Out ); }
	template< class T2 > void MultiplyTranspose  ( const Vector< T2 >& In , Vector< T2 >& Out ) const { MultiplyTranspose( In+0 , Out+0 ); }
	template< class T2 > void BMinusMX           ( ConstPointer( T2 ) B , ConstPointer( T2 ) X , Pointer( T2 ) out ) const;
	template< class T2 > void BMinusMX           (      Pointer( T2 ) B , ConstPointer( T2 ) X , Pointer( T2 ) out ) const { return BMinusMX( ( ConstPointer( T2 ) ) B ,                        X , out ); }
	template< class T2 > void BMinusMX           ( ConstPointer( T2 ) B ,      Pointer( T2 ) X , Pointer( T2 ) out ) const { return BMinusMX(                        B , ( ConstPointer( T2 ) ) X , out ); }
	template< class T2 > void BMinusMX           (      Pointer( T2 ) B ,      Pointer( T2 ) X , Pointer( T2 ) out ) const { return BMinusMX( ( ConstPointer( T2 ) ) B , ( ConstPointer( T2 ) ) X , out ); }
	template< class T2 > void BMinusMX           ( const Vector< T2 >& B , const Vector< T2 >& X , Vector< T2 >& out ) const { BMinusMX( B+0 , X+0 , out+0 ); }



	template< class T2 , class T3 , bool unitDiagonal , bool StrippedDiagonal , bool UseSOR > void SolveGaussSeidel           ( ConstPointer( T3 ) Diagonal , ConstPointer( T2 ) b , int iters , Pointer( T2 ) Solution , T sor ) const;
	template< class T2 , class T3 , bool unitDiagonal , bool StrippedDiagonal , bool UseSOR > void SolveGaussSeidelAndResidual( ConstPointer( T3 ) Diagonal , ConstPointer( T2 ) b , int iters , Pointer( T2 ) Solution , Pointer( T2 ) Residual , T sor ) const;

	template< class T2 , class T3 , bool UseSOR > void SolveGaussSeidel( ConstPointer( T3 ) Diagonal , ConstPointer( T2 ) b , int iters , Pointer( T2 ) Solution , T sor , bool unitDiagonal , bool strippedDiagonal ) const
	{
		if( strippedDiagonal )
			if( unitDiagonal ) SolveGaussSeidel< T2 , T3 , true  , true  , UseSOR >( Diagonal , b , iters , Solution , sor );
			else               SolveGaussSeidel< T2 , T3 , false , true  , UseSOR >( Diagonal , b , iters , Solution , sor );
		else
			if( unitDiagonal ) SolveGaussSeidel< T2 , T3 , true  , false , UseSOR >( Diagonal , b , iters , Solution , sor );
			else               SolveGaussSeidel< T2 , T3 , false , false , UseSOR >( Diagonal , b , iters , Solution , sor );
	}
	template< class T2 , class T3 , bool UseSOR > void SolveGaussSeidelAndResidual( ConstPointer( T3 ) Diagonal , ConstPointer( T2 ) b , int iters , Pointer( T2 ) Solution , Pointer( T2 ) Residual , T sor , bool unitDiagonal , bool strippedDiagonal ) const
	{
		if( strippedDiagonal )
			if( unitDiagonal ) SolveGaussSeidelAndResidual< T2 , T3 , true  , true  , UseSOR >( Diagonal , b , iters , Solution , Residual , sor );
			else               SolveGaussSeidelAndResidual< T2 , T3 , false , true  , UseSOR >( Diagonal , b , iters , Solution , Residual , sor );
		else
			if( unitDiagonal ) SolveGaussSeidelAndResidual< T2 , T3 , true  , false , UseSOR >( Diagonal , b , iters , Solution , Residual , sor );
			else               SolveGaussSeidelAndResidual< T2 , T3 , false , false , UseSOR >( Diagonal , b , iters , Solution , Residual , sor );
	}
	template< class T2 , class T3 , bool UseSOR > void SolveGaussSeidel           ( const Vector< T3 >& Diagonal , const Vector< T2 >& b , int iters , Vector< T2 >& Solution ,                          T sor , bool unitDiagonal , bool strippedDiagonal ) const { SolveGaussSeidel           < T2 , T3 , UseSOR >( Diagonal+0 , b+0 , iters , Solution+0 ,              sor , unitDiagonal , strippedDiagonal ); }
	template< class T2 , class T3 , bool UseSOR > void SolveGaussSeidelAndResidual( const Vector< T3 >& Diagonal , const Vector< T2 >& b , int iters , Vector< T2 >& Solution , Vector< T2 >& Residual , T sor , bool unitDiagonal , bool strippedDiagonal ) const { SolveGaussSeidelAndResidual< T2 , T3 , UseSOR >( Diagonal+0 , b+0 , iters , Solution+0 , Residual+0 , sor , unitDiagonal , strippedDiagonal ); }

	template< class T2 , class T3 >
	void SolveGaussSeidel( const Vector< T3 >& Diagonal , const Vector< T2 >& b , int iters , Vector< T2 >& Solution , T sor , bool unitDiagonal , bool strippedDiagonal ) const
	{
		if( sor!=T(1.) ) SolveGaussSeidel< T2 , T3 , true  >( Diagonal , b , iters , Solution , sor , unitDiagonal , strippedDiagonal );
		else             SolveGaussSeidel< T2 , T3 , false >( Diagonal , b , iters , Solution , sor , unitDiagonal , strippedDiagonal );
	}
	template< class T2 , class T3 >
	void SolveGaussSeidelAndResidual( const Vector< T3 >& Diagonal , const Vector< T2 >& b , int iters , Vector< T2 >& Solution , Vector< T2 >& Residual , T sor , bool unitDiagonal , bool strippedDiagonal ) const
	{
		if( sor!=T(1.) ) SolveGaussSeidelAndResidual< T2 , T3 , true  >( Diagonal , b , iters , Solution , Residual , sor , unitDiagonal , strippedDiagonal );
		else             SolveGaussSeidelAndResidual< T2 , T3 , false >( Diagonal , b , iters , Solution , Residual , sor , unitDiagonal , strippedDiagonal );
	}


	template< class T2 > void MultiplyParallel( ParallelSolution< T2 >* in , ParallelSolution< T2 >* out , int multiplyFlag=MULTIPLY_ADD ) const;
	template< class T2 > void MultiplyParallel( ConstPointer( T2 ) in , ParallelSolution< T2 >* out , int multiplyFlag=MULTIPLY_ADD ) const;
	template< class T2 > void MultiplyParallel( const Vector< T2 >& in , ParallelSolution< T2 >* out , int multiplyFlag=MULTIPLY_ADD ) const { MultiplyParallel( in+0 , out , multiplyFlag); }
	template< class T2 > void MultiplyTransposeParallel( const ParallelSolution< T2 >* in , ParallelSolution< T2 >* out ) const;

	template< class T2 , class T3 , bool UnitDiagonal , bool StrippedDiagonal , bool UseSOR >
	void SolveGaussSeidelParallel( ConstPointer( T3 ) Diagonal , ConstPointer( T2 ) b , int iters , ParallelSolution< T2 >* Solution , T sor ) const;

	template< class T2 , class T3 , bool UnitDiagonal , bool StrippedDiagonal , bool UseSOR >
	void SolveGaussSeidelAndResidualParallel( ConstPointer( T3 ) Diagonal , ConstPointer( T2 ) b , int iters , ParallelSolution< T2 >* Solution , Pointer( T2 ) Residual , T sor ) const;

	template< class T2 , class T3 , bool UseSOR >
	void SolveGaussSeidelParallel( ConstPointer( T3 ) Diagonal , ConstPointer( T2 ) b , int iters , ParallelSolution< T2 >* Solution , T sor ,	bool unitDiagonal , bool strippedDiagonal ) const
	{
		if( strippedDiagonal )
			if( unitDiagonal ) SolveGaussSeidelParallel< T2 , T3 , true  , true  , UseSOR >( Diagonal , b , iters , Solution , sor );
			else               SolveGaussSeidelParallel< T2 , T3 , false , true  , UseSOR >( Diagonal , b , iters , Solution , sor );
		else
			if( unitDiagonal ) SolveGaussSeidelParallel< T2 , T3 , true  , false , UseSOR >( Diagonal , b , iters , Solution , sor );
			else               SolveGaussSeidelParallel< T2 , T3 , false , false , UseSOR >( Diagonal , b , iters , Solution , sor );
	}
	template< class T2 , class T3 , bool UseSOR >
	void SolveGaussSeidelAndResidualParallel( ConstPointer( T3 ) Diagonal , ConstPointer( T2 ) b , int iters , ParallelSolution< T2 >* Solution , Pointer( T2 ) Residual ,  T sor , bool unitDiagonal , bool strippedDiagonal ) const
	{
		if( strippedDiagonal )
			if( unitDiagonal ) SolveGaussSeidelAndResidualParallel< T2 , T3 ,true  , true  , UseSOR >( Diagonal , b , iters , Solution , Residual , sor );
			else               SolveGaussSeidelAndResidualParallel< T2 , T3 ,false , true  , UseSOR >( Diagonal , b , iters , Solution , Residual , sor );
		else
			if( unitDiagonal ) SolveGaussSeidelAndResidualParallel< T2 , T3 ,true  , false , UseSOR >( Diagonal , b , iters , Solution , Residual , sor );
			else               SolveGaussSeidelAndResidualParallel< T2 , T3 ,false , false , UseSOR >( Diagonal , b , iters , Solution , Residual , sor );
	}
	template< class T2 , class T3 , bool UseSOR > void SolveGaussSeidelParallel           ( const Vector< T3 >& Diagonal , const Vector< T2 >& b , int iters , ParallelSolution< T2 >* Solution ,                          T sor , bool unitDiagonal , bool strippedDiagonal ) const { SolveGaussSeidelParallel           < T2 , T3 , UseSOR >( Diagonal+0 , b+0 , iters , Solution ,              sor , unitDiagonal , strippedDiagonal ); }
	template< class T2 , class T3 , bool UseSOR > void SolveGaussSeidelAndResidualParallel( const Vector< T3 >& Diagonal , const Vector< T2 >& b , int iters , ParallelSolution< T2 >* Solution , Vector< T2 >& Residual , T sor , bool unitDiagonal , bool strippedDiagonal ) const { SolveGaussSeidelAndResidualParallel< T2 , T3 , UseSOR >( Diagonal+0 , b+0 , iters , Solution , Residual+0 , sor , unitDiagonal , strippedDiagonal ); }

	template< class T2 , class T3 >
	void SolveGaussSeidelParallel( const Vector< T3 >& Diagonal , const Vector< T2 >& b , int iters , ParallelSolution< T2 >* Solution , T sor , bool unitDiagonal , bool strippedDiagonal ) const
	{
		if( sor!=T(1.) ) SolveGaussSeidelParallel< T2 , T3 , true  >( Diagonal , b , iters , Solution , sor , unitDiagonal , strippedDiagonal );
		else             SolveGaussSeidelParallel< T2 , T3 , false >( Diagonal , b , iters , Solution , sor , unitDiagonal , strippedDiagonal );
	}
	template< class T2 , class T3 >
	void SolveGaussSeidelAndResidualParallel( const Vector< T3 >& Diagonal , const Vector< T2 >& b , int iters , ParallelSolution< T2 >* Solution , Vector< T2 >& Residual , T sor , bool unitDiagonal , bool strippedDiagonal ) const
	{
		if( sor!=T(1.) ) SolveGaussSeidelAndResidualParallel< T2 , T3 , true  >( Diagonal , b , iters , Solution , Residual , sor , unitDiagonal , strippedDiagonal );
		else             SolveGaussSeidelAndResidualParallel< T2 , T3 , false >( Diagonal , b , iters , Solution , Residual , sor , unitDiagonal , strippedDiagonal );
	}
};
#include "SparseMatrixInterface.inl"

#endif // SPARSE_MATRIX_INTERFACE_INCLUDED
