/* -*- C++ -*-
Copyright (c) 2006, Michael Kazhdan and Matthew Bolitho
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

#include "LaplacianMatrix1D.h"
#include "TriangleIntegrals.h"
#include "LineIntegrals.h"

#if FAST_QUADRATURE
typedef LineIntegrator3          PrimalLineIntegrator;
typedef TriangleIntegratorMedian PrimalTriangleIntegrator;
typedef LineIntegrator9          DualLineIntegrator;
//typedef TriangleIntegrator10     DualTriangleIntegrator;
typedef TriangleIntegrator4     DualTriangleIntegrator;
#else // !FAST_QUADRATURE
typedef LineIntegrator5			PrimalLineIntegrator;
typedef TriangleIntegrator6		PrimalTriangleIntegrator;
typedef LineIntegrator9			DualLineIntegrator;
typedef TriangleIntegrator12	DualTriangleIntegrator;
#endif // FAST_QUADRATURE
#if SUPER_FAST_QUADRATURE
typedef MedianPolygonIntegrator                                    PrimalPolygonIntegrator;
#else // !SUPER_FAST_QUADRATURE
typedef TriangulatingPolygonIntegrator< PrimalTriangleIntegrator > PrimalPolygonIntegrator;
#endif // SUPER_FAST_QUADRATURE
typedef TriangulatingPolygonIntegrator<   DualTriangleIntegrator >   DualPolygonIntegrator;

template< class Real , bool Primal >
int MeshOctree< Real , Primal >::getDotProductMatrix( ElementMatrix& M , bool progress ) const
{
	if( Primal ) return _getMatrixIntegrals< DOT_MATRIX , PrimalPolygonIntegrator >( M , progress , Real(1.) * _scale * _scale , Real(0.) );
	else         return _getMatrixIntegrals< DOT_MATRIX ,   DualPolygonIntegrator >( M , progress , Real(1.) * _scale * _scale , Real(0.) );
}
template< class Real , bool Primal >
int MeshOctree< Real , Primal >::getLaplacianMatrix( ElementMatrix& M , bool progress ) const
{
	if( Primal ) return _getMatrixIntegrals< LAP_MATRIX , PrimalPolygonIntegrator >( M , progress , Real(0.) , Real(1.) );
	else         return _getMatrixIntegrals< LAP_MATRIX ,   DualPolygonIntegrator >( M , progress , Real(0.) , Real(1.) );
}
template< class Real , bool Primal >
int MeshOctree< Real , Primal >::getConstrainedLaplacianMatrix( ElementMatrix& M , Real dotWeight , Real lapWeight , bool progress ) const
{
	if( Primal ) return _getMatrixIntegrals< DOT_MATRIX | LAP_MATRIX , PrimalPolygonIntegrator >( M , progress , dotWeight * _scale * _scale , lapWeight );
	else         return _getMatrixIntegrals< DOT_MATRIX | LAP_MATRIX ,   DualPolygonIntegrator >( M , progress , dotWeight * _scale * _scale , lapWeight );
}
template< class Real , bool Primal >
int MeshOctree< Real , Primal >::getLaplacianMatrix( ElementMatrix& lMin , ElementMatrix& lMax , Vector< Real >& minCurvatures , Vector< Real >& maxCurvatures , bool progress ) const
{
	std::vector< Point2D< Point3D< Real > > > baseCurvatureDirections;
	std::vector< Point2D< Real > > baseCurvatureValues;
	_setBaseCurvatures( baseCurvatureValues , baseCurvatureDirections );
	if( Primal )
	{
		_getLaplacianMatrixIntegrals< PrimalPolygonIntegrator >( lMin , lMax , baseCurvatureValues , baseCurvatureDirections , progress );
		return _getCurvatures< PrimalPolygonIntegrator >( minCurvatures , maxCurvatures , baseCurvatureValues , baseCurvatureDirections , progress );
	}
	else
	{
		_getLaplacianMatrixIntegrals< DualPolygonIntegrator >( lMin , lMax , baseCurvatureValues , baseCurvatureDirections , progress );
		return _getCurvatures< DualPolygonIntegrator >( minCurvatures , maxCurvatures , baseCurvatureValues , baseCurvatureDirections , progress );
	}
}

template< class Real , bool Primal >
int MeshOctree< Real , Primal >::getDotProductMatrix( SparseMatrix< Real , int >& dotProduct , bool progress ) const
{
	if( Primal ) return _getSystemMatrix< int , DOT_MATRIX , PrimalPolygonIntegrator >( _maxDepth , dotProduct , _scale*_scale , 0 , progress );
	else         return _getSystemMatrix< int , DOT_MATRIX ,   DualPolygonIntegrator >( _maxDepth , dotProduct , _scale*_scale , 0 , progress );
}
template< class Real , bool Primal >
int MeshOctree< Real , Primal >::getLaplacianMatrix( SparseMatrix< Real , int >& laplacian , bool progress ) const
{
	if( Primal ) return _getSystemMatrix< int , LAP_MATRIX , PrimalPolygonIntegrator >( _maxDepth , laplacian , 0 , 1 , progress );
	else         return _getSystemMatrix< int , LAP_MATRIX ,   DualPolygonIntegrator >( _maxDepth , laplacian , 0 , 1 , progress );
}

template< class Real , bool Primal >
int MeshOctree< Real , Primal >::getConstrainedLaplacianMatrix( SparseMatrix< Real , int >& laplacian , Real dotWeight , Real lapWeight , bool progress ) const
{
	if( Primal ) return _getSystemMatrix< int , DOT_MATRIX | LAP_MATRIX , PrimalPolygonIntegrator >( _maxDepth , laplacian , dotWeight * _scale * _scale , lapWeight , progress );
	else         return _getSystemMatrix< int , DOT_MATRIX | LAP_MATRIX ,   DualPolygonIntegrator >( _maxDepth , laplacian , dotWeight * _scale * _scale , lapWeight , progress );
}

template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function >
void MeshOctree< Real , Primal >::getPreciseDotVector( const std::vector< Function >& functions , ElementVector< V >& b , bool progress ) const
{
	std::vector< Function > _functions = functions;
	for( int i=0 ; i<_functions.size() ; i++ ) _functions[i].translateAndScale( -_translation , Real(1.)/_scale );
	if( Primal ) _getElementVector< V , HighPrecisionV , Function , DOT_MATRIX , PrimalPolygonIntegrator >( _functions , b , progress , _scale * _scale , 0 );
	else         _getElementVector< V , HighPrecisionV , Function , DOT_MATRIX ,   DualPolygonIntegrator >( _functions , b , progress , _scale * _scale , 0 );
}

template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function >
void MeshOctree< Real , Primal >::getPreciseGradientDotGradientVector( const std::vector< Function >& functions , ElementVector< V >& b , bool progress ) const
{
	std::vector< Function > _functions = functions;
	for( int i=0 ; i<_functions.size() ; i++ ) _functions[i].translateAndScale( -_translation , Real(1.)/_scale );
	if( Primal ) _getElementVector< V , HighPrecisionV , Function , LAP_MATRIX , PrimalPolygonIntegrator >( _functions , b , progress , 0 , 1 );
	else         _getElementVector< V , HighPrecisionV , Function , LAP_MATRIX ,   DualPolygonIntegrator >( _functions , b , progress , 0 , 1 );
}

template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function >
void MeshOctree< Real , Primal >::getPreciseConstrainedGradientDotGradientVector( const std::vector< Function >& functions , ElementVector< V >& b , Real dotWeight , Real lapWeight , bool progress ) const
{
	std::vector< Function > _functions = functions;
	for( int i=0 ; i<_functions.size() ; i++ ) _functions[i].translateAndScale( -_translation , Real(1.)/_scale );
	if( Primal ) _getElementVector< V , HighPrecisionV , Function , DOT_MATRIX | LAP_MATRIX , PrimalPolygonIntegrator >( _functions , b , progress , dotWeight * _scale * _scale , lapWeight );
	else         _getElementVector< V , HighPrecisionV , Function , DOT_MATRIX | LAP_MATRIX ,   DualPolygonIntegrator >( _functions , b , progress , dotWeight * _scale * _scale , lapWeight );
}
template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function >
void MeshOctree< Real , Primal >::getPreciseDotVector( const std::vector< Function >& functions , std::vector< V >& b , bool progress ) const
{
	std::vector< Function > _functions = functions;
	for( int i=0 ; i<_functions.size() ; i++ ) _functions[i].translateAndScale( -_translation , Real(1.)/_scale );
	if( Primal ) _getPreciseVector< V , HighPrecisionV , Function , DOT_MATRIX , PrimalPolygonIntegrator >( _functions , _maxDepth , b , _scale * _scale , 0 , progress , false );
	else         _getPreciseVector< V , HighPrecisionV , Function , DOT_MATRIX ,   DualPolygonIntegrator >( _functions , _maxDepth , b , _scale * _scale , 0 , progress , false );
}

template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function >
void MeshOctree< Real , Primal >::getPreciseGradientDotGradientVector( const std::vector< Function >& functions , std::vector< V >& b , bool progress ) const
{
	std::vector< Function > _functions = functions;
	for( int i=0 ; i<_functions.size() ; i++ ) _functions[i].translateAndScale( -_translation , Real(1.)/_scale );
	if( Primal ) _getPreciseVector< V , HighPrecisionV , Function , LAP_MATRIX , PrimalPolygonIntegrator >( _functions , _maxDepth , b , 0 , 1 , progress , true );
	else         _getPreciseVector< V , HighPrecisionV , Function , LAP_MATRIX ,   DualPolygonIntegrator >( _functions , _maxDepth , b , 0 , 1 , progress , true );
}

template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function >
void MeshOctree< Real , Primal >::getPreciseConstrainedGradientDotGradientVector( const std::vector< Function >& functions , std::vector< V >& b , Real dotWeight , Real lapWeight , bool progress ) const
{
	std::vector< Function > _functions = functions;
	for( int i=0 ; i<_functions.size() ; i++ ) _functions[i].translateAndScale( -_translation , Real(1.)/_scale );
	if( Primal ) _getPreciseVector< V , HighPrecisionV , Function , DOT_MATRIX | LAP_MATRIX , PrimalPolygonIntegrator >( _functions , _maxDepth , b , dotWeight * _scale * _scale , lapWeight , progress , true );
	else         _getPreciseVector< V , HighPrecisionV , Function , DOT_MATRIX | LAP_MATRIX ,   DualPolygonIntegrator >( _functions , _maxDepth , b , dotWeight * _scale * _scale , lapWeight , progress , true );
}
template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function >
V MeshOctree< Real , Primal >::getPreciseIntegral( const std::vector< Function >& functions ) const
{
	std::vector< Function > _functions = functions;
	for( int i=0 ; i<_functions.size() ; i++ ) _functions[i].translateAndScale( -_translation , Real(1.)/_scale );
	if( Primal ) return _getPreciseIntegral< V , HighPrecisionV , Function , PrimalPolygonIntegrator >( _functions ) * _scale * _scale;
	else         return _getPreciseIntegral< V , HighPrecisionV , Function ,   DualPolygonIntegrator >( _functions ) * _scale * _scale;
}

template< class Real , bool Primal >
template< class V , class HighPrecisionV >
V MeshOctree< Real , Primal >::getPreciseCoefficientIntegral( const Vector< V >& coefficients ) const
{
	if( Primal ) return _getPreciseCoefficientIntegral< V , HighPrecisionV , PrimalPolygonIntegrator >( coefficients , _maxDepth ) * _scale * _scale;
	else         return _getPreciseCoefficientIntegral< V , HighPrecisionV ,   DualPolygonIntegrator >( coefficients , _maxDepth ) * _scale * _scale;
}

template< class Real , bool Primal >
template< class Vertex , class C >
void MeshOctree< Real , Primal >::getValues( const std::vector<Vertex>& points , const Vector< C >& coefficients , std::vector< C >& pointValues )
{
	pointValues.resize( points.size() );
	std::vector< Point3D< Real > > _points( points.size() );
	for( int i=0 ; i<points.size() ; i++ ) _points[i] = ( Point3D< Real >( points[i] ) - _translation ) / _scale;
	_getValues( GetPointer( _points ) , _points.size() , coefficients+0 , GetPointer( pointValues ) );
}
template< class Real , bool Primal >
template< class Vertex >
void MeshOctree< Real , Primal >::getEvaluationMatrix( const std::vector< Vertex >& points , SparseMatrix< Real , int >& E ) const
{
	E.Resize( points.size() );
	for( int i=0 ; i<points.size() ; i++ ) E.SetGroupSize( i , Primal?8:27 );
	E.MakeContiguous();
#pragma omp parallel for num_threads( _threads )
	for( int t=0 ; t<_threads ; t++ )
	{
		typename Node::ConstNeighborKey3 neighborKey3;
		neighborKey3.set( _maxDepth );
		for( int p=(points.size()*t)/_threads ; p<(points.size()*(t+1))/_threads ; p++ )
		{
			Point3D< Real > point = ( Point3D< Real >( points[p] ) - _translation ) / _scale;
			int offset[3];
			typename Node::ConstNeighbors3 neighbors = _getFinestNodeNeighbors( neighborKey3 , point , _maxDepth , offset );
			double values[Primal?2:3][3];
			for( int o=Primal?0:-1 ; o<=1 ; o++ )
				for( int c=0 ; c<3 ; c++ )
				{
					if( offset[c]+o<0 || offset[c]+o>=(1<<_maxDepth) ) continue;
					int idx = Primal ? BinaryNode< Real >::CornerIndex( _maxDepth , offset[c]+o ) : BinaryNode<Real>::CenterIndex( _maxDepth , offset[c]+o );
					values[Primal?o:o+1][c] = _fData.baseFunctions[idx]( point[c] );
				}
			for( int i=0 , idx=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ , idx++ )
					{
						const Node* node = neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k];
						if( node && node->nodeData.basisIndex!=-1 )
							E[p][idx] = MatrixEntry< Real , int >( node->nodeData.basisIndex , values[i][0]*values[j][1]*values[k][2] );
						else 
							E[p][idx] = MatrixEntry< Real , int >( -1                        , values[i][0]*values[j][1]*values[k][2] );
					}
		}
	}
}
template< class Real , bool Primal >
template< class Vertex > 
void MeshOctree< Real , Primal >::getGeometryIndices( const std::vector< Vertex >& points , std::vector< int >& eIndices ) const
{
	eIndices.resize( points.size()  );
	for( int p=0 ; p<points.size() ; p++ )
	{
		Point3D< Real > point = Point3D< Real >( points[p] );
		point = ( point - _translation ) / _scale;
		const Node* node = _getLeafNode( point );
		if( node->depth()==_maxDepth && node->nodeData.geometryIndex!=-1 ) eIndices[p] = node->nodeData.geometryIndex;
		else                                                               eIndices[p] = -1;
	}
}
template< class Real , bool Primal >
template< class Vertex > 
void MeshOctree< Real , Primal >::getGeometryPositions( std::vector< Vertex >& positions ) const
{
	int shift = _maxDepth;
	positions.resize( _geometryDimension[_maxDepth] );	
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) )
		if( node->depth()==_maxDepth && node->nodeData.geometryIndex!=-1)
		{
			int idx=node->nodeData.geometryIndex;
			int d , off[3];
			node->depthAndOffset( d , off );
			for( int c=0 ; c<3 ; c++ ) positions[idx][c] = Real(off[c]+0.5) / (1<<shift);
			for( int c=0 ; c<3 ; c++ ) positions[idx][c] = positions[idx][c] * _scale + _translation[c];
		}
}

/*************************************************************/
template< class Real , bool Primal >
template< class IndexType >
void MeshOctree< Real , Primal >::_setAdjacency( int depth , SparseMatrix< byte , IndexType >& matrix , std::vector< const Node* >& nodes ) const
{
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) ) if( node->depth()==depth && node->nodeData.tEnd>node->nodeData.tStart ) nodes.push_back( node );
	matrix.Resize( nodes.size() );
#pragma omp parallel for num_threads( _threads )
	for( int i=0 ; i<_threads ; i++ )
	{
		typename OctNode< MeshTreeNodeData< Real > , Real >::ConstNeighborKey3 neighborKey3;
		neighborKey3.set( depth );
		for( int j=(nodes.size()*i) / _threads ; j<(nodes.size()*(i+1)) / _threads ; j++ )
		{
			const Node* node = nodes[j];
			int idx = j;
			_tree.getNeighbors( node , neighborKey3 );
			typename Node::ConstNeighbors3& neighbors = neighborKey3.neighbors[depth];
			matrix.SetGroupSize( idx , (Primal?8:27) );

			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
						if( neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k] )
						{
							matrix[idx][i*(Primal?4:9)+j*(Primal?2:3)+k].N = neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k]->nodeData.basisIndex;
							matrix[idx][i*(Primal?4:9)+j*(Primal?2:3)+k].Value = 0;
						}
						else 
							matrix[idx][i*(Primal?4:9)+j*(Primal?2:3)+k].N = -1;
		}
	}
}

template< class Real , bool Primal >
template< class IndexType >
void MeshOctree< Real , Primal >::_setAdjacency( int depth , SparseMatrix< Real , IndexType >& matrix ) const
{
	matrix.Resize( _basisDimension[depth] );

	std::vector< const Node* > nodes;
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) ) if( node->depth()==depth && node->nodeData.basisIndex!=-1 ) nodes.push_back( node );

#pragma omp parallel for num_threads( _threads )
	for( int i=0 ; i<_threads ; i++ )
	{
		typename OctNode< MeshTreeNodeData< Real > , Real >::ConstNeighborKey3 neighborKey3;	
		typename OctNode< MeshTreeNodeData< Real > , Real >::ConstNeighborKey5 neighborKey5;
		neighborKey3.set( depth );
		neighborKey5.set( depth );
		for( int j=(nodes.size()*i) / _threads ; j<(nodes.size()*(i+1)) / _threads ; j++ )
		{
			const Node* node = nodes[j];
			int idx = node->nodeData.basisIndex;
			if( Primal )
			{
				matrix.SetGroupSize( idx , 3*3*3 );

				_tree.getNeighbors( node , neighborKey3 );
				typename Node::ConstNeighbors3& neighbors = neighborKey3.neighbors[depth];
				for( int i=0 ; i<3 ; i++ )
					for( int j=0 ; j<3 ; j++ )
						for( int k=0 ; k<3 ; k++ )
							if( neighbors.neighbors[i][j][k] && neighbors.neighbors[i][j][k]->nodeData.basisIndex!=-1 )
							{
								matrix[idx][i*3*3+j*3+k].N = neighbors.neighbors[i][j][k]->nodeData.basisIndex;
								matrix[idx][i*3*3+j*3+k].Value = 0;
							}
							else 
								matrix[idx][i*3*3+j*3+k].N = -1;
			}
			else
			{
				matrix.SetGroupSize( idx , 5*5*5 );

				_tree.getNeighbors( node , neighborKey5 );
				typename Node::ConstNeighbors5& neighbors = neighborKey5.neighbors[depth];
				for( int i=0 ; i<5 ; i++ )
					for( int j=0 ; j<5 ; j++ )
						for( int k=0 ; k<5 ; k++ )
							if( neighbors.neighbors[i][j][k] && neighbors.neighbors[i][j][k]->nodeData.basisIndex!=-1 )
							{
								matrix[idx][i*5*5+j*5+k].N = neighbors.neighbors[i][j][k]->nodeData.basisIndex;
								matrix[idx][i*5*5+j*5+k].Value = 0;
							}
							else
								matrix[idx][i*5*5+j*5+k].N = -1;
			}
		}
	}
}

template< class Real , bool Primal >
template< class IndexType >
void MeshOctree< Real , Primal >::_compressMatrix( SparseMatrix< Real , IndexType >& matrix , bool resize ) const
{
	if( resize )
	{
		MatrixEntry< Real , IndexType > entries[(Primal?3:5)][(Primal?3:5)][(Primal?3:5)];
		for( int idx=0 ; idx<matrix.groups ; idx++ )
		{
			int count = 0;
			for( int x=0 ; x<(Primal?3:5) ; x++ ) for( int y=0 ; y<(Primal?3:5) ; y++ ) for( int z=0 ; z<(Primal?3:5) ; z++ )
				if( matrix[idx][x*(Primal?9:25)+y*(Primal?3:5)+z].N!=-1 )
				{
					entries[x][y][z] = matrix[idx][x*(Primal?9:25)+y*(Primal?3:5)+z];
					count++;
				}
				else entries[x][y][z].N = -1;
				matrix.SetGroupSize( idx , count );
				matrix.groupSizes[idx] = 0;
				for( int x=0 ; x<(Primal?3:5) ; x++ ) for( int y=0 ; y<(Primal?3:5) ; y++ ) for( int z=0 ; z<(Primal?3:5) ; z++ )
					if( entries[x][y][z].N != -1 ) matrix[idx][ matrix.groupSizes[idx]++ ] = entries[x][y][z];
		}
	}
	else
		for( int idx=0 ; idx<matrix.groups ; idx++ )
			for( int j=matrix.groupSizes[idx]-1 ; j>=0 ; j-- )
				if( matrix[idx][j].N != -1 ) matrix[idx][j] = matrix[idx][--matrix.groupSizes[idx]];
}


template< class Real , bool Primal >
template< class IndexType >
void MeshOctree< Real , Primal >::_setFromSums( SparseMatrix< Real , IndexType >& matrix , const SparseMatrix< byte , IndexType >& adjacency , double sums[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)][(Primal?2:3)][(Primal?2:3)][(Primal?2:3)] , int idx ) const
{
	for( int i=0 ; i<(Primal?2:3) ; i++ )
		for( int j=0 ; j<(Primal?2:3) ; j++ )
			for( int k=0 ; k<(Primal?2:3) ; k++ )
				if( adjacency[idx][i*(Primal?4:9)+j*(Primal?2:3)+k].N != -1 )
				{
					int idx1 = adjacency[idx][i*(Primal?4:9)+j*(Primal?2:3)+k].N;
					for( int ii=i ; ii<(Primal?2:3) ; ii++ )
						for( int jj=0 ; jj<(Primal?2:3) ; jj++ )
							if( ii==i && jj<j ) continue;
							else
								for( int kk=0 ; kk<(Primal?2:3) ; kk++ )
									if( ii==i && jj==j && kk<k ) continue;
									else if( adjacency[idx][ii*(Primal?4:9)+jj*(Primal?2:3)+kk].N != -1 )
									{
										double temp = sums[i][j][k][ii][jj][kk];
										int idx2 = adjacency[idx][ii*(Primal?4:9)+jj*(Primal?2:3)+kk].N;
										int subIndex1 = (i-ii+(Primal?1:2))*(Primal?9:25)+(j-jj+(Primal?1:2))*(Primal?3:5)+(k-kk+(Primal?1:2));
										int subIndex2 = (ii-i+(Primal?1:2))*(Primal?9:25)+(jj-j+(Primal?1:2))*(Primal?3:5)+(kk-k+(Primal?1:2));
										matrix[ idx1 ][ subIndex2 ].Value += Real( temp );
										if( idx1!=idx2 ) matrix[ idx2 ][ subIndex1 ].Value += Real( temp );
									}
				}
}
template< class Real , bool Primal >
template< bool SetDValues >
void MeshOctree< Real , Primal >::_setSampleSubValues( const Node* node , ConstPointer( Point3D< double > ) samples , int sampleCount , Pointer( double ) values[(Primal?2:3)][3] , Pointer( double ) dValues[(Primal?2:3)][3] ) const
{
	Polynomial< Primal ? 1 : 2 >  polys[(Primal?2:3)][3];
	Polynomial< Primal ? 0 : 1 > dPolys[(Primal?2:3)][3];
	int dd , off[3];
	node->depthAndOffset( dd , off );

	// Accumulate the polynomials corresponding to the node
	for( int o=(Primal?0:-1) ; o<=1 ; o++ )
		for( int c=0 ; c<3 ; c++ )
		{
			if( off[c]+o<0 || off[c]+o>=(1<<dd) )	continue;
			int idx = Primal ? BinaryNode<Real>::CornerIndex( dd , off[c]+o ) : BinaryNode<Real>::CenterIndex( dd , off[c]+o );
			for( int j=0 ; j<=1-o ; j++ ) polys[(Primal?o:o+1)][c] += _fData.baseFunctions[idx].polys[j].p;
			if( SetDValues ) dPolys[(Primal?o:o+1)][c] = polys[(Primal?o:o+1)][c].derivative();
		}

	for( int o=(Primal?0:-1) ; o<=1 ; o++ )
		for( int c=0 ; c<3 ; c++ )
			if( off[c]+o>=0 && off[c]+(Primal?o:o+1)<=(1<<dd) )
				for( int s=0 ; s<sampleCount ; s++ )
				{
					values[(Primal?o:o+1)][c][s] = polys[(Primal?o:o+1)][c]( samples[s][c] );
					if( SetDValues ) dValues[(Primal?o:o+1)][c][s] = dPolys[(Primal?o:o+1)][c]( samples[s][c] );
				}
}
template< class Real , bool Primal >
template< class IndexType , int MatrixType , class Integrator >
int MeshOctree< Real , Primal >::_getSystemMatrix( int depth , SparseMatrix< Real , IndexType >& matrix , Real dotWeight , Real lapWeight , bool progress ) const
{
	SparseMatrix< byte , IndexType > adjacency;
	std::vector< const Node* > nodes;
	_setAdjacency( depth , adjacency , nodes );
	_setAdjacency( depth , matrix );
	int max = _maxSamplesPerNode< Integrator >( );

	ProgressBar* bar=NULL;
	if( progress ) bar = new ProgressBar( 20 , _polygons.size() , "Setting Matrix" );
#pragma omp parallel for num_threads( _threads )
	for( int i=0 ; i<_threads ; i++ )
	{
		Point3D< double > vertices[ PolygonData::MAX_VERTICES ];
		Integrator integrator;
		Pointer( Point3D< double > ) samples;
		Pointer( Point3D< double > ) normals;
		Pointer( double ) weights;
		Pointer( double )  subValues[(Primal?2:3)][3];
		Pointer( double ) subDValues[(Primal?2:3)][3];
		Pointer( double )            fullDotValues[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)];
		Pointer( Point3D< double > ) fullLapValues[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)];
		samples = AllocPointer< Point3D< double > >( max );
		normals = AllocPointer< Point3D< double > >( max );
		weights = AllocPointer< double >( max );
		for( int ii=0 ; ii<(Primal?2:3) ; ii++ ) for( int jj=0 ; jj<3 ; jj++ ) subValues[ii][jj] = AllocPointer< double >( max ) , subDValues[ii][jj] = AllocPointer< double >( max );
		for( int ii=0 ; ii<(Primal?2:3) ; ii++ ) for( int jj=0 ; jj<(Primal?2:3) ; jj++ ) for( int kk=0 ; kk<(Primal?2:3) ; kk++ ) fullDotValues[ii][jj][kk] = AllocPointer< double >( max ) , fullLapValues[ii][jj][kk] = AllocPointer< Point3D< double > >( max );
		for( int j=(nodes.size()*i) / _threads ; j<(nodes.size()*(i+1)) / _threads ; j++ )
		{
			int sampleCount;
			const Node* node = nodes[j];
			int idx = j;

			double sums[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)][(Primal?2:3)][(Primal?2:3)][(Primal?2:3)];
			memset( sums , 0 , sizeof( sums ) );

			sampleCount = 0;
			for( int t=0 ; t<node->nodeData.tEnd-node->nodeData.tStart ; t++ )
			{
				if( progress ) bar->update( i==0 );
				int pIdx = t+node->nodeData.tStart;
				for( int j=0 ; j<_polygons[pIdx].count ; j++ ) vertices[j] = Point3D< double >( _vertices[ _polygons[pIdx][j] ] );
				integrator.set( vertices , _polygons[pIdx].count );
				for( int i=0 ; i<Integrator::SampleNum( _polygons[pIdx].count ) ; i++ )
				{
					samples[ sampleCount ] = integrator[i];
					weights[ sampleCount ] = integrator.weight(i);
					if( MatrixType&LAP_MATRIX ) normals[ sampleCount ] = Point3D< double >( _baseNormals[ _polygons[pIdx].parent ] );
					sampleCount++;
				}
			}
			// In case the area is zero, want to make sure we don't accidently take the square-root of a negative number.
			for( int s=0 ; s<sampleCount ; s++ ) weights[s] = sqrt( fabs( weights[s] ) );

			_setSampleSubValues< MatrixType&LAP_MATRIX >( node , samples , sampleCount , subValues , subDValues );

			// Iterate over all pairs of neighbors
			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
						if( adjacency[idx][i*(Primal?4:9)+j*(Primal?2:3)+k].N != -1 )
							for( int s=0 ; s<sampleCount ; s++ )
							{
								if( MatrixType&DOT_MATRIX )
									fullDotValues[i][j][k][s]    =  subValues[i][0][s] *  subValues[j][1][s] *  subValues[k][2][s] * weights[ s ];
								if( MatrixType&LAP_MATRIX )
								{
									fullLapValues[i][j][k][s][0] = subDValues[i][0][s] *  subValues[j][1][s] *  subValues[k][2][s];
									fullLapValues[i][j][k][s][1] =  subValues[i][0][s] * subDValues[j][1][s] *  subValues[k][2][s];
									fullLapValues[i][j][k][s][2] =  subValues[i][0][s] *  subValues[j][1][s] * subDValues[k][2][s];
									fullLapValues[i][j][k][s] -= normals[s] * Point3D< double >::Dot( normals[s] , fullLapValues[i][j][k][s] );
									fullLapValues[i][j][k][s] *= weights[s];
								}
							}
			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
						if( adjacency[idx][i*(Primal?4:9)+j*(Primal?2:3)+k].N != -1 )
							for( int ii=i ; ii<(Primal?2:3) ; ii++ )
								for( int jj=0 ; jj<(Primal?2:3) ; jj++ )
									if( ii==i && jj<j ) continue;
									else
										for( int kk=0 ; kk<(Primal?2:3) ; kk++ )
											if( ii==i && jj==j && kk<k ) continue;
											else if( adjacency[idx][ii*(Primal?4:9)+jj*(Primal?2:3)+kk].N != -1 )
											{
												double tempLap = 0 , tempDot = 0;
												for( int s=0 ; s<sampleCount ; s++ )
												{
													if( MatrixType&DOT_MATRIX ) tempDot += fullDotValues[i][j][k][s] * fullDotValues[ii][jj][kk][s];
													if( MatrixType&LAP_MATRIX ) tempLap += Point3D< double >::Dot( fullLapValues[i][j][k][s] , fullLapValues[ii][jj][kk][s] );
												}
												if     ( MatrixType&DOT_MATRIX && MatrixType&LAP_MATRIX ) sums[i][j][k][ii][jj][kk] += tempDot*dotWeight + tempLap*lapWeight;
												else if( MatrixType&DOT_MATRIX                          ) sums[i][j][k][ii][jj][kk] += tempDot*dotWeight                    ;
												else if(                          MatrixType&LAP_MATRIX ) sums[i][j][k][ii][jj][kk] +=                     tempLap*lapWeight;

											}
			_setFromSums( matrix , adjacency , sums , idx );
		}
		FreePointer( samples );
		FreePointer( normals );
		FreePointer( weights );
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<3 ; j++ ) { FreePointer( subValues[i][j] ) ; FreePointer( subDValues[i][j] ); }
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<(Primal?2:3) ; j++ ) for( int k=0 ; k<(Primal?2:3) ; k++ ) { FreePointer( fullDotValues[i][j][k] ) ; FreePointer( fullLapValues[i][j][k] ); }
	}
	if( progress ) delete bar;

	MemoryUsage();
	_compressMatrix( matrix );
	MemoryUsage();
	return 1;
}
template< class Real , bool Primal >
template< class Integrator >
int MeshOctree< Real , Primal >::_getCurvatures( Vector< Real >& minCurvatures , Vector< Real >& maxCurvatures ,
												 const std::vector< Point2D< Real > >& _baseCurvatureValues , const std::vector< Point2D< Point3D< Real > > > _baseCurvatureDirections ,
												 bool progress ) const
{
	std::vector< Real > weightSum;

	std::vector< const Node* > nodes;
	nodes.resize( _geometryDimension[_maxDepth] );
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) )
		if( node->depth()==_maxDepth && node->nodeData.geometryIndex!=-1 )
			nodes[node->nodeData.geometryIndex] = node;
	minCurvatures.Resize( _geometryDimension[_maxDepth] );
	maxCurvatures.Resize( _geometryDimension[_maxDepth] );
	weightSum.resize    ( _geometryDimension[_maxDepth] );
	for( int i=0 ; i<weightSum.size() ; i++ ) maxCurvatures[i] = minCurvatures[i] = weightSum[i] = 0;
	int max = _maxSamplesPerNode< Integrator >( );

	ProgressBar* bar=NULL;
	if( progress ) bar = new ProgressBar( 20 , _polygons.size() , "Setting Anisotropic Laplacian Curvatures" );
#pragma omp parallel for num_threads( _threads )
	for( int i=0 ; i<_threads ; i++ )
	{
		Point3D< double > vertices[ PolygonData::MAX_VERTICES ];
		Integrator integrator;
		int sampleCount;
		double* weights;
		double* minCurvature;
		double* maxCurvature;
		weights = new double[ max ];
		minCurvature = new double[ max ];
		maxCurvature = new double[ max ];
		for( int j=(nodes.size()*i) / _threads ; j<(nodes.size()*(i+1)) / _threads ; j++ )
		{
			const Node* node = nodes[j];
			int idx = j;

			sampleCount = 0;
			for( int t=0 ; t<node->nodeData.tEnd-node->nodeData.tStart ; t++ )
			{
				if( progress ) bar->update( i==0 );
				int pIdx = t+node->nodeData.tStart;
				for( int j=0 ; j<_polygons[pIdx].count ; j++ ) vertices[j] = Point3D< double >( _vertices[ _polygons[pIdx][j] ] );
				integrator.set( vertices , _polygons[pIdx].count );
				for( int i=0 ; i<Integrator::SampleNum( _polygons[pIdx].count ) ; i++ )
				{
					weights[ sampleCount ] = integrator.weight(i);
					minCurvature[sampleCount] = _baseCurvatureValues[ _polygons[pIdx].parent ][MIN_CURVATURE];
					maxCurvature[sampleCount] = _baseCurvatureValues[ _polygons[pIdx].parent ][MAX_CURVATURE];
					sampleCount++;
				}
			}
			
			for( int s=0 ; s<sampleCount ; s++ )
			{
				minCurvatures[ idx ] += Real( minCurvature[s] * weights[s] );
				maxCurvatures[ idx ] += Real( maxCurvature[s] * weights[s] );
				weightSum[ idx ] += Real( weights[s] );
			}
		}
		delete[] minCurvature;
		delete[] maxCurvature;
		delete[] weights;
	}
	if( progress ) delete bar;

	MemoryUsage();
	for( int i=0 ; i<weightSum.size() ; i++ ) if( weightSum[i] ) maxCurvatures[i] /= weightSum[i] , minCurvatures[i] /=weightSum[i];
	return 1;
}
template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function , int MatrixType , class Integrator >
int MeshOctree< Real , Primal >::_getPreciseVector( const std::vector< Function >& functions , int depth , std::vector< V >& b , Real dotWeight , Real lapWeight , bool progress , bool normalize ) const
{
	b.resize( _basisDimension[depth] );
	for( int i=0 ; i<_basisDimension[depth] ; i++ ) b[i] *= 0;

	int count = _basisDimension[depth];

	ProgressBar* bar=NULL;

	int max = 0;
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) )
		if( node->depth()==depth && node->nodeData.basisIndex!=-1 && node->nodeData.tEnd>node->nodeData.tStart )
		{
			int c=0;
			for( int t=node->nodeData.tStart ; t<node->nodeData.tEnd ; t++ ) c += Integrator::SampleNum( _polygons[t].count );
			max = std::max< int >( max , c );
		}

	std::vector< const Node* > nodes;
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) ) if( node->depth()==depth && node->nodeData.tEnd>node->nodeData.tStart ) nodes.push_back( node );
	if( progress ) bar = new ProgressBar( 20 , nodes.size() , "Setting Constraints" );
#pragma omp parallel for num_threads( _threads )
	for( int i=0 ; i<_threads ; i++ )
	{
		Point3D< double > vertices[ PolygonData::MAX_VERTICES ];
		Integrator integrator;
		Pointer( Point3D< double > ) samples;
		Pointer( double ) weights;
		Pointer( HighPrecisionV ) tValues;
		Pointer( double )  subValues[(Primal?2:3)][3];
		Pointer( double ) subDValues[(Primal?2:3)][3];
		Pointer( Point3D< double > ) fullValues[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)];
		Pointer( Gradient3D< HighPrecisionV > ) grads;
		samples = AllocPointer< Point3D< double > >( max );
		weights = AllocPointer< double >( max );
		tValues = AllocPointer< HighPrecisionV >( max );
		grads = AllocPointer< Gradient3D< HighPrecisionV > >( max );
		for( int i=0 ; i<max ; i++ ) tValues[i] *= 0 , grads[i] *= 0;
		for( int ii=0 ; ii<(Primal?2:3) ; ii++ ) for( int jj=0 ; jj<3 ; jj++ ) subValues[ii][jj] = AllocPointer< double >( max ) , subDValues[ii][jj] = AllocPointer< double >( max );
		for( int ii=0 ; ii<(Primal?2:3) ; ii++ ) for( int jj=0 ; jj<(Primal?2:3) ; jj++ ) for( int kk=0 ; kk<(Primal?2:3) ; kk++ ) fullValues[ii][jj][kk] = AllocPointer< Point3D< double > >( max );
		typename OctNode< MeshTreeNodeData< Real > , Real >::ConstNeighborKey3 neighborKey3;	
		neighborKey3.set( depth );
		for( int j=(nodes.size()*i) / _threads ; j<(nodes.size()*(i+1)) / _threads ; j++ )
		{
			int sampleCount;
			const Node* node = nodes[j];
			if( progress ) bar->update( i==0 );

			_tree.getNeighbors( node , neighborKey3 );
			typename Node::ConstNeighbors3& neighbors=neighborKey3.neighbors[depth];

			HighPrecisionV sums[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)];
			for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<(Primal?2:3) ; j++ ) for( int k=0 ; k<(Primal?2:3) ; k++ ) sums[i][j][k] *= 0;

			sampleCount = 0;
			for( int t=0 ; t<node->nodeData.tEnd-node->nodeData.tStart ; t++ )
			{
				int pIdx = t+node->nodeData.tStart;
				Point3D< double > normal = Point3D< double >( _baseNormals[ _polygons[pIdx].parent ] );
				for( int j=0 ; j<_polygons[pIdx].count ; j++ ) vertices[j] = Point3D< double >( _vertices[ _polygons[pIdx][j] ] );
				integrator.set( vertices , _polygons[pIdx].count );
				for( int i=0 ; i<Integrator::SampleNum( _polygons[pIdx].count ) ; i++ )
				{
					samples[ sampleCount ] = integrator[i];
					weights[ sampleCount ] = integrator.weight(i);
					if( MatrixType&LAP_MATRIX )
					{
						if( normalize )
						{
							HighPrecisionV dot;
							dot *= 0;
							Gradient3D< V >  g = functions[_polygons[pIdx].parent].gradient( Point3D< typename V::R >( samples[sampleCount] ) );
							for( int c=0 ; c<3 ; c++ ) grads[sampleCount][c] = HighPrecisionV( g[c] );
							for( int c=0 ; c<3 ; c++ ) dot += grads[sampleCount][c] * normal[c];
							for( int c=0 ; c<3 ; c++ ) grads[sampleCount][c] -= dot * normal[c];
						}
						else
						{
							Gradient3D< V >  g = functions[_polygons[pIdx].parent].gradient( Point3D< typename V::R >( samples[sampleCount] ) );
							for( int c=0 ; c<3 ; c++ ) grads[sampleCount][c] = HighPrecisionV( g[c] );
						}
						grads[sampleCount] *= weights[sampleCount];
					}
					if( MatrixType&DOT_MATRIX ) tValues[sampleCount] = HighPrecisionV( functions[_polygons[pIdx].parent]( Point3D< typename V::R >( samples[sampleCount]) ) ) * weights[sampleCount];
					sampleCount++;
				}
			}
			_setSampleSubValues< MatrixType&LAP_MATRIX >( node , samples , sampleCount , subValues , subDValues );

			// Iterate over all neighbors
			if( MatrixType&LAP_MATRIX )
				for( int i=0 ; i<(Primal?2:3) ; i++ )
					for( int j=0 ; j<(Primal?2:3) ; j++ )
						for( int k=0 ; k<(Primal?2:3) ; k++ )
							if( neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k] && neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k]->nodeData.basisIndex!=-1 )
								for( int s=0 ; s<sampleCount ; s++ )
								{
									fullValues[i][j][k][s][0] = subDValues[i][0][s] *  subValues[j][1][s] *  subValues[k][2][s];
									fullValues[i][j][k][s][1] =  subValues[i][0][s] * subDValues[j][1][s] *  subValues[k][2][s];
									fullValues[i][j][k][s][2] =  subValues[i][0][s] *  subValues[j][1][s] * subDValues[k][2][s];
								}
			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
						if( neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k] && neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k]->nodeData.basisIndex!=-1 )
						{
							HighPrecisionV dotTemp , lapTemp;
							dotTemp *= 0;
							lapTemp *= 0;
							if( MatrixType&DOT_MATRIX )	for( int s=0 ; s<sampleCount ; s++ ) dotTemp += tValues[s] * subValues[i][0][s] * subValues[j][1][s] * subValues[k][2][s];
							if( MatrixType&LAP_MATRIX )	for( int s=0 ; s<sampleCount ; s++ ) for( int c=0 ; c<3 ; c++ ) lapTemp += grads[s][c] * fullValues[i][j][k][s][c];
							sums[i][j][k] += dotTemp * dotWeight + lapTemp * lapWeight;
						}
			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
						if( neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k] && neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k]->nodeData.basisIndex!=-1 )
							b[ neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k]->nodeData.basisIndex ] += V( sums[i][j][k] );
		}
		FreePointer( samples );
		FreePointer( weights );
		FreePointer( tValues );
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<3 ; j++ ) { FreePointer( subValues[i][j] ) ; FreePointer( subDValues[i][j] ); }
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<(Primal?2:3) ; j++ ) for( int k=0 ; k<(Primal?2:3) ; k++ ) { FreePointer( fullValues[i][j][k] ); }
		FreePointer( grads );
	}
	if( progress ) delete bar;
	MemoryUsage();

	return 1;
}

template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Integrator >
V MeshOctree< Real , Primal >::_getPreciseCoefficientIntegral( const Vector< V >& coefficients, int depth ) const
{
	Integrator integrator;
	Point3D< double > scratchSample;
	double scratchWeight;
	int sampleCount = Integrator::SampleNum( PolygonData::MAX_VERTICES );
	Point3D< double > vertices[ PolygonData::MAX_VERTICES ];
	double* weights = new double[ sampleCount ];
	Pointer( Point3D< double > ) samples = AllocPointer< Point3D< double > >( sampleCount );

	typename OctNode< MeshTreeNodeData< Real > , Real >::ConstNeighborKey3 neighborKey3;	
	neighborKey3.set( depth );

	HighPrecisionV integral;
	integral *= 0;
	int count = _basisDimension[depth];
	Pointer( double ) subValues[(Primal?2:3)][3];
	for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<3 ; j++ ) subValues[i][j] = AllocPointer< double >( Integrator::SampleNum( PolygonData::MAX_VERTICES ) );

	// Iterate over all nodes with geometry
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) )
	{
		if( node->depth()!=depth || node->nodeData.tEnd<=node->nodeData.tStart ) continue;
		_tree.getNeighbors( node , neighborKey3 );
		typename Node::ConstNeighbors3& neighbors=neighborKey3.neighbors[depth];

		HighPrecisionV nodeSum;
		nodeSum *= 0;
		// Iterate over all the triangles in the node
		for( int t=node->nodeData.tStart ; t<node->nodeData.tEnd ; t++ )
		{
			for( int j=0 ; j<_polygons[t].count ; j++ ) vertices[j] = Point3D< double >( _vertices[ _polygons[t][j] ] );
			integrator.set( vertices , _polygons[t].count );
			for( int j=0 ; j<Integrator::SampleNum( _polygons[t].count ) ; j++ )
			{
				weights[j] = integrator.weight(j);
				samples[j] = integrator[j];
			}
			_setSampleSubValues< false >( node , samples , Integrator::SampleNum( _polygons[t].count ) , subValues , NULL );
			HighPrecisionV triangleSum;
			triangleSum *= 0;
			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
						if( neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k] && neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k]->nodeData.basisIndex!=-1 )
						{
							double integral = 0;
							for( int s=0 ; s<Integrator::SampleNum( _polygons[t].count ) ; s++ ) integral += subValues[i][0][s] * subValues[j][1][s] * subValues[k][2][s] * weights[s];
							triangleSum += HighPrecisionV( coefficients[ neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k]->nodeData.basisIndex ] ) * integral;
						}
			nodeSum += triangleSum;
		}
		integral += nodeSum;
	}
	for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<3 ; j++ ) { FreePointer( subValues[i][j] ); }
	delete[] weights;
	FreePointer( samples );
	return V( integral );
}
template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function , class Integrator >
V MeshOctree< Real , Primal >::_getPreciseIntegral( const std::vector< Function >& functions ) const
{
	HighPrecisionV integral;
	Integrator integrator;
	HighPrecisionV* values = new HighPrecisionV[ Integrator::SampleNum( PolygonData::MAX_VERTICES ) ];
	Point3D< double > vertices[ PolygonData::MAX_VERTICES ];
	integral *= 0;
	for( int i=0 ; i<_polygons.size() ; i++ )
	{
		for( int j=0 ; j<_polygons[i].count ; j++ ) vertices[j] = Point3D< double >( _vertices[ _polygons[i][j] ] );
		integrator.set( vertices , _polygons[i].count );
		for( int s=0 ; s<Integrator::SampleNum( _polygons[i].count ) ; s++ ) values[s] = HighPrecisionV( functions[ _polygons[i].parent ]( integrator[s] ) );
		integral += integrator.integrate( values );
	}
	delete[] values;
	return V( integral );
}
template< class Real , bool Primal >
template< class Vertex , class C >
void MeshOctree< Real , Primal >::_getValues( ConstPointer( Vertex ) points , int pCount , ConstPointer( C ) coefficients , Pointer( C ) pointValues )
{
	typename Node::ConstNeighborKey3 key3;
	key3.set( _maxDepth );
	for( int p=0 ; p<pCount ; p++ )
	{
		Point3D< Real > point = Point3D< Real >( points[p] );
		int depth , offset[3];
		depth = _maxDepth;
		typename Node::ConstNeighbors3 neighbors = _getFinestNodeNeighbors( key3 , point , depth , offset );
		double values[(Primal?2:3)][3];

		C value;
		value *= Real( 0 );
		for( int o=(Primal?0:-1) ; o<=1 ; o++ )
			for( int c=0 ; c<3 ; c++ )
			{
				if( offset[c]+o<0 || offset[c]+o>=(1<<depth) ) continue;
				int idx = Primal ? BinaryNode<Real>::CornerIndex( depth , offset[c]+o ) : BinaryNode<Real>::CenterIndex( depth , offset[c]+o );
				values[(Primal?o:o+1)][c] = _fData.baseFunctions[idx]( point[c] );
			}
		for( int i=0 ; i<(Primal?2:3) ; i++ )
			for( int j=0 ; j<(Primal?2:3) ; j++ )
				for( int k=0 ; k<(Primal?2:3) ; k++ )
				{
					const Node* node = neighbors.neighbors[Primal?i+1:i][Primal?j+1:j][Primal?k+1:k];
					if( node && node->nodeData.basisIndex!=-1 )
						value += coefficients[ node->nodeData.basisIndex ]*Real( values[i][0]*values[j][1]*values[k][2] );
				}
		pointValues[ p ] = value;
	}
}
template< class Real , bool Primal >
template< class MReal , class IndexType >
void MeshOctree< Real , Primal >::_getDownSampleMatrix( int depth , SparseMatrix< MReal , IndexType > &highToLow ) const
{
	typename Node::ConstNeighborKey3 neighborKey3;
	neighborKey3.set( depth );
	highToLow.rowMajor =  true;
	highToLow.Resize( _basisDimension[depth-1] );
	const Node* childNodes[Primal?3:4][Primal?3:4][Primal?3:4];

	typename FiniteElements1D< Real , ZERO_DERIVATIVE , Primal?1:2 >::FullProlongationStencil pStencil;
	typename FiniteElements1D< Real , ZERO_DERIVATIVE , Primal?1:2 >::FullProlongationStencil::ProlongationStencil pStencils[3];
	int tempD;
	// Modify this in case depth = 1 in which case there are no interior stencil values
	FiniteElements1D< Real , ZERO_DERIVATIVE , Primal?1:2 >::ProlongationStencil( Primal?5:4 , pStencil , tempD );

	for( const Node* node = _tree.nextNode() ; node ; node = _tree.nextNode(node) )
	{
		if( node->depth()!=depth-1 || node->nodeData.basisIndex==-1 ) continue;

		int idx = node->nodeData.basisIndex;
		int cCount , d , off[3];
		_tree.getNeighbors( node , neighborKey3 );
		typename Node::ConstNeighbors3& neighbors=neighborKey3.neighbors[depth-1];

		node->depthAndOffset( d , off );

		// Set the array of children that could be effected by a prolongation
		memset( childNodes , 0 , sizeof(childNodes) );

		for( int i=0 ; i<3 ; i++ ) for( int j=0 ; j<3 ; j++ ) for( int k=0 ; k<3 ; k++ )
			if( neighbors.neighbors[i][j][k] && neighbors.neighbors[i][j][k]->children )
				for( int x=0 ; x<2 ; x++ )
				{
					int ii=2*i+x-1;
					if( ii<0 || ii>=(Primal?3:4) ) continue;
					for( int y=0 ; y<2 ; y++ )
					{
						int jj=2*j+y-1;
						if( jj<0 || jj>=(Primal?3:4) ) continue;
						for( int z=0 ; z<2 ; z++ )
						{
							int kk=2*k+z-1;
							if( kk<0 || kk>=(Primal?3:4) ) continue;

							int corner=Cube::CornerIndex(x,y,z);
							if(neighbors.neighbors[i][j][k]->children[corner].nodeData.basisIndex!=-1)
								childNodes[ii][jj][kk] = neighbors.neighbors[i][j][k]->children+corner;
						}
					}
				}
		cCount = 0;
		for( int i=0 ; i<(Primal?3:4) ;i++ ) for( int j=0 ; j<(Primal?3:4) ; j++ ) for( int k=0 ; k<(Primal?3:4) ; k++ ) if( childNodes[i][j][k] ) cCount++;
		highToLow.SetGroupSize( idx , cCount );

		for(int c=0;c<3;c++) pStencils[c] = pStencil.caseTable[1];
		cCount = 0;
		for( int i=0 ; i<(Primal?3:4) ; i++ ) for( int j=0 ; j<(Primal?3:4) ; j++ ) for( int k=0 ; k<(Primal?3:4) ; k++ )
			if(childNodes[i][j][k])
			{
				highToLow[idx][cCount].N = childNodes[i][j][k]->nodeData.basisIndex;
				highToLow[idx][cCount].Value += pStencils[0].values[i]*pStencils[1].values[j]*pStencils[2].values[k];
				cCount++;
			}
	}
}

template< class Real , bool Primal >
template< int MatrixType , class Integrator >
int MeshOctree< Real , Primal >::_getMatrixIntegrals( ElementMatrix& eM , bool progress , Real dWeight , Real lWeight ) const
{
	std::vector< const Node* > nodes;
	nodes.resize( _geometryDimension[_maxDepth] );
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) )
		if( node->depth()==_maxDepth && node->nodeData.geometryIndex!=-1 )
			nodes[node->nodeData.geometryIndex] = node;
	eM.resize( nodes.size() );
	int max = _maxSamplesPerNode< Integrator >( );

	ProgressBar* bar=NULL;
	if( progress ) bar = new ProgressBar( 20 , _polygons.size() , "Setting System Integrals" );
#pragma omp parallel for num_threads( _threads )
	for( int i=0 ; i<_threads ; i++ )
	{
		Point3D< double > vertices[ PolygonData::MAX_VERTICES ];
		Integrator integrator;
		int sampleCount;
		Pointer( Point3D< double > ) samples;
		Pointer( Point3D< double > ) normals;
		Pointer( double ) weights;
		Pointer( double )  subValues[(Primal?2:3)][3];
		Pointer( double ) subDValues[(Primal?2:3)][3];
		Pointer( double )             fullValues[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)];
		Pointer( Point3D< double > ) fullDValues[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)];
		samples = AllocPointer< Point3D< double > >( max );
		normals = AllocPointer< Point3D< double > >( max );
		weights = AllocPointer< double >( max );
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<3 ; j++ ) subValues[i][j] = AllocPointer< double >( max ) , subDValues[i][j] = AllocPointer< double >( max );
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<(Primal?2:3) ; j++ ) for( int k=0 ; k<(Primal?2:3) ; k++ ) fullValues[i][j][k] = AllocPointer< double >( max ) , fullDValues[i][j][k] = AllocPointer< Point3D< double > >( max );
		for( int j=(nodes.size()*i) / _threads ; j<(nodes.size()*(i+1)) / _threads ; j++ )
		{
			const Node* node = nodes[j];
			int idx = j;

			eM[idx].clear();

			sampleCount = 0;
			for( int t=0 ; t<node->nodeData.tEnd-node->nodeData.tStart ; t++ )
			{
				if( progress ) bar->update( i==0 );
				int pIdx = t+node->nodeData.tStart;
				for( int j=0 ; j<_polygons[pIdx].count ; j++ ) vertices[j] = Point3D< double >( _vertices[ _polygons[pIdx][j] ] );
				integrator.set( vertices , _polygons[pIdx].count );
				for( int i=0 ; i<Integrator::SampleNum( _polygons[pIdx].count ) ; i++ )
				{
					samples[ sampleCount ] = integrator[i];
					weights[ sampleCount ] = integrator.weight(i);
					if( MatrixType&LAP_MATRIX ) normals[ sampleCount ] = Point3D< double >( _baseNormals[ _polygons[pIdx].parent ] );
					sampleCount++;
				}
			}
			for( int s=0 ; s<sampleCount ; s++ ) weights[s] = sqrt( weights[s] );

			_setSampleSubValues< MatrixType&LAP_MATRIX >( node , samples , sampleCount , subValues , subDValues );

			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
						for( int s=0 ; s<sampleCount ; s++ )
						{
							if( MatrixType&DOT_MATRIX )
								fullValues[i][j][k][s] = subValues[i][0][s] * subValues[j][1][s] * subValues[k][2][s] * weights[ s ];
							if( MatrixType&LAP_MATRIX )
							{
								fullDValues[i][j][k][s][0] = subDValues[i][0][s] *  subValues[j][1][s] *  subValues[k][2][s];
								fullDValues[i][j][k][s][1] =  subValues[i][0][s] * subDValues[j][1][s] *  subValues[k][2][s];
								fullDValues[i][j][k][s][2] =  subValues[i][0][s] *  subValues[j][1][s] * subDValues[k][2][s];
								fullDValues[i][j][k][s] -= normals[s] * Point3D< double >::Dot( normals[s] , fullDValues[i][j][k][s] );
								fullDValues[i][j][k][s] *= weights[s];
							}
						}
			// Iterate over all pairs of neighbors
			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
						for( int ii=0 ; ii<(Primal?2:3) ; ii++ )
							for( int jj=0 ; jj<(Primal?2:3) ; jj++ )
								for( int kk=0 ; kk<(Primal?2:3) ; kk++ )
								{
									double dTemp = 0 , lTemp = 0;
									if( MatrixType&DOT_MATRIX )
										for( int s=0 ; s<sampleCount ; s++ ) dTemp += fullValues[i][j][k][s] * fullValues[ii][jj][kk][s];
									if( MatrixType&LAP_MATRIX )
										for( int s=0 ; s<sampleCount ; s++ ) lTemp += Point3D< double >::Dot( fullDValues[i][j][k][s] , fullDValues[ii][jj][kk][s] );
									eM[idx][ Primal?(i*4+j*2+k):(i*9+j*3+k) ][ Primal?(ii*4+jj*2+kk):(ii*9+jj*3+kk) ] = dTemp*dWeight + lTemp*lWeight;
								}
		}
		FreePointer( samples );
		FreePointer( weights );
		FreePointer( normals );
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<3 ; j++ ) { FreePointer( subValues[i][j] ) ; FreePointer( subDValues[i][j] ); }
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<(Primal?2:3) ; j++ ) for( int k=0 ; k<(Primal?2:3) ; k++ ) { FreePointer( fullValues[i][j][k] ) ; FreePointer( fullDValues[i][j][k] ); }
	}
	if( progress ) delete bar;

	MemoryUsage();
	return 1;
}
template< class Real , bool Primal >
template< class V , class HighPrecisionV , class Function , int MatrixType , class Integrator >
int MeshOctree< Real , Primal >::_getElementVector( const std::vector< Function >& functions , ElementVector< V >& integralValues , bool progress , Real dWeight , Real lWeight ) const
{
	std::vector< const Node* > nodes;
	nodes.resize( _geometryDimension[_maxDepth] );
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) )
		if( node->depth()==_maxDepth && node->nodeData.geometryIndex!=-1 )
			nodes[node->nodeData.geometryIndex] = node;

	integralValues.resize( nodes.size() );
	int max = _maxSamplesPerNode< Integrator >( );

	ProgressBar* bar=NULL;
	if( progress ) bar = new ProgressBar( 20 , _polygons.size() , "Setting Vector Integrals" );
#pragma omp parallel for num_threads( _threads )
	for( int i=0 ; i<_threads ; i++ )
	{
		Point3D< double > vertices[ PolygonData::MAX_VERTICES ];
		Integrator integrator;
		int sampleCount;
		Pointer( Point3D< double > ) samples;
		Pointer( HighPrecisionV ) values;
		Pointer( Gradient3D< HighPrecisionV > ) grads;
		Pointer( double ) weights;
		Pointer( double )  subValues[(Primal?2:3)][3];
		Pointer( double ) subDValues[(Primal?2:3)][3];
		Pointer( double )             fullValues[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)];
		Pointer( Point3D< double > ) fullDValues[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)];
		samples = AllocPointer< Point3D< double > >( max );
		weights = AllocPointer< double >( max );
		values  = AllocPointer< HighPrecisionV >( max );
		grads   = AllocPointer< Gradient3D< HighPrecisionV > >( max );

		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<3 ; j++ ) subValues[i][j] = AllocPointer< double >( max ) , subDValues[i][j] = AllocPointer< double >( max );
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<(Primal?2:3) ; j++ ) for( int k=0 ; k<(Primal?2:3) ; k++ ) fullValues[i][j][k] = AllocPointer< double >( max ) , fullDValues[i][j][k] = AllocPointer< Point3D< double > >( max );
		for( int j=(nodes.size()*i) / _threads ; j<(nodes.size()*(i+1)) / _threads ; j++ )
		{
			const Node* node = nodes[j];
			int idx = j;

			integralValues[idx].clear();

			sampleCount = 0;
			for( int t=0 ; t<node->nodeData.tEnd-node->nodeData.tStart ; t++ )
			{
				if( progress ) bar->update( i==0 );
				int pIdx = t+node->nodeData.tStart;
				for( int j=0 ; j<_polygons[pIdx].count ; j++ ) vertices[j] = Point3D< double >( _vertices[ _polygons[pIdx][j] ] );
				integrator.set( vertices , _polygons[pIdx].count );
				for( int i=0 ; i<Integrator::SampleNum( _polygons[pIdx].count ) ; i++ )
				{
					samples[ sampleCount ] = integrator[i];
					weights[ sampleCount ] = integrator.weight(i);

					if( MatrixType&LAP_MATRIX )
					{
						Point3D< double > normal = Point3D< double >( _baseNormals[ _polygons[pIdx].parent ] );
						HighPrecisionV dot;
						dot *= 0;
						Gradient3D< V > g = functions[_polygons[pIdx].parent].gradient( Point3D< typename V::R >( samples[sampleCount] ) );
						for( int c=0 ; c<3 ; c++ ) grads[sampleCount][c] = HighPrecisionV( g[c] );
						for( int c=0 ; c<3 ; c++ ) dot += grads[sampleCount][c] * normal[c];
						for( int c=0 ; c<3 ; c++ ) grads[sampleCount][c] -= dot * normal[c];
						grads[sampleCount] *= weights[sampleCount];
					}
					if( MatrixType&DOT_MATRIX ) values[sampleCount] = HighPrecisionV( functions[_polygons[pIdx].parent]( Point3D< typename V::R >( samples[sampleCount] ) ) ) * weights[sampleCount];
					sampleCount++;
				}
			}
			_setSampleSubValues< MatrixType&LAP_MATRIX >( node , samples , sampleCount , subValues , subDValues );

			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
						for( int s=0 ; s<sampleCount ; s++ )
						{
							if( MatrixType&DOT_MATRIX )
								fullValues[i][j][k][s] = subValues[i][0][s] * subValues[j][1][s] * subValues[k][2][s];
							if( MatrixType&LAP_MATRIX )
							{
								fullDValues[i][j][k][s][0] = subDValues[i][0][s] *  subValues[j][1][s] *  subValues[k][2][s];
								fullDValues[i][j][k][s][1] =  subValues[i][0][s] * subDValues[j][1][s] *  subValues[k][2][s];
								fullDValues[i][j][k][s][2] =  subValues[i][0][s] *  subValues[j][1][s] * subDValues[k][2][s];
							}
						}
			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
					{
						HighPrecisionV dTemp , lTemp;
						dTemp *= 0 , lTemp *= 0;
						if( MatrixType&DOT_MATRIX )
							for( int s=0 ; s<sampleCount ; s++ ) dTemp += values[s] * fullValues[i][j][k][s];
						if( MatrixType&LAP_MATRIX )
							for( int s=0 ; s<sampleCount ; s++ ) for( int c=0 ; c<3 ; c++ ) lTemp += grads[s][c] * fullDValues[i][j][k][s][c];
						integralValues[idx][ Primal?(i*4+j*2+k):(i*9+j*3+k) ] = V( dTemp*dWeight + lTemp*lWeight );
					}
		}
		FreePointer( samples );
		FreePointer( weights );
		FreePointer( values  );
		FreePointer( grads   );
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<3 ; j++ ) { FreePointer( subValues[i][j] ) ; FreePointer( subDValues[i][j] ); }
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<(Primal?2:3) ; j++ ) for( int k=0 ; k<(Primal?2:3) ; k++ ) { FreePointer( fullValues[i][j][k] ) ; FreePointer( fullDValues[i][j][k] ); }
	}
	if( progress ) delete bar;

	MemoryUsage();
	return 1;
}

template< class Real , bool Primal >
template< class Integrator >
int MeshOctree< Real , Primal >::_getLaplacianMatrixIntegrals( ElementMatrix& minEM , ElementMatrix& maxEM ,
															   const std::vector< Point2D< Real > >& _baseCurvatureValues , const std::vector< Point2D< Point3D< Real > > > _baseCurvatureDirections ,
															   bool progress ) const
{
	std::vector< const Node* > nodes;
	nodes.resize( _geometryDimension[_maxDepth] );
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) )
		if( node->depth()==_maxDepth && node->nodeData.geometryIndex!=-1 )
			nodes[node->nodeData.geometryIndex] = node;

	minEM.resize( nodes.size() );
	maxEM.resize( nodes.size() );
	int max = _maxSamplesPerNode< Integrator >( );

	ProgressBar* bar=NULL;
	if( progress ) bar = new ProgressBar( 20 , _polygons.size() , "Setting Anisotropic Laplacian Integrals" );
#pragma omp parallel for num_threads( _threads )
	for( int i=0 ; i<_threads ; i++ )
	{
		Point3D< double > vertices[ PolygonData::MAX_VERTICES ];
		Integrator integrator;
		int sampleCount;
		Pointer( Point3D< double > ) samples;
		Pointer( Point3D< double > ) normals;
		Pointer( Point3D< double > ) tanMin;
		Pointer( Point3D< double > ) tanMax;
		Pointer( double )  subValues[Primal?2:3][3];
		Pointer( double ) subDValues[Primal?2:3][3];
		Pointer( double ) weights;
		Pointer( double ) fullValuesTMin[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)];
		Pointer( double ) fullValuesTMax[(Primal?2:3)][(Primal?2:3)][(Primal?2:3)];

		tanMin  = AllocPointer< Point3D< double > >( max );
		tanMax  = AllocPointer< Point3D< double > >( max );
		samples = AllocPointer< Point3D< double > >( max );
		normals = AllocPointer< Point3D< double > >( max );
		for( int ii=0 ; ii<(Primal?2:3) ; ii++ ) for( int jj=0 ; jj<3 ; jj++ ) subValues[ii][jj] = AllocPointer< double >( max ) , subDValues[ii][jj] = AllocPointer< double >( max );
		for( int ii=0 ; ii<(Primal?2:3) ; ii++ ) for( int jj=0 ; jj<(Primal?2:3) ; jj++ ) for( int kk=0 ; kk<(Primal?2:3) ; kk++ ) fullValuesTMin[ii][jj][kk] = AllocPointer< double >( max ) , fullValuesTMax[ii][jj][kk] = AllocPointer< double >( max );
		weights = AllocPointer< double >( max );
		for( int j=(nodes.size()*i) / _threads ; j<(nodes.size()*(i+1)) / _threads ; j++ )
		{
			const Node* node = nodes[j];
			int idx = j;

			minEM[idx].clear() , maxEM[idx].clear();

			sampleCount = 0;
			for( int t=0 ; t<node->nodeData.tEnd-node->nodeData.tStart ; t++ )
			{
				if( progress ) bar->update( i==0 );
				int pIdx = t+node->nodeData.tStart;
				for( int j=0 ; j<_polygons[pIdx].count ; j++ ) vertices[j] = Point3D< double >( _vertices[ _polygons[pIdx][j] ] );
				integrator.set( vertices , _polygons[pIdx].count );
				for( int i=0 ; i<Integrator::SampleNum( _polygons[pIdx].count ) ; i++ )
				{
					samples[ sampleCount ] = integrator[i];
					weights[ sampleCount ] = integrator.weight(i);
					normals[ sampleCount ] = Point3D< double > ( _baseNormals[ _polygons[pIdx].parent ] );
					tanMin [ sampleCount ] = Point3D< double > ( _baseCurvatureDirections[ _polygons[pIdx].parent ][MIN_CURVATURE] );
					tanMax [ sampleCount ] = Point3D< double > ( _baseCurvatureDirections[ _polygons[pIdx].parent ][MAX_CURVATURE] );
					sampleCount++;
				}
			}
			
			for( int s=0 ; s<sampleCount ; s++ ) weights[s] = sqrt( weights[s] );

			_setSampleSubValues< true >( node , samples , sampleCount , subValues , subDValues );

			// Iterate over all pairs of neighbors
			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
						for( int s=0 ; s<sampleCount ; s++ )
						{
							Point3D< double > gradient;
							gradient[0] = subDValues[i][0][s] *  subValues[j][1][s] *  subValues[k][2][s];
							gradient[1] =  subValues[i][0][s] * subDValues[j][1][s] *  subValues[k][2][s];
							gradient[2] =  subValues[i][0][s] *  subValues[j][1][s] * subDValues[k][2][s];
							fullValuesTMin[i][j][k][s] = Point3D< double >::Dot( gradient , tanMin[s] ) * weights[s];
							fullValuesTMax[i][j][k][s] = Point3D< double >::Dot( gradient , tanMax[s] ) * weights[s];
						}
			for( int i=0 ; i<(Primal?2:3) ; i++ )
				for( int j=0 ; j<(Primal?2:3) ; j++ )
					for( int k=0 ; k<(Primal?2:3) ; k++ )
						for( int ii=0 ; ii<(Primal?2:3) ; ii++ )
							for( int jj=0 ; jj<(Primal?2:3) ; jj++ )
								for( int kk=0 ; kk<(Primal?2:3) ; kk++ )
								{
									double tempTMin=0, tempTMax=0;
									for( int s=0 ; s<sampleCount ; s++ )
										tempTMin += fullValuesTMin[i][j][k][s] * fullValuesTMin[ii][jj][kk][s],
										tempTMax += fullValuesTMax[i][j][k][s] * fullValuesTMax[ii][jj][kk][s];
									minEM[idx][Primal?i*4+j*2+k:i*9+j*3+k][Primal?ii*4+jj*2+kk:ii*9+jj*3+kk] = Real( tempTMin );
									maxEM[idx][Primal?i*4+j*2+k:i*9+j*3+k][Primal?ii*4+jj*2+kk:ii*9+jj*3+kk] = Real( tempTMax );
								}
		}
		FreePointer( samples );
		FreePointer( normals );
		FreePointer( tanMin );
		FreePointer( tanMax );
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<3 ; j++ ) { FreePointer( subValues[i][j] ) ; FreePointer( subDValues[i][j] ); }
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<(Primal?2:3) ; j++ ) for( int k=0 ; k<(Primal?2:3) ; k++ ) { FreePointer( fullValuesTMin[i][j][k] ) ; FreePointer( fullValuesTMax[i][j][k] ); }
		FreePointer( weights );
	}
	if( progress ) delete bar;

	MemoryUsage();
	return 1;
}
template< class Real , bool Primal >
template< class Vertex >
void MeshOctree< Real , Primal >::setMatrixTrackingData( TrackingData< Vertex >& tData , bool progress ) const
{
	if( Primal ) _setMatrixTrackingData< Vertex , PrimalPolygonIntegrator >( tData , progress );
	else         _setMatrixTrackingData< Vertex ,   DualPolygonIntegrator >( tData , progress );
#pragma omp parallel for num_threads( _threads )
	for( int t=0 ; t<_threads ; t++ )
	{
		double sValues[Primal?2:3][3] , sDValues[Primal?2:3][3];
		for( int i=(tData.mSamples.size()*t) / _threads ; i<(tData.mSamples.size()*(t+1))/_threads ; i++ )
		{
			tData.mSamples[i].weight *= _scale * _scale;
			tData.mSamples[i].getSubValues( sValues , sDValues , true );
			for( int j=0 ; j<(Primal?2:3) ; j++ ) for( int k=0 ; k<3 ; k++ ) sDValues[j][k] = double( sValues[j][k] ) / _scale;
			tData.mSamples[i].setSubValues( sValues , sDValues , true );
		}
	}
}
template< class Real , bool Primal >
template< class Vertex , class Integrator >
void MeshOctree< Real , Primal >::_setMatrixTrackingData( TrackingData< Vertex >& tData , bool progress ) const
{
	int max = _maxSamplesPerNode< Integrator >();
	std::vector< const Node* > nodes;
	nodes.resize( _geometryDimension[_maxDepth] );
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) )
		if( node->depth()==_maxDepth && node->nodeData.geometryIndex!=-1 )
			nodes[node->nodeData.geometryIndex] = node;
	tData.sampleStart.resize( nodes.size()+1 );

	int sampleCount = 0;
	for( int i=0 ; i<nodes.size() ; i++ )
	{
		tData.sampleStart[i] = sampleCount;
		for( int t=nodes[i]->nodeData.tStart ; t<nodes[i]->nodeData.tEnd ; t++ )
			sampleCount += Integrator::SampleNum( _polygons[t].count );
	}
	tData.sampleStart.back() = sampleCount;
	tData.mSamples.resize( sampleCount );

	tData.tangents.resize( _baseNormals.size() );
#pragma omp parallel for num_threads( _threads )
	for( int i=0 ; i<_baseNormals.size() ; i++ )
	{
		int mi = 0;
		if( fabs(_baseNormals[i][mi]*_baseNormals[i][mi]) > fabs(_baseNormals[i][1]*_baseNormals[i][1]) ) mi=1;
		if( fabs(_baseNormals[i][mi]*_baseNormals[i][mi]) > fabs(_baseNormals[i][2]*_baseNormals[i][2]) ) mi=2;
		Point3D< double > t1 , t2;
		t1[(mi+1)%3] = _baseNormals[i][(mi+2)%3];
		t1[(mi+2)%3] =-_baseNormals[i][(mi+1)%3];
		t1[ mi     ] = 0;
		t2 = Point3D< double >::CrossProduct( Point3D< double >(_baseNormals[i]) , t1 );
		t1 /= Length( t1 );
		t2 /= Length( t2 );
		tData.tangents[i].first  = Point3D< Real >( t1 );
		tData.tangents[i].second = Point3D< Real >( t2 );
	}

	ProgressBar* bar=NULL;
	if( progress ) bar = new ProgressBar( 20 , _polygons.size() , "Setting Tracking Data" );

#pragma omp parallel for num_threads( _threads )
	for( int thread=0 ; thread<_threads ; thread++ )
	{
		Point3D< Real > vertices[ PolygonData::MAX_VERTICES ];
		Integrator integrator;
		Pointer( Point3D< double > ) samples;
		Pointer( double )  subValues[(Primal?2:3)][3];
		Pointer( double ) subDValues[(Primal?2:3)][3];

		int sampleNum = max;

		samples  = AllocPointer< Point3D< double > >( sampleNum );
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<3 ; j++ ) subValues[i][j] = AllocPointer< double>( sampleNum ) , subDValues[i][j] = AllocPointer< double >( sampleNum );
		int sampleStart = 0;
		for( int idx=0 ; idx<(nodes.size()*thread) / _threads ; idx++ )
			for( int t=nodes[idx]->nodeData.tStart ; t<nodes[idx]->nodeData.tEnd ; t++ ) sampleStart += Integrator::SampleNum( _polygons[t].count );
		for( int idx=(nodes.size()*thread) / _threads ; idx<(nodes.size()*(thread+1)) / _threads ; idx++ )
		{
			const Node* node = nodes[idx];
			int dd , off[3];
			node->depthAndOffset( dd , off );
			int sampleCount = 0;
			for( int t=0 ; t<node->nodeData.tEnd-node->nodeData.tStart ; t++ )
			{
				if( progress ) bar->update( thread==0 );
				int pIdx = node->nodeData.tStart + t;
				for( int j=0 ; j<_polygons[pIdx].count ; j++ ) vertices[j] = _vertices[ _polygons[pIdx][j] ];
				integrator.set( vertices , _polygons[pIdx].count );
				for( int i=0 ; i<Integrator::SampleNum( _polygons[pIdx].count ) ; i++ )
				{
					tData.mSamples[sampleStart+sampleCount].weight = integrator.weight(i);
					tData.mSamples[sampleStart+sampleCount].index = _polygons[pIdx].parent;
					samples[ sampleCount ] = integrator[i];
					sampleCount++;
				}
			}

			_setSampleSubValues< true >( node , samples , sampleCount , subValues , subDValues );
			for( int s=0 ; s<sampleCount ; s++ )
			{
				double sValues[Primal?2:3][3] , sDValues[Primal?2:3][3];
				for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int c=0 ; c<3 ; c++ ) sValues[i][c] = subValues[i][c][s] , sDValues[i][c] = subDValues[i][c][s];
				tData.mSamples[sampleStart+s].setSubValues( sValues , sDValues , false );
			}
			sampleStart += sampleCount;
		}
		FreePointer( samples );
		for( int i=0 ; i<(Primal?2:3) ; i++ ) for( int j=0 ; j<3 ; j++ ) { FreePointer( subValues[i][j] ) ; FreePointer( subDValues[i][j] ); }
	}
	if( progress ) delete bar;
}
template< class Real , bool Primal >
template< class Integrator >
int MeshOctree< Real , Primal >::_maxSamplesPerNode( void ) const
{
	int max = 0;
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) ) if( node->depth()==_maxDepth && node->nodeData.geometryIndex!=-1 )
	{
		int c=0;
		for( int t=node->nodeData.tStart ; t<node->nodeData.tEnd ; t++ ) c += Integrator::SampleNum( _polygons[t].count );
		max = std::max< int >( max , c );
	}
	return max;
}