/* -*- C++ -*-
Copyright (c) 2006, Michael Kazhdan and Matthew Bolitho
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/

#include "MemoryUsage.h"
#include "CmdLineParser.h"
#include "LaplacianMatrix1D.h"
#include "TriangleIntegrals.h"
#include "LineIntegrals.h"
#include "Util.h"

//////////////////////
// MeshTreeNodeData //
//////////////////////
template< class Real >
MeshTreeNodeData< Real >::MeshTreeNodeData( void )
{
	geometryIndex = basisIndex = tempIndex = -1;
	tStart = tEnd = 0;
}
template< class Real > MeshTreeNodeData< Real >::~MeshTreeNodeData(void)	{	;	}


////////////////
// MeshOctree //
////////////////
template< class Real , bool Primal >
double MeshOctree< Real , Primal >::maxMemoryUsage=0;

template< class Real , bool Primal >
double MeshOctree< Real , Primal >::MemoryUsage( void )
{
	double mem = double( MemoryInfo::Usage() ) / (1<<20);
	if(mem>maxMemoryUsage) maxMemoryUsage=mem;
	return mem;
}

template< class Real , bool Primal >
MeshOctree< Real , Primal >::MeshOctree( void )
{
}
class DataSortIndex
{
public:
	int sortIndex , dataIndex;
	static int Compare( const void* v1 , const void* v2 )
	{
		DataSortIndex* si1 = (DataSortIndex*)v1;
		DataSortIndex* si2 = (DataSortIndex*)v2;
		return si1->sortIndex-si2->sortIndex;
	}
};
class DataSortIndex3D
{
public:
	int x , y , z , dataIndex;
	static int Compare( const void* v1 , const void* v2 )
	{
		DataSortIndex3D* si1 = (DataSortIndex3D*)v1;
		DataSortIndex3D* si2 = (DataSortIndex3D*)v2;
		if     ( si1->x<si2->x ) return -1;
		else if( si1->x>si2->x ) return  1;
		else if( si1->y<si2->y ) return -1;
		else if( si1->y>si2->y ) return  1;
		else                     return si1->z-si2->z;
	}
};

template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_addTriangles( const std::vector< TriangleIndex >& triangles , int depth )
{
	class VoxelIndex
	{
	public:
		int idx[3];
		bool operator == ( const VoxelIndex& i ) const { return (idx[0]==i.idx[0]) && (idx[1]==i.idx[1]) && (idx[2]==i.idx[2]); }
	};

	int nodeCount=0;
	std::vector< DataSortIndex > triangleIndices;
	std::vector< int > triangleStart;
	triangleIndices.resize( triangles.size() );

	for( int i=0 ; i<triangles.size() ; i++ )
	{
		const TriangleIndex& tIndex = triangles[i];

		// Compute the finest depth at which the triangle is entirely contained within a node
		int tDepth;
		VoxelIndex idx[3];
		for( tDepth=0 ; tDepth<depth ; tDepth++ )
		{
			for( int j=0 ; j<3 ; j++ ) for( int c=0 ; c<3 ; c++ ) idx[j].idx[c] = int( _vertices[ tIndex[j] ][c] * (1<<(tDepth+1)) );
			if( !(idx[0]==idx[1]) || !(idx[0]==idx[2]) ) break;
		}

		// Add the triangle index to every node that completely contains the triangle
		for( int j=0 ; j<3 ; j++ ) for( int c=0 ; c<3 ; c++ ) idx[j].idx[c] = int( _vertices[ tIndex[j] ][c] * (1<<tDepth) );
		Point3D< Real > p;
		for( int c=0 ; c<3 ; c++ ) p[c] = ( Real(idx[0].idx[c])+Real( 0.5 ) ) / (1<<tDepth);
		Node* node = &_tree;
		while( node->depth()<tDepth )
		{
			Point3D< Real > center;
			Real width;
			int x , y , z;
			x = y = z = 0;
			node->centerAndWidth( center , width );
			if( p[0]>center[0] ) x=1;
			if( p[1]>center[1] ) y=1;
			if( p[2]>center[2] ) z=1;

			if( !node->children ) node->initChildren();
			node = node->children + Cube::CornerIndex( x , y , z );
		}
		if( node->nodeData.basisIndex==-1 ) node->nodeData.basisIndex = nodeCount++;
		triangleIndices[i].dataIndex = i;
		triangleIndices[i].sortIndex = node->nodeData.basisIndex;
	}
	qsort( &triangleIndices[0] , triangleIndices.size() , sizeof( DataSortIndex ) , DataSortIndex::Compare );

	triangleStart.resize( nodeCount+1 );
	nodeCount=-1;
	for( int i=0 ; i<triangles.size() ; i++ ) while( triangleIndices[i].sortIndex!=nodeCount ) triangleStart[++nodeCount] = i;
	triangleStart[++nodeCount] = (int) triangles.size();

	std::vector< PolygonData > initialPolygons;
	initialPolygons.resize( triangles.size() );
	for( int i=0 ; i<triangles.size() ; i++ )
	{
		initialPolygons[i].parent = triangleIndices[i].dataIndex;
		initialPolygons[i].count = 3;
		for( int j=0 ; j<3 ; j++ ) initialPolygons[i][j] = triangles[ triangleIndices[i].dataIndex ][j];
	}

	for( Node* node = _tree.nextNode() ; node ; node = _tree.nextNode( node ) )
		if( node->nodeData.basisIndex!=-1 )
		{
			node->nodeData.tStart = triangleStart[ node->nodeData.basisIndex   ];
			node->nodeData.tEnd   = triangleStart[ node->nodeData.basisIndex+1 ];
			node->nodeData.basisIndex  = - 1;
		}

	std::vector< PolygonData > polygons;
#if KEEP_MANIFOLD
	hash_map< long long , int > vTable;
	_addPolygons( &_tree , initialPolygons , polygons , depth , &vTable , _polygons );
#else // !KEEP_MANIFOLD
	_addPolygons( &_tree , initialPolygons , polygons , depth , NULL    , _polygons );
#endif // KEEP_MANIFOLD
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_addPolygons( Node* node , const std::vector< PolygonData >& initialPolygons , std::vector< PolygonData >& polygons , int depth ,
												hash_map< long long , int >* vTable , std::vector< PolygonData >& splitPolygons )
{
	int start = node->nodeData.tStart;
	int end   = node->nodeData.tEnd;
	node->nodeData.tStart = node->nodeData.tEnd = 0;

	int d = node->depth();
	// If we're deep enough, return
	if( d>depth ) return;
	// If there is nothing to add, return
	if( !polygons.size() && start==end && !node->children ) return;

	// Add the triangles from the polygons
	if( d==depth )
	{
		node->nodeData.tStart = (int)splitPolygons.size();

		size_t sz = splitPolygons.size();
		int count = polygons.size();
		splitPolygons.resize( sz + (end-start) + count );

		for( int i=0 ; i<polygons.size() ; i++ ) splitPolygons[sz++] = polygons[i];
		for( int i=start ; i<end ; i++ )         splitPolygons[sz++] = initialPolygons[i];
		node->nodeData.tEnd = (int)splitPolygons.size();
		return;
	}

	// Split up the polygons and pass them on to the children

	Point3D< Real > axis[3] , center;
	Real width;
	node->centerAndWidth( center , width );
	for( int i=0 ; i<3 ; i++ )
	{
		axis[i] *= 0;
		axis[i][i] = 1;
	}
	std::vector< PolygonData > subPolygons[ Cube::CORNERS ];

	if( !node->children ) node->initChildren();

	PolygonData temp;
	Real values[9];
	for( int i=0 ; i<polygons.size() + (end-start) ; i++ )
	{
		if( i<polygons.size() ) temp = polygons[i];
		else                    temp = initialPolygons[ i-polygons.size()+start ];
		PolygonData xPolygons[2] , yPolygons[2] , zPolygons[2];
		zPolygons[0].parent = zPolygons[1].parent = temp.parent;
		SplitPolygon( _vertices , vTable , temp.vertices , values , temp.count , axis[0] , center[0] , xPolygons[0].vertices , xPolygons[0].count , xPolygons[1].vertices , xPolygons[1].count );
		for( int x=0 ; x<2 ; x++ )
		{
			if( !xPolygons[x].count ) continue;
			SplitPolygon( _vertices , vTable , xPolygons[x].vertices , values , xPolygons[x].count , axis[1] , center[1] , yPolygons[0].vertices , yPolygons[0].count , yPolygons[1].vertices , yPolygons[1].count );
			for( int y=0 ; y<2 ; y++ )
			{
				if( !yPolygons[y].count ) continue;
				SplitPolygon( _vertices , vTable , yPolygons[y].vertices , values , yPolygons[y].count , axis[2] , center[2] , zPolygons[0].vertices , zPolygons[0].count , zPolygons[1].vertices , zPolygons[1].count );
				for( int z=0 ; z<2 ; z++ )
				{
					if( !zPolygons[z].count ) continue;
					int idx = Cube::CornerIndex( x , y , z );
					subPolygons[idx].push_back( zPolygons[z] );
				}
			}
		}
	}

	for( int c=0 ; c<Cube::CORNERS ; c++ ) _addPolygons( node->children+c , initialPolygons , subPolygons[c] , depth , vTable , splitPolygons );
}
long long InterleaveBits( int x , int y )
{
	long long key = 0;
	for( int i=0 ; i<32 ; i++ ) key |= ( ( x & (1<<i) )<<i ) | ( ( y & (1<<i) )<<(i+1) );
	return key;
}
struct NodeSortData
{
	int height;
	long long index;
};
int MyCompareFunction( const void* v1 , const void* v2 )
{
	const  std::pair< int , NodeSortData > *p1 , *p2;
	p1 = ( std::pair< int , NodeSortData >* )v1;
	p2 = ( std::pair< int , NodeSortData >* )v2;
	if      ( p1->second.height<p2->second.height ) return -1;
	else if ( p1->second.height>p2->second.height ) return  1;
	else
		if     ( p1->second.index<p2->second.index ) return  1;
		else if( p1->second.index>p2->second.index ) return -1;
	return 0;
}

template< class  Real , bool Primal >
void MeshOctree< Real , Primal >::_sortIndices( void )
{
	std::vector< std::vector< std::pair< int , NodeSortData > > > heights;
	std::vector< std::vector< int > > map;
	std::vector< int > count;
	heights.resize   ( _maxDepth+1 );
	map.resize       ( _maxDepth+1 );
	count.resize     ( _maxDepth+1 );

	for( int d=0 ; d<=_maxDepth ; d++ ) count[d] = 0;

	for( Node* node = _tree.nextNode( ) ; node ; node=_tree.nextNode( node ) ) if( node->nodeData.basisIndex!=-1 ) count[node->depth()]++;
	for( int d=0 ; d<=_maxDepth ; d++ ) heights[d].resize( count[d] ) , map[d].resize( count[d] );

	// Set the sorting information
	for( Node* node = _tree.nextNode( ) ; node ; node=_tree.nextNode( node ) )
		if( node->nodeData.basisIndex!=-1 )
		{
			int d , off[3];
			node->depthAndOffset( d , off );
			heights[d][ node->nodeData.basisIndex ].first = node->nodeData.basisIndex;
			heights[d][ node->nodeData.basisIndex ].second.height = off[2];
			heights[d][ node->nodeData.basisIndex ].second.index = InterleaveBits( off[0] , off[1] );
		}
	// Sort the nodes
	for( int d=0 ; d<=_maxDepth ; d++ )
	{
		qsort( &heights[d][0] , heights[d].size() , sizeof( std::pair< int , NodeSortData > ) , MyCompareFunction );
		for( int i=0 ; i<heights[d].size() ; i++ ) map[d][ heights[d][i].first ] = i;
	}
	for( Node* node = _tree.nextNode( ) ; node ; node=_tree.nextNode( node ) )
		if( node->nodeData.basisIndex!=-1 ) node->nodeData.basisIndex = map[node->depth()][ node->nodeData.basisIndex ];
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_setSolverData( std::vector< SparseMatrix< Real , int > >& DS ) const
{
	DS.resize( _maxDepth+1-_depthOffset );
	for( int d=1 ; d<=_maxDepth-_depthOffset ; d++ ) _getDownSampleMatrix( d+_depthOffset , DS[d] ) , DS[d].MakeContiguous( );
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_setSolverData( std::vector< std::vector< int > >& slicesPerThread , std::vector< std::vector< int > >& entriesPerSlice , int pDepth , int threads ) const
{
	std::vector< int > count;
	std::vector< std::vector< int > > sliceCount;
	sliceCount.resize( _maxDepth+1 );
	count.resize     ( _maxDepth+1 );

	slicesPerThread.resize( _maxDepth+1-_depthOffset );
	entriesPerSlice.resize( _maxDepth+1-_depthOffset );

	for( int d=pDepth ; d<=_maxDepth ; d++ )
	{
		count[d] = 0;
		int slices = Primal ? (1<<d)+1 : (1<<d);
		sliceCount[d].resize( slices );
		for( int i=0 ; i<sliceCount[d].size() ; i++ ) sliceCount[d][i] = 0;
		slicesPerThread[d-_depthOffset].resize( threads );
	}

	for( const Node* node = _tree.nextNode( ) ; node ; node=_tree.nextNode( node ) )
		if( node->nodeData.basisIndex!=-1 && node->depth()>=pDepth ) count[node->depth()]++;

	// Set the information associating an index to each node
	for( const Node* node = _tree.nextNode( ) ; node ; node=_tree.nextNode( node ) )
		if( node->nodeData.basisIndex!=-1 && node->depth()>=pDepth )
		{
			int d , off[3];
			node->depthAndOffset( d , off );
			sliceCount[d][ off[2] ]++;
		}

	for( int d=_maxDepth ; d>=pDepth ; d-- )
	{
		int flagIndex = 0;
		int chunkSize = count[d]/threads;
		int slices = Primal ? (1<<d)+1 : (1<<d);
		for( int i=0 ; i<sliceCount[d].size() ; i++ ) entriesPerSlice[d-_depthOffset].push_back( sliceCount[d][i] );

		if( d==_maxDepth )
		{
			count[d]=0;
			for( int i=0 ; i<entriesPerSlice[d-_depthOffset].size() ; i++)
			{
				count[d] += entriesPerSlice[d-_depthOffset][i];
				if( count[d]>=(flagIndex+1)*chunkSize ) slicesPerThread[d-_depthOffset][flagIndex++] = i+1;		
			}
		}
		else for( int i=0 ; i<threads ; i++ ) slicesPerThread[d-_depthOffset][i] = ( slicesPerThread[d+1-_depthOffset][i] * slices ) / slicesPerThread[d+1-_depthOffset][threads-1];
		slicesPerThread[d-_depthOffset][threads-1] = slices;
	}
	for( int d=_maxDepth ; d>=pDepth ; d-- )  for( int i=threads-1 ; i>0 ; i-- ) slicesPerThread[d-_depthOffset][i] -= slicesPerThread[d-_depthOffset][i-1];
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_setSolverData( std::vector< ElementRestrictionOperator >& DS ) const
{
	DS.resize( _maxDepth+1-_depthOffset );
	for( int d=1 ; d<=_maxDepth-_depthOffset ; d++ ) _setElementRestriction( DS[d] , d+_depthOffset );
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_setSolverData( std::vector< ElementToBasisOperator< true > >& e2b ) const
{
	e2b.resize( _maxDepth+1-_depthOffset );
	for( int d=0 ; d<=_maxDepth-_depthOffset ; d++ ) _setElementToBasisMatrix( e2b[d].e2b , d+_depthOffset ) , e2b[d].init( _threads );
}

template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::setSolver( MultigridSolver< Real , C , int >& solver ) const
{
	_setSolverData( solver.DS );
	solver.Init( _basisDimension.back() );
}
template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::setSolver( ParallelMultigridSolver< Real , C , int >& solver , int pDepth , int threads , int blockSize ) const
{
	pDepth += _depthOffset;
	solver.sliceDependence = Primal ? 1 : 2;
	_setSolverData( solver.DS );
	_setSolverData( solver.slicesPerThread , solver.entriesPerSlice , pDepth , threads );
	solver.Init( blockSize , _basisDimension.back() );
}
template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::setSolver( ParallelSolver< C >& solver , int pDepth , int threads , int blockSize , bool setDSFromElements ) const
{
	pDepth += _depthOffset;

	solver.sliceDependence = Primal ? 1 : 2;
	if( !setDSFromElements ) _setSolverData( solver.DS );
	_setSolverData( solver.slicesPerThread , solver.entriesPerSlice , pDepth , threads );
	_setSolverData( solver.e2b );
	_setSolverData( solver.restriction );
	solver.Init( blockSize , _basisDimension.back() , setDSFromElements );
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_setBaseCurvatures( std::vector< Point2D< Real > >& baseCurvatureValues , std::vector< Point2D< Point3D< Real > > >& baseCurvatureDirections , double normalize ) const
{
// Computing Per-Face Curvature
// Code partially tailored from TriMesh2: http://www.cs.princeton.edu/gfx/proj/trimesh2/
// Author: Szymon Rusinkiewicz
	baseCurvatureDirections.resize( _baseTriangles.size() );
	baseCurvatureValues.resize( _baseTriangles.size() );
	int maxV = 0;
	std::vector< Point3D< Real > > vNormals;
	for( int i=0 ; i<_baseTriangles.size() ; i++ ) for( int j=0 ; j<3 ; j++ ) if( _baseTriangles[i][j]>maxV ) maxV = _baseTriangles[i][j];
	vNormals.resize( maxV+1 );
	for( int i=0 ; i<_baseTriangles.size() ; i++ )
	{
		double area = TriangleArea( _vertices, _baseTriangles[i] );
		Point3D< Real > normal = _baseNormals[i] * Real( area );
		for( int j=0 ; j<3 ; j++ ) vNormals[ _baseTriangles[i][j] ] += normal;
	}
#pragma omp parallel for
	for( int i=0 ; i<vNormals.size() ; i++ )
	{
		double l = Length( vNormals[i] );
		if( l ) vNormals[i] /= l;
		else    vNormals[i] *= 0;
	}

#pragma omp parallel for
	for( int i=0 ; i<_baseTriangles.size() ; i++ )
	{
		// build the n-t-b frame
		TriangleIndex tIndex = _baseTriangles[i];
		Point3D< double > e[3] = { _vertices[tIndex[2]] - _vertices[tIndex[1]],
								   _vertices[tIndex[0]] - _vertices[tIndex[2]],
								   _vertices[tIndex[1]] - _vertices[tIndex[0]] };
		Point3D< double > t = e[0];
		t /= Length( t );
		Point3D< double > n = Point3D< double >( _baseNormals[i] );
		n /= Length( n );
		Point3D< double > b = Point3D< double >::CrossProduct( n , t );
		b /= Length( b ); // <= should be ok to take off

		// build the linear system to solve
		double m[3] = { 0, 0, 0 };
		double w[3][3] = { {0,0,0}, {0,0,0}, {0,0,0} };
		for (int j = 0; j < 3; j++)
		{
			double u = Point3D< double >::Dot( e[j] , t );
			double v = Point3D< double >::Dot( e[j] , b );
			w[0][0] += u*u;
			w[0][1] += u*v;
			w[2][2] += v*v;
			Point3D< double > dn = vNormals[ tIndex[(j+2)%3] ] - vNormals[ tIndex[(j+1)%3] ];
			double dnu = Point3D< double >::Dot( dn , t );
			double dnv = Point3D< double >::Dot( dn , b );
			m[0] += dnu*u;
			m[1] += dnu*v + dnv*u;
			m[2] += dnv*v;
		}
		w[1][1] = w[0][0] + w[2][2];
		w[1][2] = w[0][1];

		// Solve by Least-Square 
		double diag[3];
		if( !ldltdc< double , 3 >( w, diag ) )
		{
#if 0
			printf( "ldltdc failed!\n" );
#endif
			continue;
		}		
		ldltsl< double , 3 >( w , diag , m , m );

		double disc = 4*m[1]*m[1] + m[0]*m[0] - 2*m[0]*m[2] + m[2]*m[2];
		if( disc<0 )
		{
			printf( "Negative discriminant: %g\n" , disc );
			disc = 0;
		}
		else disc = sqrt( disc );
		baseCurvatureValues[i][MIN_CURVATURE] = Real( 0.5 ) * ( m[0] + m[2] - disc );
		baseCurvatureValues[i][MAX_CURVATURE] = Real( 0.5 ) * ( m[0] + m[2] + disc );
		
		if( baseCurvatureValues[i][MIN_CURVATURE]==baseCurvatureValues[i][MAX_CURVATURE] ) 
		{
			baseCurvatureDirections[i][MIN_CURVATURE] = Point3D< Real >( t ); 
			baseCurvatureDirections[i][MAX_CURVATURE] = Point3D< Real >( b );
		}
		else
		{
			Real KminusM2, x1, x2;
			KminusM2 = (baseCurvatureValues[i][MAX_CURVATURE] - m[2]);
			//x1 = KminusM2	/ sqrt( m[1]*m[1] + KminusM2*KminusM2 );
			//x2 = m[1]		/ sqrt( m[1]*m[1] + KminusM2*KminusM2 );
			x1 = m[1]-m[2]+baseCurvatureValues[i][MAX_CURVATURE];
			x2 = m[1]-m[0]+baseCurvatureValues[i][MAX_CURVATURE];
			baseCurvatureDirections[i][MAX_CURVATURE]  = Point3D< Real >( ( t * x1 ) + ( b * x2 ) );
			baseCurvatureDirections[i][MAX_CURVATURE] /= sqrt( Point3D< Real >::SquareNorm( baseCurvatureDirections[i][MAX_CURVATURE] ) );

			baseCurvatureDirections[i][MIN_CURVATURE] = Point3D< Real >::CrossProduct( Point3D<Real>( n ) , baseCurvatureDirections[i][MAX_CURVATURE] );
			baseCurvatureDirections[i][MIN_CURVATURE] /= sqrt( Point3D< Real >::SquareNorm( baseCurvatureDirections[i][MIN_CURVATURE] ) );
		}
	}
	double var = 0;
	double sum = 0;
	for( int i=0 ; i<_baseTriangles.size() ; i++ )
	{
		double area = TriangleArea( _vertices, _baseTriangles[i] );
		sum += 2*area;
		var += (baseCurvatureValues[i][MIN_CURVATURE]*baseCurvatureValues[i][MIN_CURVATURE] + baseCurvatureValues[i][MAX_CURVATURE]*baseCurvatureValues[i][MAX_CURVATURE]) * area;
	}
	var = sqrt( var/sum );
	if( normalize>0 )
	{
		var *= normalize;
		for( int i=0 ; i<_baseTriangles.size() ; i++ )
			baseCurvatureValues[i][MIN_CURVATURE] /= var , baseCurvatureValues[i][MAX_CURVATURE] /= var;
	}
#if 0  // sanity test
	for( int i=0 ; i<baseCurvatureDirections.size() ; i++ )
	{
		if( fabs(1.0 - Point3D< Real >::SquareNorm( baseCurvatureDirections[i][MIN_CURVATURE] ) ) > 0.0000005 ) printf("case 1 failed! (%f) \n", Point3D< Real >::SquareNorm( baseCurvatureDirections[i][MIN_CURVATURE] ));
		if( fabs(1.0 - Point3D< Real >::SquareNorm( baseCurvatureDirections[i][MAX_CURVATURE] ) ) > 0.0000005 ) printf("case 2 failed! (%f) \n", Point3D< Real >::SquareNorm( baseCurvatureDirections[i][MAX_CURVATURE] ));
		if( fabs(Point3D< Real >::Dot( baseCurvatureDirections[i][MIN_CURVATURE], baseCurvatureDirections[i][MAX_CURVATURE]))	> 0.0000005 ) printf("case 3 failed! (%f) \n", Point3D< Real >::Dot( baseCurvatureDirections[i][MIN_CURVATURE], baseCurvatureDirections[i][MAX_CURVATURE]));
		if( fabs(baseCurvatureDirections[i][MIN_CURVATURE][0])==std::numeric_limits<Real>::infinity() || fabs(baseCurvatureDirections[i][MIN_CURVATURE][1])==std::numeric_limits<Real>::infinity() || fabs(baseCurvatureDirections[i][MIN_CURVATURE][2])==std::numeric_limits<Real>::infinity() ) printf("case 4 failed!\n");
		if( fabs(baseCurvatureDirections[i][MAX_CURVATURE][0])==std::numeric_limits<Real>::infinity() || fabs(baseCurvatureDirections[i][MAX_CURVATURE][1])==std::numeric_limits<Real>::infinity() || fabs(baseCurvatureDirections[i][MAX_CURVATURE][2])==std::numeric_limits<Real>::infinity() ) printf("case 4 failed!\n");
	}
#endif // sanity test
}
template< class Real , bool Primal >
bool MeshOctree< Real , Primal >::_clipEdge( Node* node , Point3D< Real > in1 , Point3D< Real > in2 , Point3D< Real >& out1 , Point3D< Real >& out2 )
{
	Point3D< Real > axis[3] , center;
	Real width;
	node->centerAndWidth( center , width );
	for( int i=0 ; i<3 ; i++ )
	{
		axis[i] *= 0;
		axis[i][i] = 1;
	}
	out1 = in1;
	out2 = in2;
	Point3D< Real > v1 , v2 , v3;
	int s;
	for( int d=0 ; d<3 ; d++ )
	{
		s = SplitEdge( out1 , out2 , axis[d] , center[d]-width/2 , v1 , v2 , v3 );
		if( s!=-1 ) out1 = v2 , out2 = v3;
		else return false;
		s = SplitEdge( out1 , out2 , axis[d] , center[d]+width/2 , v1 , v2 , v3 );
		if( s!= 1 ) out1 = v1 , out2 = v2;
		else return false;
	}
	return true;
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_addEdge( Node* node , Point3D< Real > v1 , Point3D< Real > v2 )
{
	Point3D< Real > _v1 , _v2;
	node->nodeData.tempIndex = 1;
	if( node->children )
	{
		int count = 0;
		for( int c=0 ; c<Cube::CORNERS ; c++ )
			if( _clipEdge( &node->children[c] , v1 , v2 , _v1 , _v2 ) ) _addEdge( &node->children[c] , _v1 , _v2 ) , count++;
	}
}
template< class Real , bool Primal >
template< class Vertex >
void MeshOctree< Real , Primal >::setEdgeSupportingIndices( const std::vector< Vertex >& vertices , const std::vector< std::pair< int , int > >& edges , std::vector< int >& indices )
{
	// Index:
	// 0 does not contain geoemtry
	// 1 contains geometry
	// 2 supported by geometry

	// Clear the temporary flag
	for( Node* node=_tree.nextNode() ; node ; node=_tree.nextNode( node ) ) node->nodeData.tempIndex = 0;

	// Add the edges into the tree
	for( int i=0 ; i<edges.size() ; i++ )
	{
		Point3D< Real > v1 , v2;
		if( _clipEdge( &_tree , ( Point3D< Real >( vertices[edges[i].first] ) - _translation ) / _scale , ( Point3D< Real >( vertices[edges[i].second] ) - _translation ) / _scale , v1 , v2 ) ) _addEdge( &_tree , v1 , v2 );
	}
	typename OctNode< MeshTreeNodeData< Real > , Real >::NeighborKey3 neighborKey3;	
	neighborKey3.set( _maxDepth );
	for( Node* node=_tree.nextNode() ; node ; node=_tree.nextNode( node ) )
	{
		if( node->depth()!=_maxDepth || node->nodeData.tempIndex!=1 ) continue;
		_tree.getNeighbors( node , neighborKey3 );
		typename Node::Neighbors3& neighbors = neighborKey3.neighbors[_maxDepth];
		if( Primal )
		{
			for( int i=0 ; i<2 ; i++ ) for( int j=0 ; j<2 ; j++ ) for( int k=0 ; k<2 ; k++ )
				if( neighbors.neighbors[i+1][j+1][k+1] && neighbors.neighbors[i+1][j+1][k+1]->nodeData.tempIndex!=1 ) neighbors.neighbors[i+1][j+1][k+1]->nodeData.tempIndex=2;
		}
		else
		{
			for( int i=0 ; i<3 ; i++ ) for( int j=0 ; j<3 ; j++ ) for( int k=0 ; k<3 ; k++ )
				if( neighbors.neighbors[i][j][k] && neighbors.neighbors[i][j][k]->nodeData.tempIndex!=1 ) neighbors.neighbors[i][j][k]->nodeData.tempIndex=2;
		}
	}
	int count = 0;
	for( Node* node=_tree.nextNode() ; node ; node=_tree.nextNode( node ) ) if( node->depth()==_maxDepth && node->nodeData.basisIndex!=-1 && node->nodeData.tempIndex ) count++;
	indices.resize( count );
	count = 0;
	for( Node* node=_tree.nextNode() ; node ; node=_tree.nextNode( node ) ) if( node->depth()==_maxDepth && node->nodeData.basisIndex!=-1 && node->nodeData.tempIndex ) indices[count++] = node->nodeData.basisIndex;
}



template< class Real , bool Primal >
int MeshOctree< Real , Primal >::_setTree( int depth , double cutOff , bool progress )
{
	_fData.set( depth , PPolynomial< ( Primal ? 1 : 2 ) >::GaussianApproximation() , 0 , 1 , false , _fData.D0_FLAG | _fData.D1_FLAG , Primal , 0 , 1 );
	double co = cutOff * 1.0 / ( 1<<( 2 * depth ) );
	std::vector< Node* > nodes;
	typename Node::NeighborKey3 neighborKey3;
	neighborKey3.set( depth );

	// Add the nodes whose support overlaps the geometry
	// This is not implemented in parallel because the memory allocator for the octrees is not thread safe.
	for( Node* node = _tree.nextNode() ; node ; node = _tree.nextNode( node ) )
	{
		node->nodeData.basisIndex = -1;
		node->nodeData.geometryIndex = -1;
		if( node->depth()!=depth ) node->nodeData.tStart = node->nodeData.tEnd = 0;
		if( node->depth()==depth && node->nodeData.tStart<node->nodeData.tEnd ) // If the node has geometry inside of it
		{
			_tree.setNeighbors( node , neighborKey3 , Primal ? 0 : -1 , 1 );
			nodes.push_back( node );
		}
	}

#pragma omp parallel for num_threads( _threads )
	for( int i=0 ; i<_threads ; i++ )
	{
		typename OctNode< MeshTreeNodeData< Real > , Real >::NeighborKey3 neighborKey3;	
		neighborKey3.set( depth );
		for( int j=(nodes.size()*i) / _threads ; j<(nodes.size()*(i+1)) / _threads ; j++ )
		{
			Node* node = nodes[j];
			// Compute the area of the intersection of the geometry with the node, and if it's large enough
			// mark all basis-nodes with overlapping support
			double sz = 0;
			for( int t=node->nodeData.tStart ; t<node->nodeData.tEnd ; t++ )
			{
				for( int j=1 ; j<_polygons[t].count-1 ; j++ )
				{
					Point3D< Real > v1 = _vertices[ _polygons[t][0] ] , v2 = _vertices[ _polygons[t][j] ] , v3 = _vertices[ _polygons[t][j+1] ];
					sz += Real( Length( Point3D< Real >::CrossProduct( v2-v1 , v3-v1 ) ) )/2;
				}
			}
			// If the area is big enough, flag the basis functions overlapping the node, and all of the nodes ancestors
			if( sz>co )
			{
				_tree.getNeighbors( node , neighborKey3 , Primal ? 0  : -1 );
				for( int d=0 ; d<=depth ; d++ )
				{
					typename Node::Neighbors3& neighbors = neighborKey3.neighbors[d];
					for( int i=(Primal?1:0) ; i<3 ; i++ )
						for( int j=(Primal?1:0) ; j<3 ; j++ )
							for( int k=(Primal?1:0) ; k<3 ; k++ )
								if( neighbors.neighbors[i][j][k] ) 
									neighbors.neighbors[i][j][k]->nodeData.basisIndex = 0;
					// Mark the node and all its ancestors as having geometry and set the geometry pointers
					// at coarser nodes appropriately.
					Node* temp = neighbors.neighbors[1][1][1];
					temp->nodeData.geometryIndex = 0;
					if( temp->nodeData.tStart==temp->nodeData.tEnd  ) temp->nodeData.tStart = node->nodeData.tStart , temp->nodeData.tEnd = node->nodeData.tEnd;
					if( node->nodeData.tStart<temp->nodeData.tStart ) temp->nodeData.tStart = node->nodeData.tStart;
					if( node->nodeData.tEnd  >temp->nodeData.tEnd   ) temp->nodeData.tEnd   = node->nodeData.tEnd;
				}
			}
#if 0 // There is no reason why coarser nodes should have direct access to the geometry
			else
				for( Node* temp=node ; temp ; temp=temp->parent )
				{
					temp->nodeData.geometryIndex = 0;
					if( temp->nodeData.tStart==temp->nodeData.tEnd  ) temp->nodeData.tStart = node->nodeData.tStart , temp->nodeData.tEnd = node->nodeData.tEnd;
					if( node->nodeData.tStart<temp->nodeData.tStart ) temp->nodeData.tStart = node->nodeData.tStart;
					if( node->nodeData.tEnd  >temp->nodeData.tEnd   ) temp->nodeData.tEnd   = node->nodeData.tEnd;
				}
#endif
		}
	}
	_basisDimension.resize( depth+1 );
	_geometryDimension.resize( depth+1 );
	for( int d=0 ; d<=depth ; d++ ) _basisDimension[d] = _geometryDimension[d] = 0;
	for( Node* node=_tree.nextNode() ; node ; node=_tree.nextNode( node ) )
	{
		int d = node->depth();
		if( node->nodeData.geometryIndex==0 ) node->nodeData.geometryIndex = _geometryDimension[d]++;
		if( node->nodeData.basisIndex   ==0 ) node->nodeData.basisIndex    =    _basisDimension[d]++;
	}

	_maxDepth = _tree.maxDepth( );
	_sortIndices();
	MemoryUsage();
	return 1;
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_setElementRestriction( ElementRestrictionOperator& restriction , int depth ) const
{
	int count = 0;
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) ) if( node->depth()==depth-1 && node->nodeData.geometryIndex!=-1 ) count++;
	restriction.childInfo.resize ( count );
	restriction.childCount.resize( count );

	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) ) if( node->depth()==depth-1 && node->nodeData.geometryIndex!=-1 )
	{
		int idx = node->nodeData.geometryIndex;
		count = 0;
		if( node->children ) for( int c=0 ; c<Cube::CORNERS ; c++ ) if( node->children[c].nodeData.geometryIndex!=-1 ) count++;
		if( count )
		{
			restriction.childInfo[idx] = new std::pair< int , char >[ count ];
			restriction.childCount[idx] = count;
			count = 0;
			for( int c=0 ; c<Cube::CORNERS ; c++ ) if( node->children[c].nodeData.geometryIndex!=-1 )
				restriction.childInfo[idx][count++] = std::pair< int , char >( node->children[c].nodeData.geometryIndex , c );
		}
		else
		{
			fprintf( stderr , "[Warning] Geometry node without children\n" );
			restriction.childInfo[idx] = NULL;
			restriction.childCount[idx] = 0;
		}
	}
}
	
template< class Real , bool Primal >
template< class Vertex >
int MeshOctree< Real , Primal >::setTree( const std::vector< Vertex >& vertices , const std::vector< TriangleIndex >& triangles , int depth , double cutOff , bool progress , int threads )
{
	_threads = threads;
	_baseNormals.resize( triangles.size() );
	for( int i=0 ; i<triangles.size() ; i++ )
	{
		TriangleIndex tIndex = triangles[ i ];
		Point3D< double > v1 = Point3D< double >( vertices[ tIndex[0] ] ) , v2 = Point3D< double >( vertices[ tIndex[1] ] ) , v3 = Point3D< double >( vertices[ tIndex[2] ] );
		Point3D< double > normal = Point3D< double >::CrossProduct( v2-v1 , v3-v1 );
		double l = Length( normal );
		if( l!=0 ) _baseNormals[ i ] = Point3D< Real >( normal / l );
		else       _baseNormals[i] *= 0;
	}
	_baseTriangles = triangles;
	_vertices.resize( vertices.size() );
	for( int i=0 ; i<vertices.size() ; i++ ) _vertices[i] = Point3D< Real >( vertices[i].point );
	_depthOffset = Primal ? 1 : 2;
#if 0
	if( Primal ) _FitVertices( _vertices , Point3D< Real >( 0.250 , 0.250 , 0.250 ) , 0.50/1.1 , _translation , _scale );
	else         _FitVertices( _vertices , Point3D< Real >( 0.375 , 0.375 , 0.375 ) , 0.25/1.1 , _translation , _scale );
#else
	if( Primal ) _FitVertices( _vertices , Point3D< Real >( 0.250 , 0.250 , 0.250 ) , 0.499 , _translation , _scale );
	else         _FitVertices( _vertices , Point3D< Real >( 0.375 , 0.375 , 0.375 ) , 0.249 , _translation , _scale );
#endif

	double t = Time();
	_addTriangles( triangles , depth+_depthOffset );
	printf( "Chopped up triangle mesh: %d / %d -> %d / %d \t %f (s)\n" , triangles.size() , vertices.size() , _polygons.size() , _vertices.size() , Time()-t );

	return _setTree( depth+_depthOffset , cutOff , progress );
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::setNoiseCoefficients( const double& persistence , std::vector< Point3D< Real > >& coefficients ) const
{
	_setNoiseCoefficients< Point3D< Real > >( persistence , coefficients );
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::setNoiseCoefficients( const int& minDepth , const int& maxDepth , const double& persistence , std::vector< Point3D< Real > >& coefficients ) const
{
	minDepth += _depthOffset;
	maxDepth += _depthOffset
	_setNoiseCoefficients< Point3D< Real > >( minDepth , maxDepth , persistence , coefficients );
}


template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::_setNoiseCoefficients( const double& persistence , std::vector< C >& coefficients ) const
{
	_setNoiseCoefficients( _depthOffset , _maxDepth , persistence , coefficients );
}
template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::_setNoiseCoefficients( int minDepth , int maxDepth , const double& persistence , std::vector< C >& coefficients ) const
{
	std::vector< C > _coefficients;
	double p = persistence;
	if( p<0 ) p = 1;
	_coefficients.resize( _basisDimension[minDepth] );
	for( int d=minDepth ; d<=maxDepth ; d++ )
	{
		double amplitude=pow( p , d );
		coefficients.resize( _basisDimension[d] );
		for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) )
		{
			if( node->depth()!=d || node->nodeData.basisIndex<0 ) continue;
			for( int c=0 ; c<3 ; c++ )	coefficients[node->nodeData.basisIndex][c] = _coefficients[node->nodeData.basisIndex][c] + Real( amplitude*(Random2<double>()*2.0-1.0) );
		}
		if( d<maxDepth ) upSample( coefficients , d , _coefficients );
	}
	for( int d=maxDepth ; d<_maxDepth ; d++ )
	{
		_coefficients.resize( coefficients.size() );
		for( int i=0 ; i<coefficients.size() ; i++ ) _coefficients[i] = coefficients[i];
		upSample( _coefficients , d , coefficients );
	}
	Real max = 0;
	for( int i=0 ; i<coefficients.size() ; i++ ) for( int c=0 ; c<3 ; c++ ) if( fabs( coefficients[i][c] ) > max ) max = fabs( coefficients[i][c] );
//	for( int i=0 ; i<coefficients.size() ; i++ ) for( int c=0 ; c<3 ; c++ ) coefficients[i][c] = coefficients[i][c] / (2*max) + 0.5;
	for( int i=0 ; i<coefficients.size() ; i++ ) for( int c=0 ; c<3 ; c++ ) coefficients[i][c] = coefficients[i][c] / max;
}
template< class Real , bool Primal > template< class Point > void MeshOctree< Real , Primal >::setIdentityCoefficients( Vector< Point >& coefficients ) const
{
	coefficients.Resize( _basisDimension[_maxDepth] );
	_setIdentityCoefficients( coefficients+0 );
	for( int i=0 ; i<coefficients.Dimensions() ; i++ ) coefficients[i] = Point3D< Real >( coefficients[i] ) * _scale + _translation;
}
template< class Real , bool Primal >
template< class C >
void MeshOctree< Real , Primal >::_setIdentityCoefficients( Pointer( C ) coefficients ) const
{
	Real shift = Primal ? Real( 0. ) : Real( 0.5 );
	int depth = _maxDepth;

	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) )
		if( node->depth()==_maxDepth && node->nodeData.basisIndex!=-1)
		{
			int d , off[3] , idx = node->nodeData.basisIndex;
			node->depthAndOffset( d , off );
			Point3D< Real > coeff;
			for( int d=0 ; d<3 ; d++ ) coeff[d] = shift + off[d];
			coefficients[idx] = C( coeff / Real( 1<<depth ) );
		}
}

template< class Real , bool Primal >
void MeshOctree< Real , Primal >::setCheckerBoardCoefficients( int depth , std::vector< Point3D< Real > >& coefficients ) const
{
	depth += _depthOffset;

	coefficients.resize( _basisDimension[_maxDepth] );
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) )
	{
		if( node->depth()==_maxDepth && node->nodeData.basisIndex!=-1)
		{
			int d , off[3];
			node->depthAndOffset(d,off);
			for( int c=0 ; c<3 ; c++ )
			{
				off[c] >>= (_maxDepth-depth);
				if( off[c]&1 )	coefficients[node->nodeData.basisIndex][c] = 1;
				else			coefficients[node->nodeData.basisIndex][c] = 0;
			}
		}
	}
}

template< class Real , bool Primal >
void MeshOctree< Real ,  Primal >::_setElementToBasisMatrix( SparseMatrix< Real , int >& e2b , int depth ) const
{
	std::vector< const Node* > basisNodes;
	basisNodes.resize( _basisDimension[depth] );
	for( const Node* node=_tree.nextNode() ; node ; node=_tree.nextNode(node) ) if( node->depth()==depth && node->nodeData.basisIndex!=-1 ) basisNodes[node->nodeData.basisIndex] = node;

	e2b.Resize( basisNodes.size() );
#pragma omp parallel for num_threads( _threads )
	for( int t=0 ; t<_threads ; t++ )
	{
		typename OctNode< MeshTreeNodeData< Real > , Real >::ConstNeighborKey3 neighborKey3;
		neighborKey3.set( depth );
		for( int i=(basisNodes.size()*t) / _threads ; i<(basisNodes.size()*(t+1)) / _threads ; i++ )
		{
			_tree.getNeighbors( basisNodes[i] , neighborKey3 );
			typename Node::ConstNeighbors3& neighbors = neighborKey3.neighbors[depth];
			int count = 0;
			for( int x=0 ; x<(Primal?2:3) ; x++ )
				for( int y=0 ; y<(Primal?2:3) ; y++ )
					for( int z=0 ; z<(Primal?2:3) ; z++ )
					{
						const Node* node = neighbors.neighbors[x][y][z];
						if( node && node->nodeData.geometryIndex!=-1 ) count++;
					}
			e2b.SetGroupSize( i , count );
			count = 0;
			for( int x=0 ; x<(Primal?2:3) ; x++ )
				for( int y=0 ; y<(Primal?2:3) ; y++ )
					for( int z=0 ; z<(Primal?2:3) ; z++ )
					{
						const Node* node = neighbors.neighbors[x][y][z];
						if( node && node->nodeData.geometryIndex!=-1 )
						{
							int off = Primal ? (1-x)*4 + (1-y)*2 + (1-z) : (2-x)*9 + (2-y)*3 + (2-z);
							e2b[i][count++] = MatrixEntry< Real , int >( node->nodeData.geometryIndex*(Primal?8:27)+off , Real(1.) );
						}
					}
		}
	}
}

template< class Real , bool Primal >
Real MeshOctree< Real , Primal >::getArea( void ) const
{
	double area = 0;
	for( int i=0 ; i<_polygons.size() ; i++ )
		for( int j=1 ; j<_polygons[i].count-1 ; j++ )
			area += Length( Point3D< Real >::CrossProduct( _vertices[ _polygons[i][j+1] ] - _vertices[ _polygons[i][0] ] , _vertices[ _polygons[i][j] ] - _vertices[ _polygons[i][0] ] ) ) / 2;
	return Real( area ) * _scale * _scale;
}
template< class Real , bool Primal >
const OctNode< MeshTreeNodeData< Real > , Real >* MeshOctree< Real , Primal >::_getLeafNode( const Point3D< Real >& position ) const
{
	Point3D< Real > myCenter;
	Real myWidth;

	myCenter.coords[0] = myCenter.coords[1] = myCenter.coords[2] = Real(0.5);
	myWidth = Real(1.0);

	if( position.coords[0]<0 || position.coords[0]>1 ) return NULL;
	if( position.coords[1]<0 || position.coords[1]>1 ) return NULL;
	if( position.coords[2]<0 || position.coords[2]>1 ) return NULL;
	const Node* node=&_tree;

	while( 1 )
	{
		if( !node->children ) return node;

		int cIndex=Node::CornerIndex( myCenter , position );
		node = node->children + cIndex;
		myWidth/=2;
		if( cIndex&1 )	myCenter.coords[0] += myWidth/2;
		else			myCenter.coords[0] -= myWidth/2;
		if( cIndex&2 )	myCenter.coords[1] += myWidth/2;
		else			myCenter.coords[1] -= myWidth/2;
		if( cIndex&4 )	myCenter.coords[2] += myWidth/2;
		else			myCenter.coords[2] -= myWidth/2;
	}
	return NULL;
}
template< class Real , bool Primal >
const OctNode< MeshTreeNodeData< Real > , Real >* MeshOctree< Real , Primal >::_getLeafNode( const Point3D< Real >& position , int maxDepth ) const
{
	Point3D< Real > myCenter;
	Real myWidth;

	myCenter.coords[0] = myCenter.coords[1] = myCenter.coords[2] = Real(0.5);
	myWidth = Real(1.0);

	if( position.coords[0]<0 || position.coords[0]>1 ) return NULL;
	if( position.coords[1]<0 || position.coords[1]>1 ) return NULL;
	if( position.coords[2]<0 || position.coords[2]>1 ) return NULL;
	const Node* node=&_tree;
	int d=0;
	while( d<maxDepth )
	{
		if( !node->children ) return node;
		int cIndex=Node::CornerIndex( myCenter , position );
		node = node->children+cIndex;
		myWidth/=2;
		if(cIndex&1)	myCenter.coords[0] += myWidth/2;
		else			myCenter.coords[0] -= myWidth/2;
		if(cIndex&2)	myCenter.coords[1] += myWidth/2;
		else			myCenter.coords[1] -= myWidth/2;
		if(cIndex&4)	myCenter.coords[2] += myWidth/2;
		else			myCenter.coords[2] -= myWidth/2;
		d++;
	}
	return node;
}
template< class Real , bool Primal >
OctNode< MeshTreeNodeData< Real > , Real >* MeshOctree< Real , Primal >::_setLeafNode( const Point3D< Real >& position , int maxDepth )
{
	Point3D< Real > myCenter;
	Real myWidth;

	myCenter.coords[0]=myCenter.coords[1]=myCenter.coords[2]=Real(0.5);
	myWidth=Real(1.0);

	if(position.coords[0]<0 || position.coords[0]>1)	return NULL;
	if(position.coords[1]<0 || position.coords[1]>1)	return NULL;
	if(position.coords[2]<0 || position.coords[2]>1)	return NULL;
	Node* node=&_tree;
	int d=0;
	while(d<maxDepth)
	{
		if(!node->children) node->initChildren();
		int cIndex=Node::CornerIndex( myCenter , position );
		node = node->children+cIndex;
		myWidth/=2;
		if(cIndex&1)	myCenter.coords[0] += myWidth/2;
		else			myCenter.coords[0] -= myWidth/2;
		if(cIndex&2)	myCenter.coords[1] += myWidth/2;
		else			myCenter.coords[1] -= myWidth/2;
		if(cIndex&4)	myCenter.coords[2] += myWidth/2;
		else			myCenter.coords[2] -= myWidth/2;
		d++;
	}
	return node;
}

template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_getPointOffset( const Point3D< Real >& position , int depth , int offset[3] ) const
{
	offset[0] = offset[1] = offset[2] = 0;
	Point3D< Real > myCenter;
	Real myWidth;

	myCenter.coords[0] = myCenter.coords[1] = myCenter.coords[2] = Real(0.5);
	myWidth = Real(1.0);

	if( position.coords[0]<0 || position.coords[0]>1 ) return;
	if( position.coords[1]<0 || position.coords[1]>1 ) return;
	if( position.coords[2]<0 || position.coords[2]>1 ) return;
	int d=0;
	while( d<depth )
	{
		offset[0] <<= 1;
		offset[1] <<= 1;
		offset[2] <<= 1;
		int cIndex=Node::CornerIndex( myCenter , position );
		myWidth/=2;
		if(cIndex&1)	myCenter.coords[0] += myWidth/2 , offset[0] |= 1;
		else			myCenter.coords[0] -= myWidth/2;
		if(cIndex&2)	myCenter.coords[1] += myWidth/2 , offset[1] |= 1;
		else			myCenter.coords[1] -= myWidth/2;
		if(cIndex&4)	myCenter.coords[2] += myWidth/2 , offset[2] |= 1;
		else			myCenter.coords[2] -= myWidth/2;
		d++;
	}
}
template< class Real , bool Primal >
typename MeshOctree< Real , Primal >::Node::ConstNeighbors3 MeshOctree< Real , Primal >::_getFinestNodeNeighbors( typename MeshOctree< Real , Primal >::Node::ConstNeighborKey3& constNeighborKey3 , const Point3D< Real >& point , int depth , int offset[3] ) const
{
	const Node* node = _getLeafNode( point );

	if( node->depth()==depth )
	{
		int depth;
		node->depthAndOffset( depth , offset );
		_tree.getNeighbors( node , constNeighborKey3 );
		return constNeighborKey3.neighbors[depth];
	}
	else
	{
		Real width;
		width = Real( 1.0 ) / ( 1<<depth );

		typename Node::ConstNeighbors3 neighbors;
		Point3D< Real > p;
		for( int i=-1 ; i<=1 ; i++ )
		{
			p[0] = point[0] + (width * i);
			for( int j=-1 ; j<=1 ; j++ )
			{
				p[1] = point[1] + (width * j);
				for( int k=-1 ; k<=1 ; k++ )
				{
					p[2] = point[2] + (width * k);
					const Node* node = _getLeafNode( p , depth );
					if( node->depth()==depth ) neighbors.neighbors[i+1][j+1][k+1] = node;
					else                       neighbors.neighbors[i+1][j+1][k+1] = NULL;
				}
			}
		}
		_getPointOffset( point , depth , offset );
		return neighbors;
	}
}
template< class Real , bool Primal >
void MeshOctree< Real , Primal >::_FitVertices( std::vector< Point3D< Real > >& vertices , const Point3D< Real >& center , const Real& width , Point3D< Real >& translation , Real& scale )
{
	Point3D< Real > min , max;
	for( size_t i=0 ; i<vertices.size() ; i++ )
		for( int c=0 ; c<3 ;c++ )
		{
			if( !i || vertices[i][c]<min[c] )	min[c] = vertices[i][c];
			if( !i || vertices[i][c]>max[c] )	max[c] = vertices[i][c];
		}

	for( int c=0 ; c<3 ; c++ )
		if( !c || scale<max[c]-min[c] ) scale = max[c]-min[c];
	scale /= width;
	for( int c=0 ; c<3 ; c++ ) translation[c] = (max[c]+min[c])/2 - center[c]*scale;

	for( size_t s=0 ; s<vertices.size() ; s++ ) vertices[s] = ( vertices[s] - translation ) / scale;
}
