/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#ifndef FAST_POINT_3D_INCLUDED
#define FAST_POINT_3D_INCLUDED

#include <emmintrin.h>

// This class defines a 3D point whose binary operators are implemented using the SSE instruction set.
// As a result, there is a storage overhead due to the fact that four values are stored instead of three.
template< class Real >
class FastPoint3D
{
	__declspec( align( 16 ) ) Real _v[4];
	FastPoint3D( __m128 _values ) { *( (__m128*)_v ) = _values; }
	FastPoint3D( __m128d _values1 , __m128d _values2 ) { *( (__m128d*)(_v+0) ) = _values1 , *( (__m128d*)(_v+2) ) = _values2; }
public:
	typedef Real R;
	FastPoint3D( void );
	FastPoint3D( Real v1 , Real v2 , Real v3 );
	template< class Real2 > FastPoint3D( const       Point< Real2 , 3 >& p );
	template< class Real2 > FastPoint3D( const     Point3D< Real2     >& p );
	template< class Real2 > FastPoint3D( const FastPoint3D< Real2     >& p );
	operator Point3D< Real >( ) const;
	const Real& operator[] ( int idx ) const;

	inline FastPoint3D operator  +  ( const FastPoint3D& p ) const;
	inline FastPoint3D operator  -  ( const FastPoint3D& p ) const;
	inline FastPoint3D operator  *  ( const Real& r ) const;
	inline FastPoint3D operator  /  ( const Real& r ) const;
	inline FastPoint3D& operator += ( const FastPoint3D& p );
	inline FastPoint3D& operator -= ( const FastPoint3D& p );
	inline FastPoint3D& operator *= ( const Real& r );
	inline FastPoint3D& operator /= ( const Real& r );
	static Real Dot( const FastPoint3D& p1 , const FastPoint3D& p2 );
	static Real SquareNorm( const FastPoint3D& p );
};
template< class Real >                         FastPoint3D< Real >::FastPoint3D( Real v1 , Real v2 , Real v3 )  { _v[0] = v1         , _v[1] = v2         , _v[2] = v3         , _v[3] = Real( 0. ); }
template< class Real > template< class Real2 > FastPoint3D< Real >::FastPoint3D( const   Point< Real2 , 3 >& p ){ _v[0] = Real(p[0]) , _v[1] = Real(p[1]) , _v[2] = Real(p[2]) , _v[3] = Real( 0. ); }
template< class Real > template< class Real2 > FastPoint3D< Real >::FastPoint3D( const     Point3D< Real2 >& p ){ _v[0] = Real(p[0]) , _v[1] = Real(p[1]) , _v[2] = Real(p[2]) , _v[3] = Real( 0. ); }
template< class Real > template< class Real2 > FastPoint3D< Real >::FastPoint3D( const FastPoint3D< Real2 >& p ){ _v[0] = Real(p[0]) , _v[1] = Real(p[1]) , _v[2] = Real(p[2]) , _v[3] = Real( 0. ); }
template< class Real > FastPoint3D< Real >::operator Point3D< Real >( ) const { return Point3D< Real >( _v[0] , _v[1] , _v[2] ); }
template< class Real > const Real& FastPoint3D< Real >::operator[] ( int idx ) const { return _v[idx]; }

template<> FastPoint3D< float  >::FastPoint3D( void ){ *( (__m128*)_v )                              = _mm_setzero_ps( ); }
template<> FastPoint3D< double >::FastPoint3D( void ){ *( (__m128d*)(_v+0) ) = *( (__m128d*)(_v+2) ) = _mm_setzero_pd( ); }

template<> inline FastPoint3D< float  > FastPoint3D< float  >::operator  +  ( const FastPoint3D< float  >& p ) const { return FastPoint3D( _mm_add_ps( *( (__m128 *)(_v  ) ) , *( (__m128 *)(p._v  ) ) ) ); }
template<> inline FastPoint3D< float  > FastPoint3D< float  >::operator  -  ( const FastPoint3D< float  >& p ) const { return FastPoint3D( _mm_sub_ps( *( (__m128 *)(_v  ) ) , *( (__m128 *)(p._v  ) ) ) ); }
template<> inline FastPoint3D< float  > FastPoint3D< float  >::operator  *  ( const float&  r ) const { __m128  _r = _mm_set1_ps(       r ) ; return FastPoint3D( _mm_mul_ps( _r , *( (__m128 *)(_v  ) ) ) ); }
template<> inline FastPoint3D< float  > FastPoint3D< float  >::operator  /  ( const float&  r ) const { __m128  _r = _mm_set1_ps( 1.f / r ) ; return FastPoint3D( _mm_mul_ps( _r , *( (__m128 *)(_v  ) ) ) ); }
template<> inline FastPoint3D< float  >& FastPoint3D< float  >::operator += ( const FastPoint3D< float  >& p ) { *( (__m128 *)(_v  ) ) = _mm_add_ps( *( (__m128 *)(_v  ) ) , *( (__m128 *)(p._v  ) ) ) ; return (*this); }
template<> inline FastPoint3D< float  >& FastPoint3D< float  >::operator -= ( const FastPoint3D< float  >& p ) { *( (__m128 *)(_v  ) ) = _mm_sub_ps( *( (__m128 *)(_v  ) ) , *( (__m128 *)(p._v  ) ) ) ; return (*this); }
template<> inline FastPoint3D< float  >& FastPoint3D< float  >::operator *= ( const float&  r ) { __m128  _r = _mm_set1_ps(     r ) ; *( (__m128 *)(_v  ) ) = _mm_mul_ps( _r , *( (__m128 *)(_v  ) ) ) ; return (*this); }
template<> inline FastPoint3D< float  >& FastPoint3D< float  >::operator /= ( const float&  r ) { __m128  _r = _mm_set1_ps( 1.f/r ) ; *( (__m128 *)(_v  ) ) = _mm_mul_ps( _r , *( (__m128 *)(_v  ) ) ) ; return (*this); }

template<> float  FastPoint3D< float  >::Dot( const FastPoint3D< float  >& p1 , const FastPoint3D< float  >& p2 ){ __m128  _temp = _mm_mul_ps( *( (__m128 *)(p1._v  ) ) , *( (__m128 *)(p2._v  ) ) ) ; __declspec ( align( 16 ) ) float  temp[4] ; _mm_store_ps( temp , _temp ) ; return temp[0] + temp[1] + temp[2] + temp[3]; }
template<> float  FastPoint3D< float  >::SquareNorm( const FastPoint3D< float  >& pp )                           { __m128  _temp = _mm_mul_ps( *( (__m128 *)(pp._v  ) ) , *( (__m128 *)(pp._v  ) ) ) ; __declspec ( align( 16 ) ) float  temp[4] ; _mm_store_ps( temp , _temp ) ; return temp[0] + temp[1] + temp[2] + temp[3]; }

template<> inline FastPoint3D< double > FastPoint3D< double >::operator  +  ( const FastPoint3D< double >& p ) const { return FastPoint3D( _mm_add_pd( *( (__m128d*)(_v+0) ) , *( (__m128d*)(p._v+0) ) ) , _mm_add_pd( *( (__m128d*)(_v+2) ) , *( (__m128d*)(p._v+2) ) ) ); }
template<> inline FastPoint3D< double > FastPoint3D< double >::operator  -  ( const FastPoint3D< double >& p ) const { return FastPoint3D( _mm_sub_pd( *( (__m128d*)(_v+0) ) , *( (__m128d*)(p._v+0) ) ) , _mm_sub_pd( *( (__m128d*)(_v+2) ) , *( (__m128d*)(p._v+2) ) ) ); }
template<> inline FastPoint3D< double > FastPoint3D< double >::operator  *  ( const double& r ) const { __m128d _r = _mm_set1_pd(       r ) ; return FastPoint3D( _mm_mul_pd( _r , *( (__m128d*)(_v+0) ) ) , _mm_mul_pd( _r , *( (__m128d*)(_v+2) ) ) ); }
template<> inline FastPoint3D< double > FastPoint3D< double >::operator  /  ( const double& r ) const { __m128d _r = _mm_set1_pd( 1.0 / r ) ; return FastPoint3D( _mm_mul_pd( _r , *( (__m128d*)(_v+0) ) ) , _mm_mul_pd( _r , *( (__m128d*)(_v+2) ) ) ); }
template<> inline FastPoint3D< double >& FastPoint3D< double >::operator += ( const FastPoint3D< double >& p ) { *( (__m128d*)(_v+0) ) = _mm_add_pd( *( (__m128d*)(_v+0) ) , *( (__m128d*)(p._v+0) ) ) , *( (__m128d*)(_v+2) ) = _mm_add_pd( *( (__m128d*)(_v+2) ) , *( (__m128d*)(p._v+2) ) ) ; return (*this); }
template<> inline FastPoint3D< double >& FastPoint3D< double >::operator -= ( const FastPoint3D< double >& p ) { *( (__m128d*)(_v+0) ) = _mm_sub_pd( *( (__m128d*)(_v+0) ) , *( (__m128d*)(p._v+0) ) ) , *( (__m128d*)(_v+2) ) = _mm_sub_pd( *( (__m128d*)(_v+2) ) , *( (__m128d*)(p._v+2) ) ) ; return (*this); }
template<> inline FastPoint3D< double >& FastPoint3D< double >::operator *= ( const double& r ) { __m128d _r = _mm_set1_pd(     r ) ; *( (__m128d*)(_v+0) ) = _mm_mul_pd( _r , *( (__m128d*)(_v+0) ) ) , *( (__m128d*)(_v+2) ) = _mm_mul_pd( _r , *( (__m128d*)(_v+2) ) ) ; return (*this); }
template<> inline FastPoint3D< double >& FastPoint3D< double >::operator /= ( const double& r ) { __m128d _r = _mm_set1_pd( 1.0/r ) ; *( (__m128d*)(_v+0) ) = _mm_mul_pd( _r , *( (__m128d*)(_v+0) ) ) , *( (__m128d*)(_v+2) ) = _mm_mul_pd( _r , *( (__m128d*)(_v+2) ) ) ; return (*this); }

template<> double FastPoint3D< double >::Dot( const FastPoint3D< double >& p1 , const FastPoint3D< double >& p2 ){ __m128d _temp = _mm_add_pd( _mm_mul_pd( *( (__m128d*)(p1._v+0) ) , *( (__m128d*)(p2._v+0) ) ) , _mm_mul_pd( *( (__m128d*)(p1._v+2) ) , *( (__m128d*)(p2._v+2) ) ) ) ; __declspec ( align( 16 ) ) double temp[2] ; _mm_store_pd( temp , _temp ) ; return temp[0] + temp[1]; }
template<> double FastPoint3D< double >::SquareNorm( const FastPoint3D< double >& pp )                           { __m128d _temp = _mm_add_pd( _mm_mul_pd( *( (__m128d*)(pp._v+0) ) , *( (__m128d*)(pp._v+0) ) ) , _mm_mul_pd( *( (__m128d*)(pp._v+2) ) , *( (__m128d*)(pp._v+2) ) ) ) ; __declspec ( align( 16 ) ) double temp[2] ; _mm_store_pd( temp , _temp ) ; return temp[0] + temp[1]; }

#endif // FAST_POINT_3D_INCLUDED
