/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#include "lineqn.h"
#include <limits>
float maxC , minC;

#include <omp.h>
#include <math.h>
#include <float.h>
#include "Time.h"
#include "ply.h"
#include "Mesh.h"
#include "Octree.h"
#include "SparseMatrix.h"
#include "CmdLineParser.h"
#include "FunctionData.h"
#include "PPolynomial.h"
#include "MemoryUsage.h"
#include "PoissonMeshData.h"
#include <stdarg.h>
#include "Util.h"

/////////////////////////////////////////////////////////////
// OpengGL and Linear System (for real-time visualization) //
/////////////////////////////////////////////////////////////
#ifdef _WIN32
#  define WINDOWS_LEAN_AND_MEAN
#  define NOMINMAX
#  include <windows.h>
#endif
#include <GL/glew.h>
#include <GL/glut.h>
#include "glStage.h"

#include <cuda_runtime.h>
#include <cutil_inline.h>
#include <cutil_gl_inline.h>
#include <cutil_gl_error.h>
#include <cuda_gl_interop.h>
#include <vector_types.h>
#include <cuda.h>

extern "C" 
{
	void cuKernelDual2  ( float4* pos , unsigned int* d_eIndices , unsigned int* e_nIndices , float* d_eValues , float* d_pValues , float* d_cValues , GLuint num , GLuint eNum , GLuint nNum );
	void cuKernelPrimal2( float4* pos , unsigned int* d_eIndices , unsigned int* e_nIndices , float* d_eValues , float* d_pValues , float* d_cValues , GLuint num , GLuint eNum , GLuint nNum );
}

/////////////////////////////////
// Pulled out Global Variables //
/////////////////////////////////
glStage myStage;
template< class Real , bool Primal >
struct Tree
{
	static const int dependence;
	static const int supportCount;
	static void (*cuKernel)( float4* , unsigned int* , unsigned int* , float* , float* , float* , GLuint , GLuint , GLuint );
	static std::vector< Point3D< Real > > fitVertices;
	static typename MeshOctree< Real , Primal >::ParallelSolver< FastPoint3D< Real > > solver;
	static typename MeshOctree< Real , Primal >::ElementMatrix dot , lapMin , lapMax;
	static typename MeshOctree< Real , Primal >::ElementVector< FastPoint3D< Real > > B , initialB , dotB , lapB , initialLapMinB , initialLapMaxB;
	static SparseMatrix< Real , int > E;
	static std::vector< Real > weightsMin , weightsMax;
	static Vector< Real > maxCurvature, minCurvature;
	static std::vector< Real > gradientScale;
	static std::vector< int > nIndices;
	static std::vector< Point3D< Real > > elementPosition;
};
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ParallelSolver< FastPoint3D< Real > > Tree< Real , Primal >::solver;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ElementMatrix Tree< Real , Primal >::dot;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ElementMatrix Tree< Real , Primal >::lapMin;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ElementMatrix Tree< Real , Primal >::lapMax;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ElementVector< FastPoint3D< Real > > Tree< Real , Primal >::B;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ElementVector< FastPoint3D< Real > > Tree< Real , Primal >::initialB;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ElementVector< FastPoint3D< Real > > Tree< Real , Primal >::initialLapMinB;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ElementVector< FastPoint3D< Real > > Tree< Real , Primal >::initialLapMaxB;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ElementVector< FastPoint3D< Real > > Tree< Real , Primal >::dotB;
template< class Real , bool Primal > typename MeshOctree< Real , Primal >::ElementVector< FastPoint3D< Real > > Tree< Real , Primal >::lapB;
template< class Real , bool Primal > SparseMatrix< Real , int > Tree< Real , Primal >::E;
template< class Real , bool Primal > std::vector< Point3D< Real > > Tree< Real , Primal >::fitVertices;
template< class Real , bool Primal > std::vector< Real > Tree< Real , Primal >::weightsMin;
template< class Real , bool Primal > std::vector< Real > Tree< Real , Primal >::weightsMax;
template< class Real , bool Primal > Vector< Real > Tree< Real , Primal >::minCurvature;
template< class Real , bool Primal > Vector< Real > Tree< Real , Primal >::maxCurvature;
template< class Real , bool Primal > std::vector< Real > Tree< Real , Primal >::gradientScale;
template< class Real , bool Primal > std::vector< int > Tree< Real , Primal >::nIndices;
template< class Real , bool Primal > std::vector< Point3D< Real > > Tree< Real , Primal >::elementPosition;
template<> const int Tree< float , true  >::supportCount = 8;
template<> const int Tree< float , false >::supportCount = 27;
template<> const int Tree< float , true  >::dependence = 1;
template<> const int Tree< float , false >::dependence = 2;
template<> void ( *Tree< float , true  >::cuKernel )( float4* , unsigned int* , unsigned int* , float* , float* , float* , GLuint , GLuint , GLuint ) = cuKernelPrimal2;
template<> void ( *Tree< float , false >::cuKernel )( float4* , unsigned int* , unsigned int* , float* , float* , float* , GLuint , GLuint , GLuint ) = cuKernelDual2;
template<> const int Tree< double , true  >::supportCount = 8;
template<> const int Tree< double , false >::supportCount = 27;
template<> const int Tree< double , true  >::dependence = 1;
template<> const int Tree< double , false >::dependence = 2;
template<> void ( *Tree< double , true  >::cuKernel )( float4* , unsigned int* , unsigned int* , float* , float* , float* , GLuint , GLuint , GLuint ) = cuKernelPrimal2;
template<> void ( *Tree< double , false >::cuKernel )( float4* , unsigned int* , unsigned int* , float* , float* , float* , GLuint , GLuint , GLuint ) = cuKernelDual2;


int commentNum=0;
char **comments;
int file_type;

//////////////
// Comments //
//////////////
cmdLineString In( "in" ) , Logs( "logs" ) , XForm( "xForm" );
cmdLineReadable Verbose( "verbose" ) , FullVerbose( "fullVerbose" ) , Progress( "progress" ) , TestFPS( "testFPS" ) , Offline( "offline" ) , CUDAOffline( "cudaOffline" ) , NoColor( "noColor" );
cmdLineInt Depth( "depth" , 8 ) , Iters( "iters" , 2 ) , Subdivide( "subdivide" , 0 ) , VCycles( "vCycles" , 1 ) , ParallelDepth( "parallelDepth" , 5 ) , GSBlockSize( "gsBlockSize" , 2 ) , Threads( "threads" , omp_get_num_procs() ) , PlotWidth( "plotWidth" , 4 ) , Width( "width" , 640 ) , Height( "height" , 480 );
cmdLineFloat CleanTree( "cutOff" , 0.f ) , Alpha( "alpha" , 0.0001f ) , GradientScale( "gScale" , 1.0f ) , MaxGradientScale( "gScaleMax" , 2.0), GaussianModulator( "gModulator", 2.0f ), Normalize( "normalize" , 2 ) , PlotRange( "plotRange" , 10. );
cmdLineReadable Dual( "dual" ) , Minimal( "minimal" ) , ShowFPS( "showFPS" ) , CPUEvaluation( "cpuEvaluation" ) , ShowError( "showError" );
cmdLineReadable Cascadic( "cascadic" ) , WCycle( "wCycle" ) , ResetSolution( "resetSolution" );

cmdLineReadable* params[]=
{
	&In , &Depth , &Verbose , &Iters , &CleanTree , &FullVerbose , &Subdivide , &VCycles , &TestFPS, &Offline , &CUDAOffline , &NoColor ,
	&Progress , &ParallelDepth, &Threads , &GSBlockSize , &PlotWidth , &Width , &Height ,
	&Logs , &Alpha , &MaxGradientScale , &GaussianModulator , &Normalize , &PlotRange ,
	&Dual , &Minimal , &ShowFPS , &CPUEvaluation , &Cascadic , &WCycle , &ResetSolution ,
	&ShowError , &XForm ,
};

void ShowUsage( char* ex )
{
	printf( "Usage: %s\n" , ex );
	printf( "\t--%s <input triangle mesh>\n" , In.name);
	printf( "\t\t The triangle mesh to be edited.\n" );

	printf( "\t[--%s <maximum reconstruction depth>=%d]\n" , Depth.name , Depth.value );
	printf( "\t\t Running at depth d corresponds to solving on a 2^d x 2^d x 2^d\n" );
	printf( "\t\t voxel grid.\n");

	printf( "\t[--%s <solver iterations>=%d]\n" , Iters.name , Iters.value );
	printf( "\t\t The number of solver iterations within each resolution\n" );

	printf( "\t[--%s <number of threads>=%d]\n" , Threads.name , Threads.value );
	printf( "\t\t The number of threads to be used for parallelizing the solver.\n" );

	printf( "\t[--%s <max scale>=%f]\n" , MaxGradientScale.name , MaxGradientScale.value );
	printf( "\t\t The maximum gradient amplification scale.\n" );

	printf( "\t[--%s]\n" , Verbose.name );
	printf( "\t[--%s]\n" , Progress.name );
	printf( "\t[--%s]\n" , Dual.name );

}
template< class Real , bool Primal >
int Execute( int argc , char* argv[] )
{
	typedef FastPoint3D< Real > MyPoint3D;
	int paramNum=sizeof(params)/sizeof(cmdLineReadable*);
	comments=new char*[paramNum+10];
	for( int i=0 ; i<paramNum+10 ; i++ ) comments[i] = new char[1024];
	char valueString[1024];
	for(int i=0;i<paramNum;i++)
		if( params[i]->set )
		{
			params[i]->writeValue( valueString );
			DumpOutput( Logs.value , Verbose.set , comments[commentNum++] , "\t--%s %s\n" , params[i]->name , valueString );
		}
	if( Threads.value<1 ) Threads.value = 1;
	if( ParallelDepth.value>Depth.value )
	{
		ParallelDepth.value = Depth.value;
		Threads.value = 1;
	}
	double t;
	Point3D< Real > center;
	Real scale;
	std::vector< TriangleIndex > inTriangles;
	std::vector< PlyVertex< Real > > inVertices;
	std::vector< int > vIndices , tIndices;

	// Read in the mesh
	{
		PlyReadTriangles( In.value , inVertices , inTriangles , PlyVertex< Real >::ReadProperties , PlyVertex< Real >::ReadComponents , file_type );
		FitVertices< Real , PlyVertex< Real > >( inVertices , Point3D< Real >( 0.5 , 0.5 , 0.5 ) , 1. , center , scale );
		myStage.triangles.resize( inTriangles.size() );
		Tree< Real , Primal >::fitVertices.resize ( inVertices.size() );
		for( int i=0 ; i<inTriangles.size() ; i++ ) myStage.triangles[i] = inTriangles[i];
		for( int i=0 ; i<inVertices.size()  ; i++ ) Tree< Real , Primal >::fitVertices[i] = inVertices[i].point;

		// Subdivide the output for denser sampling
		tIndices.resize( myStage.triangles.size() );
		for( int i=0 ; i<myStage.triangles.size() ; i++ ) tIndices[i] = i;
		for( int s=0 ; s<Subdivide.value ; s++ )
		{
			std::vector< int > tempIndices;
			SubdivideTriangles( Tree< Real , Primal >::fitVertices , myStage.triangles , tempIndices );
			for( int i=0 ; i<tempIndices.size() ; i++ ) tempIndices[i] = tIndices[ tempIndices[i] ];
			tIndices.resize( tempIndices.size() );
			for( int i=0 ; i<tempIndices.size() ; i++ ) tIndices[i] = tempIndices[i];
		}
		// Set the index of each vertex to be equal to one of the original triangles containing it.
		// (Since we assume a continuous function, we don't care which triangle.)
		vIndices.resize( Tree< Real , Primal >::fitVertices.size() );
		for( int i=0 ; i<myStage.triangles.size() ; i++ ) for( int c=0 ; c<3 ; c++ ) vIndices[ myStage.triangles[i][c] ] = tIndices[ i ];
	}// end of Reading Mesh

	{
		MeshOctree< Real , Primal > tree;
		t=Time();
		// Set the tree
		tree.setTree( inVertices , inTriangles , Depth.value , CleanTree.value , Progress.set , Threads.value );
		// Initialize the linear solver
		tree.setSolver( Tree< Real , Primal >::solver , ParallelDepth.value , Threads.value ,  GSBlockSize.value , true );
		// Get the element matrices and curvature values
		tree.getDotProductMatrix( Tree< Real , Primal >::dot , Progress.set );
		tree.getLaplacianMatrix( Tree< Real , Primal >::lapMin , Tree< Real , Primal >::lapMax , Tree< Real , Primal >::minCurvature , Tree< Real , Primal >::maxCurvature , Progress.set );

		// Initialize the constraints and the original solution
		{
			tree.setIdentityCoefficients( Tree< Real , Primal >::solver.x() );
			Tree< Real , Primal >::solver.pX().SetFromArray( Tree< Real , Primal >::solver.x() );

			// Compute the element based solution
			typename MeshOctree< Real , Primal >::ElementVector< FastPoint3D< Real > > initialSolution;
			Tree< Real , Primal >::solver.e2b.back().MultiplyTranspose( Tree< Real , Primal >::solver.x() , initialSolution , Threads.value , true );

			// Set the initial constraints
			Tree< Real , Primal >::initialB.resize( Tree< Real , Primal >::dot.size() );
			Tree< Real , Primal >::initialLapMinB.resize( Tree< Real , Primal >::dot.size() );
			Tree< Real , Primal >::initialLapMaxB.resize( Tree< Real , Primal >::dot.size() );
			Tree< Real , Primal >::dot.MultiplyParallel< FastPoint3D< Real > >   ( initialSolution , Tree< Real , Primal >::initialB       , Threads.value , MULTIPLY_CLEAR );
			Tree< Real , Primal >::lapMin.MultiplyParallel< FastPoint3D< Real > >( initialSolution , Tree< Real , Primal >::initialLapMinB , Threads.value , MULTIPLY_CLEAR );
			Tree< Real , Primal >::lapMax.MultiplyParallel< FastPoint3D< Real > >( initialSolution , Tree< Real , Primal >::initialLapMaxB , Threads.value , MULTIPLY_CLEAR );

			// Allocate memory to store the constraint vectors that will be generated later
			Tree< Real , Primal >::dotB.resize( Tree< Real , Primal >::initialB.size() );
			Tree< Real , Primal >::lapB.resize( Tree< Real , Primal >::initialB.size() );
			Tree< Real , Primal >::B.resize   ( Tree< Real , Primal >::initialB.size() );

			myStage.vertices.resize( Tree< Real , Primal >::fitVertices.size() );
			for( int i=0 ; i<Tree< Real , Primal >::fitVertices.size() ; i++ )  myStage.vertices[i] = Point3D< float >( Tree< Real , Primal >::fitVertices[i] );
		}
		// Set up for visualization
		{
			tree.getGeometryPositions( Tree< Real , Primal >::elementPosition );
			tree.getEvaluationMatrix( Tree< Real , Primal >::fitVertices , Tree< Real , Primal >::E );
			tree.getGeometryIndices( Tree< Real , Primal >::fitVertices , Tree< Real , Primal >::nIndices );
		}

		DumpOutput( Verbose.set , comments[commentNum++],"#             Tree set in: %9.1f (s), %9.1f (MB)\n" , Time()-t , tree.maxMemoryUsage );
		DumpOutput( Logs.value ,  Verbose.set , comments[commentNum++] , "    Dimension/Elements: %d / %d\n" , tree.basisDimension() , tree.elements() );
		if( Verbose.set ) printf( "    Vertices/Triangles: %d / %d\n"  , (int)inVertices.size() , (int)inTriangles.size() );
		if( Verbose.set ) printf( "            Tree Depth: %d\n" , Depth.value );
	}// end of Tree Set-up

	return EXIT_SUCCESS;
}

#include "OpenGL.inl"

int main(int argc,char* argv[])
{
	cmdLineParse( argc-1 , &argv[1] , sizeof(params) / sizeof(cmdLineReadable*) , params , 0 );
	if( FullVerbose.set ) Verbose.set = true;
	if( !In.set ) 
	{
		ShowUsage( argv[0] ); 
		return EXIT_FAILURE;
	}

	if( Dual.set ) Execute  < double , false >( argc , argv ) , Visualize< double , false >( argc , argv );
	else		   Execute  < float  , true  >( argc , argv ) , Visualize< float  , true  >( argc , argv );
}
