/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#include <stdio.h>
#include "mouse.h"

template< class Real >
class Drawable
{
public:
	virtual void drawOpenGL( void ) = 0;
};

//////////////////
// GUI Platform //
//////////////////
template< class Real >
class Platform
{
	std::vector< Drawable<Real>* > components;
public:
	void drawOpenGL(void)
	{
		static GLint vp[4], mm, dt=glIsEnabled(GL_DEPTH_TEST), lm=glIsEnabled(GL_LIGHTING);
		glGetIntegerv(GL_VIEWPORT,vp);
		glGetIntegerv(GL_MATRIX_MODE,&mm);
		glMatrixMode(GL_MODELVIEW);	 glPushMatrix(); glLoadIdentity();
		glMatrixMode(GL_PROJECTION); glPushMatrix(); glLoadIdentity();
		glOrtho(0,vp[2],0,vp[3],0,1);
		glDisable(GL_DEPTH_TEST);
		glDisable(GL_LIGHTING);

		for(int i=0;i<components.size();i++) components[i]->drawOpenGL(); 

		if(dt){glEnable(GL_DEPTH_TEST);}
		if(lm){glEnable(GL_LIGHTING);}
		glPopMatrix();
		glMatrixMode(GL_MODELVIEW);
		glPopMatrix();
		glMatrixMode(mm);
	};
	void add( Drawable<Real>* newEntry ){ components.push_back(newEntry); };
};

////////////////
// Slider Bar //
////////////////
template< class Real >
class Slider : public Drawable<Real>
{
	class Marker
	{
		private: int _x, _y, _len;
				 Point3D< Real > _fillColor , _lineColor;
				 int _thickness;
		public:	 void initialize( int x , int y , int len , int thickness , Point3D< Real > fillColor , Point3D< Real > lineColor ){ _x=x , _y=y , _len=len , _thickness = thickness , _fillColor = fillColor , _lineColor = lineColor ; }
				 void initialize( int x , int y ){ _x=x , _y=y; };
				 void setColor( Point3D< Real > fillColor , Point3D< Real > lineColor ) { _fillColor = fillColor , _lineColor = lineColor; }
				 void update(int x){_x=x;};
				 bool isOnMarker( int x , int y ){ return( x>_x && y>_y && x<_x+_len && y<_y+_len ); };
				 void drawOpenGL( )
				 {
					 glColor3f( _fillColor[0] , _fillColor[1] , _fillColor[2] );
					 glBegin( GL_QUADS ); glVertex2f(_x,_y);glVertex2f(_x+_len,_y);glVertex2f(_x+_len,_y+_len);glVertex2f(_x,_y+_len);glEnd();
					 glLineWidth( _thickness );
					 glColor3f( _lineColor[0] , _lineColor[1] , _lineColor[2] );
					 glBegin( GL_LINE_LOOP ) , glVertex2f(_x,_y) , glVertex2f(_x+_len,_y) , glVertex2f(_x+_len,_y+_len) , glVertex2f(_x,_y+_len) , glEnd( );
				 };
	};

private:
	int _markerSize , _thickness;
	Point3D< Real > fillColor , lineColor;
	bool _writeValue;
	int _posX, _posY, _width, _height;
	Real _value, _maxValue, _minValue, _range;
	bool _isInMotion, _isDragging, _locked;
	Marker _marker;
	bool _isOnSlider(int x, int y){ return( x>_posX && y>_posY && x<_posX+_width && y<_posY+_height); };
	bool _logScale;


public:
	// <?> pass the memory add to update?
	void initialize( int posX, int posY, int width, int height , int markerDiameter , int thickness , Point3D< Real > fillColor , Point3D< Real > lineColor , Point3D< Real > markerColor , Real minValue=0.0, Real maxValue=1.0, Real value=0.5, bool lock=true , bool logScale=false , bool writeValue=true );
	void setValue(const Real& value)
	{
		if( _logScale ) _value = log( value );
		else            _value = value;
		_marker.update(_posX + _width*(_value/_range) - _markerSize/2);
	};
	void setRange( const Real& minValue , const Real& maxValue)
	{
		if( _logScale ) _minValue = log( minValue ) , _maxValue = log( maxValue );
		else            _minValue = minValue , _maxValue = maxValue;
		_range=abs(_maxValue-_minValue);
	};
	Real getValue() const
	{
		if( _logScale ) return exp( _value );
		else            return      _value  ;
	};
	void getValue( Real& v ) const
	{
		if( _logScale ) v = exp( _value );
		else            v =      _value  ;
	};
	void getRange( Real& min , Real& max ) const
	{
		if( _logScale ) min = exp( _minValue ) , max = exp( _maxValue );
		else            min = _minValue , max = _maxValue;
	}

	void setSize( int width , int height ){ _width = width , _height=height; };
	void setPosition(int posX, int posY){ _posX=posX; _posY=posY; _marker.initialize( _posX + _width*(_value-_minValue)/(_maxValue-_minValue) - _markerSize/2 , _posY - (_markerSize-_height)/2 );};
	void setColor( Point3D< Real > fill, Point3D< Real > line , Point3D< Real > marker ){ fillColor = fill , lineColor = line , _marker.setColor( marker , line ) ; }

	
	bool mouseFunc(int X, int Y);
	bool motionFunc( const Mouse& mouse );
	void drawOpenGL();
	bool hide;
};

template< class Real >
bool Slider<Real>::mouseFunc( int X, int Y )
{
	bool temp = _isInMotion;
	if     ( mouse.leftDown && _marker.isOnMarker(X, Y) )		_isInMotion = true;
	else if( !_locked && mouse.leftDown && _isOnSlider(X, Y) )  _isDragging = true;
	else														_isInMotion = _isDragging = false;
	return (temp && !mouse.leftDown);
}

template< class Real >
bool Slider<Real>::motionFunc( const Mouse& mouse )
{
	if( _isInMotion )
	{
		_value = Real(mouse.endX-_posX) / _width * _range + _minValue;
		if(_value>_maxValue) _value=_maxValue;
		if(_value<_minValue) _value=_minValue;
		_marker.update(_posX + _width*((_value-_minValue)/_range) - _markerSize/2 );
	}
	else if( _isDragging )
	{
		_posX+=mouse.shiftX; 
		_posY+=mouse.shiftY;
		_marker.initialize( _posX+_width*_value/(_maxValue-_minValue) - _markerSize/2 , _posY - (_markerSize-_height)/2 );
	}
	return (_isInMotion||_isDragging);
}

template< class Real >
void Slider< Real >::initialize(int posX, int posY, int width, int height , int markerDiameter , int thickness , Point3D< Real > fill , Point3D< Real > line , Point3D< Real > marker , Real minValue, Real maxValue, Real value, bool lock , bool logScale , bool writeValue )
{
	_logScale = logScale;
	_isInMotion = _isDragging = false;
	setSize( width, (height/4)*4 );
	setRange( minValue, maxValue );
	setValue( value );
	setPosition( posX, posY );
	_writeValue = writeValue;
	_thickness = thickness;
	_markerSize = markerDiameter;
	fillColor = fill;
	lineColor = line;
	_marker.initialize( posX + width*(_value-_minValue)/(_maxValue-_minValue) - _markerSize/2 , posY  - (_markerSize-height)/2 , _markerSize , _thickness , marker , line );
	_locked = lock;
	return;
};

template< class Real >
void Slider<Real>::drawOpenGL()
{
	if( hide ) return;
	int shift = _height/2;
	glLineWidth( _thickness );
	glColor3f( fillColor[0] , fillColor[1] , fillColor[2] );
	glBegin( GL_QUADS );
      glVertex2f( _posX       -shift , _posY         );
      glVertex2f( _posX+_width+shift , _posY         );
      glVertex2f( _posX+_width+shift , _posY+_height );
      glVertex2f( _posX       -shift , _posY+_height );
    glEnd( );

	glColor3f( lineColor[0] , lineColor[1] , lineColor[2] );
	glBegin(GL_LINE_LOOP);
      glVertex2f(_posX       -shift, _posY       );
      glVertex2f(_posX+_width+shift, _posY       );
      glVertex2f(_posX+_width+shift, _posY+_height);
      glVertex2f(_posX       -shift, _posY+_height);
    glEnd();

	glBegin(GL_LINE_LOOP);
      glVertex2f(_posX       , _posY+_height/2 );
      glVertex2f(_posX+_width, _posY+_height/2 );
    glEnd();

	_marker.drawOpenGL();

	if( _writeValue )
	{
		static char vString[32];
		sprintf( vString , " %f" , getValue() );
		int len=strlen(vString);

		glColor3f( lineColor[0] , lineColor[1] , lineColor[2] );
		glRasterPos2f(_posX+_width+shift,_posY);
		for(int i=0;i<len;i++) glutBitmapCharacter(GLUT_BITMAP_HELVETICA_18, vString[i]);
	}
}

////////////////////
// FunctionViewer //
////////////////////
template< class Real >
class FunctionViewer : public Drawable<Real>
{
private:
	int _thickness;
	Point3D< Real > _frameColor , _curveColor;
	int _posX, _posY, _width, _height;
	Real _maxValue, _minValue, _range, _unit;
	bool _isDragging, _locked;
	std::vector<Real>* _values;
	bool _isOnViewer(int x, int y){ return( x>_posX && y>_posY && x<_posX+_width && y<_posY+_height); };

public:
	void initialize( int posX , int posY , int width , int height , int thickness , Point3D< Real > curveColor , Point3D< Real > frameColor , Real minValue=0.0, Real maxValue=1.0, std::vector<Real>* values = NULL , bool lock=true);
	void setSize( int width , int height ){ _width=width; _height=height; if( _values) _unit=Real(_width)/(_values->size()-1); };
	void setPosition(int posX, int posY){ _posX=posX; _posY=posY; };
	void setRange(const Real& minValue, const Real& maxValue){ _minValue=minValue; _maxValue=maxValue; _range=abs(maxValue-minValue);};
	void setValues( std::vector<Real>* values ){_values=values; if(_values)_unit=Real(_width)/(_values->size()-1); };
	
	void mouseFunc(int X, int Y);
	bool motionFunc( const Mouse& mouse );
	void drawOpenGL();
	bool hide;
};

template< class Real >
void FunctionViewer<Real>::initialize( int posX, int posY, int width, int height, int thickness , Point3D< Real > curveColor , Point3D< Real > frameColor , Real minValue=0.0, Real maxValue=1.0, std::vector<Real>* values = NULL , bool lock=true )
{
	_thickness = thickness;
	_curveColor = curveColor;
	_frameColor = frameColor;
	_isDragging = false;
	setPosition( posX, posY );
	setSize( width, height );
	setRange( minValue, maxValue );
	setValues( values );
	_locked = lock;
}
template< class Real >
void FunctionViewer<Real>::mouseFunc( int X, int Y )
{
	if( !_locked && mouse.leftDown && _isOnViewer(X, Y) ) _isDragging = true;
	else												  _isDragging = false;
}
template< class Real >
bool FunctionViewer<Real>::motionFunc( const Mouse& mouse )
{
	if( _isDragging )_posX += mouse.shiftX, _posY += mouse.shiftY;
	return _isDragging;
}
template< class Real >
void FunctionViewer<Real>::drawOpenGL()
{
	if(hide) return;
	glLineWidth( _thickness );
	glColor4f( _curveColor[0] , _curveColor[1] , _curveColor[2] , 1 );

	if(_values)
	{
	  glBegin(GL_LINE_STRIP);
	  for(int i=0;i<_values->size();i++) glVertex2f(_unit*i+_posX , (_values->at(i)/_range)*_height + _posY );
      glEnd();
	}
	glLineWidth( _thickness );
	glColor4f( _frameColor[0] , _frameColor[1] , _frameColor[2] , 1 );
	glBegin(GL_LINE_LOOP);
      glVertex2f(_posX       , _posY       );
      glVertex2f(_posX+_width, _posY       );
      glVertex2f(_posX+_width, _posY+_height);
      glVertex2f(_posX       , _posY+_height);
    glEnd();
}


//////////////////
// MessageBoard //
//////////////////
template< class Real >
class MessageBoard : public Drawable<Real>
{
private:
	char message[1024];
	Point3D<Real> color3f;
	int _posX, _posY;
	bool _isDragging, _locked;
	bool _isOnViewer(int x, int y){ return( x>_posX && y>_posY && x<_posX+_width && y<_posY+_height); };

public:
	void initialize( int posX, int posY, Point3D<Real> color, bool lock=true){_posX=posX;_posY=posY;color3f=color;_locked=lock;};
	void setSize(int width, int height){ _width=width; _height=height; };
	void setPosition(int posX, int posY){ _posX=posX; _posY=posY; };
	void setColor( float R, float G, float B){color3f[0]=R;color3f[1]=G;color3f[2]=B;}
	void setMessage( char newMessage[] );

	void mouseFunc(int X, int Y);
	bool motionFunc( const Mouse& mouse );
	void drawOpenGL();
	bool hide;
};
template< class Real >
void MessageBoard<Real>::drawOpenGL()
{
	if(hide) return;
	glColor3f(color3f[0],color3f[1],color3f[2]);
	glRasterPos2f(_posX,_posY);
	int len=strlen(message);
	for(int i=0;i<len;i++){glutBitmapCharacter(GLUT_BITMAP_HELVETICA_18, message[i]);}
}
template< class Real >
void MessageBoard<Real>::setMessage( char newMessage[] )
{
	strcpy (message,newMessage);
}

template< class Real >
class SubWindow
{
protected:
	Real windowWidth , windowHeight;
	Point2D< Real > bottomLeft , topRight;
public:
	void SetPosition( Real width , Real height , Point2D< Real > bl , Point2D< Real > tr )
	{
		windowWidth  = width;
		windowHeight = height;
		bottomLeft[0] = bl[0] * width;
		bottomLeft[1] = bl[1] * height;
		topRight[0]   = tr[0] * width;
		topRight[1]   = tr[1] * height;
	}
};
///////////
// Plane //
///////////
template< class Real >
class Plane : public SubWindow< Real >
{
public:
	Point3D< Real > color;
	Real transparency;

	Plane( void ){ ; }
	void Initialize( Point3D< Real > color , Real transparency )
	{
		this->color = color;
		this->transparency = transparency;
	}
	void drawOpenGL( void )
	{
		float x , y;
		Point2D< Real > p;
		glDisable( GL_DEPTH_TEST );
		glDisable( GL_LIGHTING );

		glMatrixMode( GL_PROJECTION );
		glPushMatrix( );
		glLoadIdentity();
		glOrtho( 0 , windowWidth , 0 , windowHeight , 0 , 1 );
		glMatrixMode( GL_MODELVIEW );
		glPushMatrix( );
		glLoadIdentity();

		glEnable( GL_BLEND );
		glBlendFunc( GL_SRC_ALPHA , GL_ONE_MINUS_SRC_ALPHA );
		glColor4f( color[0] , color[1] , color[1] , transparency );
		glBegin( GL_QUADS );
		{
			Point2D< Real > p;
			glVertex2f( bottomLeft[0] , bottomLeft[1] );
			glVertex2f(   topRight[0] , bottomLeft[1] );
			glVertex2f(   topRight[0] ,   topRight[1] );
			glVertex2f( bottomLeft[0] ,   topRight[1] );
		}
		glEnd();
		glDisable( GL_BLEND );

		glPopMatrix( );
		glMatrixMode( GL_PROJECTION );
		glPopMatrix( );
		glEnable( GL_LIGHTING );
		glEnable( GL_DEPTH_TEST );
	}
};
///////////////
// Histogram //
///////////////
template< class Real >
class Histogram : public SubWindow< Real >
{
	int bins;
	Real* binValues;
	Real xMin , xMax;
	Real xPosition , yPosition;
public:
	int lineWidth;
	Point3D< Real > color , lineColor;
	Point3D< Real > (*colorFunction)( Real );
	Real transparency;

	Histogram( void )
	{
		bins = 0;
		binValues = NULL;
		colorFunction = NULL;
		lineWidth = 2;
		color = Point3D< Real >( 0.5 , 0.5 , 0.5 );
		lineColor = Point3D< Real >( 0. , 0. , 0. );
		transparency = 0.5;
	}
	void Initialize( Real xMin , Real xMax , int bins )
	{
		this->xMin = xMin;
		this->xMax = xMax;
		if( binValues ) delete[] binValues;
		binValues = NULL;
		this->bins = bins;
		binValues = new Real[ bins ];
		for( int i=0 ; i<bins ; i++ ) binValues[i] = 0;
	}
	void AddEntry( Real x , Real weight , bool clamp=false )
	{
		Real b = ( x - xMin ) / ( xMax - xMin );
		b  *= bins-1;
		int b1 = int( b );
		int b2 = b1 + 1;
		Real db = b - b1;
		if( clamp )
		{
			if( b1<0 ) b1 = 0;
			if( b2<0 ) b2 = 0;
			if( b1>=bins ) b1 = bins-1;
			if( b2>=bins ) b2 = bins-1;
		}
		if( b1>=0 && b1<bins ) binValues[b1] += (1.-db) * weight;
		if( b2>=0 && b2<bins ) binValues[b2] +=     db  * weight;
	}
	void Normalize( void )
	{
		Real sum = 0;
		for( int i=0 ; i<bins ; i++ ) sum += binValues[i];
		for( int i=0 ; i<bins ; i++ ) binValues[i] /= sum;
	}
	void SetMax( Real maxValue )
	{
		Real max = 0;
		for( int i=0 ; i<bins ; i++ ) if( binValues[i]>max ) max = binValues[i];
		for( int i=0 ; i<bins ; i++ ) binValues[i] *= maxValue / max;
	}

	void drawOpenGL( void )
	{
		glDisable( GL_DEPTH_TEST );
		glDisable( GL_LIGHTING );
		glEnable( GL_BLEND );

		glMatrixMode( GL_PROJECTION );
		glPushMatrix( );
		glLoadIdentity( );
		glOrtho( 0 , windowWidth , 0 , windowHeight , 0 , 1 );
		glMatrixMode( GL_MODELVIEW );
		glPushMatrix( );
		glLoadIdentity( );

		Real width  = topRight[0] - bottomLeft[0];
		Real height = topRight[1] - bottomLeft[1];

		glBlendFunc( GL_SRC_ALPHA , GL_ONE_MINUS_SRC_ALPHA );
		glColor4f( color[0] , color[1] , color[2] , transparency );
		glBegin( GL_QUADS );
		for( int i=0 ; i<bins ; i++ )
		{
			if( colorFunction )
			{
				Real x = xMin + ( (xMax-xMin) * i ) / (bins-1);
				Point3D< Real > c = colorFunction( x );
				glColor4f( c[0] , c[1] , c[2]  , transparency );
			}
			Real xStart = bottomLeft[0] + (  width *  i    ) / bins;
			Real xEnd   = bottomLeft[0] + (  width * (i+1) ) / bins;
			Real yStart = bottomLeft[1];
			Real yEnd   = bottomLeft[1] + ( height * binValues[i] );
			glVertex2f( xStart , yStart );
			glVertex2f( xEnd   , yStart );
			glVertex2f( xEnd   , yEnd   );
			glVertex2f( xStart , yEnd   );
		}
		glEnd();
		glDisable( GL_BLEND );

		glLineWidth( lineWidth );
		glColor4f( lineColor[0] , lineColor[1] , lineColor[2] , transparency );
		for( int i=0 ; i<bins ; i++ )
		{
			glBegin( GL_LINE_LOOP );
			Real xStart = bottomLeft[0] + ( width *  i    ) / bins;
			Real xEnd   = bottomLeft[0] + ( width * (i+1) ) / bins;
			Real yStart = bottomLeft[1];
			Real yEnd   = bottomLeft[1] + ( height * binValues[i] );
			glVertex2f( xStart , yStart );
			glVertex2f( xEnd   , yStart );
			glVertex2f( xEnd   , yEnd   );
			glVertex2f( xStart , yEnd   );
			glEnd();
		}

		glPopMatrix( );
		glMatrixMode( GL_PROJECTION );
		glPopMatrix( );
		glEnable( GL_LIGHTING );
		glEnable( GL_DEPTH_TEST );
	}
};

/////////////////
// KnottedPlot //
/////////////////
#include "spline.h"
template< class Real >
class KnottedPlot : public SubWindow< Real >
{
	const static int POINT_SIZE=10;
	int selectedIndex , overIndex;
	int stepNum;
	int lineWidth;
	int newPoint;
	float lineHeight;
	int dim;
	Real xMin , yMin , xMax , yMax;
	Real defaultValue;
	Real xPosition,yPosition;

	struct PlotPoint
	{
		Point2D< Real > point;
		int multiplicity;
		PlotPoint( void ) { point = Point2D< Real >( ) , multiplicity = 2; }
		PlotPoint( Point2D< Real > p , int m=1 ) { point = p , multiplicity = m; }
	};
	int addPoint( PlotPoint p );
	void removePoint( int idx );
	int reorderPoints( int s );

	Point2D< Real > getScreenPoint( Point2D< Real > p );
	Point2D< Real > getSpacePoint ( Point2D< Real > p );
	Real* knots;
	int * indices;
	std::vector< PlotPoint > points;
public:
	bool drawKnots;
	bool lockPlot;
	Point3D< Real > color;
	int resetKnots( void );
	Mouse mouse;

	KnottedPlot( void );
	void Initialize( Real xMin , Real xMax , Real yMin , Real yMax , Point3D< Real > color=Point3D< Real >( ) );
	void SetDefault( Real value );

	bool MouseDown( int , int , int );
	bool MouseUp  ( int , int , int );
	bool DepressedMotion( int , int );
	bool PassiveMotion  ( int , int );
	bool Keyboard( unsigned char , int , int );
	bool SpecialFunction( int key, int x, int y );

	void IncreaseHeight( Real value );

	void drawOpenGL( void );

	void setStepNum( int e );
	void setLineWidth( int e );
	void setDimension( int d );
	Real getValue( Real t ) const;
	void getValues( Real* values , int count ) const;
};
template< class Real >
KnottedPlot< Real >::KnottedPlot( void )
{
	knots      = NULL;
	indices    = NULL;
	stepNum    = 400;
	lineWidth  = 4;
	lineHeight = 10;
	lockPlot   = false;
	dim = 4;
	selectedIndex = -1;
	overIndex = -1;
	drawKnots	= true;
	Initialize( 0 , 1 , 0 , 1 , Point3D< Real >( ) );
}

template< class Real >
void KnottedPlot< Real >::Initialize( Real xMin , Real xMax , Real yMin , Real yMax , Point3D< Real > color )
{
	this->xMin = xMin;
	this->xMax = xMax;
	this->yMin = yMin;
	this->yMax = yMax;
	this->color = color;
	knots = NULL;
	indices = NULL;
	points.clear();
	addPoint( PlotPoint( Point2D< Real >( xMin , yMin ) ) );
	addPoint( PlotPoint( Point2D< Real >( xMax , yMax ) ) );
}
template< class Real >
void KnottedPlot< Real >::SetDefault( Real value )
{
	if( value<yMin ) value = yMin;
	if( value>yMax ) value = yMax;
	defaultValue = value;
	for( int i=0 ; i<points.size() ; i++ ) points[i].point[1] = value;
}

template< class Real >
Point2D< Real > KnottedPlot< Real >::getSpacePoint( Point2D< Real > p )
{
	Real x = ( p[0] - bottomLeft[0] ) / ( topRight[0] - bottomLeft[0] ) * ( xMax - xMin ) + xMin;
	Real y = ( p[1] - bottomLeft[1] ) / ( topRight[1] - bottomLeft[1] ) * ( yMax - yMin ) + yMin;
	return Point2D< Real >( x , y );
}
template< class Real >
Point2D< Real > KnottedPlot< Real >::getScreenPoint( Point2D< Real > p )
{
	Real x = ( p[0] - xMin ) / ( xMax - xMin ) * ( topRight[0] - bottomLeft[0] ) + bottomLeft[0];
	Real y = ( p[1] - yMin ) / ( yMax - yMin ) * ( topRight[1] - bottomLeft[1] ) + bottomLeft[1];
	return Point2D< Real >( x , y );
}

template< class Real >
int KnottedPlot< Real >::resetKnots( void )
{
	if( knots ) delete[] knots;
	if( indices ) delete[] indices;
	knots   = NULL;
	indices = NULL;
	int d=0;
	for( int i=0 ; i<points.size() ; i++ ) d += points[i].multiplicity;
	indices = new int[ d ];
	d += 2*(dim-1);
	knots = new Real[d];
	for( int i=0 ; i<dim-1 ; i++ ) knots[i] = xMin , knots[d-1-i] = xMax;
	for( int i=0 , idx=0 ; i<points.size() ; i++ ) for( int j=0 ; j<points[i].multiplicity ; j++ , idx++ ) knots[idx+dim-1] = points[i].point[0] , indices[idx] = i;
	return 1;
}
template< class Real >
int KnottedPlot< Real >::addPoint( PlotPoint p )
{
	int i;
	for( i=0 ; i<points.size() ; i++ ) if( p.point[0]<points[i].point[0] ) break;
	points.push_back( PlotPoint() );
	for( int j=points.size()-1 ; j>i ; j-- ) points[j]=points[j-1];
	points[i]=p;
	resetKnots();
	return i;
}
template< class Real >
void KnottedPlot< Real >::removePoint( int idx )
{
	if( idx<0 || idx>=points.size() ) return;
	if     ( overIndex==idx ) overIndex = -1;
	else if( overIndex> idx ) overIndex--;
	for( int i=idx ; i<points.size()-1 ; i++ ) points[i]=points[i+1];
	points.pop_back();
	resetKnots();
}
template< class Real >
int KnottedPlot< Real >::reorderPoints( int s )
{
	for( int i=0 ; i<points.size() ; i++ )
		for( int j=0 ; j<i ; j++ )
			if( points[j][0]>points[i][0] )
			{
				Point2D< Real > temp = points[j];
				points[j] = points[i];
				points[i] = temp;
				if     (i==s) s=j;
				else if(j==s) s=i;
			}
	resetKnots();
	return s;
}
template< class Real >
bool KnottedPlot< Real >::Keyboard( unsigned char c , int , int )
{
	if( lockPlot ) return  false;
	switch( c )
	{
	case 'r':
		for( int i=0 ; i<points.size() ; i++ ) points[i].point[1] = yMin + yMax - points[i].point[1];
		return true;
	case 'R':
		for( int i=0 ; i<points.size() ; i++ ) points[i].point[0] = xMin + xMax - points[i].point[0];
		for( int i=0 ; i<points.size()/2 ; i++ )
		{
			PlotPoint temp = points[i];
			points[i] = points[points.size()-1-i];
			points[points.size()-1-i] = temp;
		}
		resetKnots( );
		return true;
	case ' ':
		for( int i=points.size()-2 ; i>=1 ; i-- ) removePoint(i);
		points[0].point[1] = points[1].point[1] = defaultValue - (yMax-yMin) * 0.03;
		return true;
	case '1':
		for( int i=points.size()-2 ; i>=1 ; i-- ) removePoint(i);
		points[0].point[1] = yMin/3*2;
		points[1].point[1] = yMax/3*2;
		addPoint( PlotPoint( Point2D< Real >( (3.*xMin+xMax)/4 , yMin/3*2 ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( (xMin+3.*xMax)/4 , yMax/3*2 ) , 2 ) );
		return true;
	case '2':
		for( int i=points.size()-2 ; i>=1 ; i-- ) removePoint(i);
		points[0].point[1] = yMax/3*2;
		points[1].point[1] = yMin/3*2;
		addPoint( PlotPoint( Point2D< Real >( (3.*xMin+xMax)/4 , yMax/3*2 ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( (xMin+3.*xMax)/4 , yMin/3*2 ) , 2 ) );
		return true;
	case '3':
		for( int i=points.size()-2 ; i>=1 ; i-- ) removePoint(i);
		points[0].point[1] = points[1].point[1] = yMin/3*2;
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3   , yMin/3*2 ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3   , yMax/3*2 ) , 2 ) );
//		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/9*4 , yMax/3*2 ) , 2 ) );
//		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/9*5 , yMax/3*2 ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3*2 , yMax/3*2 ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3*2 , yMin/3*2 ) , 2 ) );
		resetKnots();
		return true;
	case '4':
		for( int i=points.size()-2 ; i>=1 ; i-- ) removePoint(i);
		points[0].point[1] = points[1].point[1] = yMax/3*2;
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3   , yMax/3*2 ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3   , yMin/3*2 ) , 2 ) );
//		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/9*4 , yMin/3*2 ) , 2 ) );
//		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/9*5 , yMin/3*2 ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3*2 , yMin/3*2 ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3*2 , yMax/3*2 ) , 2 ) );
		resetKnots();
		return true;
	case '=':
		for( int i=1 ; i<points.size()-1 ; i++ )
		{
			points[i].point[0] = ( points[i].point[0] - (xMin+xMax)/2 ) * 1.1 + (xMin+xMax)/2;
			if( points[i].point[0]<xMin ) points[i].point[0] = xMin;
			if( points[i].point[0]>xMax ) points[i].point[0] = xMax;
		}
		resetKnots();
		return true;
	case '-':
		for( int i=1 ; i<points.size()-1 ; i++ )
		{
			points[i].point[0] = ( points[i].point[0] - (xMin+xMax)/2 ) / 1.1 + (xMin+xMax)/2;
			if( points[i].point[0]<xMin ) points[i].point[0] = xMin;
			if( points[i].point[0]>xMax ) points[i].point[0] = xMax;
		}
		resetKnots();
		return true;
	case '+':
		for( int i=0 ; i<points.size() ; i++ )
		{
			points[i].point[1] = ( points[i].point[1] - (yMin+yMax)/2 ) * 1.1 + (yMin+yMax)/2;
			if( points[i].point[1]<yMin ) points[i].point[1] = yMin;
			if( points[i].point[1]>yMax ) points[i].point[1] = yMax;
		}
		resetKnots();
		return true;
	case '_':
		for( int i=0 ; i<points.size() ; i++ )
		{
			points[i].point[1] = ( points[i].point[1] - (yMin+yMax)/2 ) / 1.1 + (yMin+yMax)/2;
			if( points[i].point[1]<yMin ) points[i].point[1] = yMin;
			if( points[i].point[1]>yMax ) points[i].point[1] = yMax;
		}
		resetKnots();
		return true;
		/*
	case 'b':
		for( int i=points.size()-2 ; i>=1 ; i-- ) removePoint(i);
		points[0].point[1] = points[1].point[1] = yMin;
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3   , defaultValue ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3*2 , defaultValue ) , 2 ) );
		resetKnots();
		return true;
	case 'B':
		for( int i=points.size()-2 ; i>=1 ; i-- ) removePoint(i);
		points[0].point[1] = points[1].point[1] = yMin;
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3   - (xMax-xMin)/1000 , yMin ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3   + (xMax-xMin)/1000 , defaultValue ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3*2 - (xMax-xMin)/1000 , defaultValue ) , 2 ) );
		addPoint( PlotPoint( Point2D< Real >( xMin + (xMax-xMin)/3*2 + (xMax-xMin)/1000 , yMin ) , 2 ) );
		resetKnots();
		return true;
		*/
	case 'k':
	case 'K':
		drawKnots = !drawKnots;
		return true;
	}
	return false;
}
template< class Real >
bool KnottedPlot< Real >::SpecialFunction( int key , int x , int y )
{
	if( lockPlot ) return false;
	switch( key )
	{
		case GLUT_KEY_UP:
			for( int i=0 ; i<points.size() ; i++ )
			{
				points[i].point[1] += (yMax-yMin) * 0.025;
				if( points[i].point[1]>yMax ) points[i].point[1] = yMax;
			}
			return true;
		case GLUT_KEY_DOWN:
			for( int i=0 ; i<points.size() ; i++ )
			{
				points[i].point[1] -= (yMax-yMin) * 0.025;
				if( points[i].point[1]<yMin) points[i].point[1] = yMin;
			}
			return true;
		case GLUT_KEY_RIGHT:
			for( int i=1 ; i<points.size()-1 ; i++ )
			{
				points[i].point[0] += (xMax-xMin) * 0.025;
				if( points[i].point[0]>xMax ) points[i].point[1] = xMax;
			}
			resetKnots( );
			return true;
		case GLUT_KEY_LEFT:
			for( int i=1 ; i<points.size()-1 ; i++ )
			{
				points[i].point[0] -= (xMax-xMin) * 0.025;
				if( points[i].point[0]<xMin ) points[i].point[1] = xMin;
			}
			resetKnots( );
			return true;
	}

	return false;
}
template< class Real >
void KnottedPlot< Real >::IncreaseHeight( Real value )
{
	for( int i=0 ; i<points.size() ; i++ )
			{
				points[i].point[1] += (yMax-yMin) * value;
				if( points[i].point[1]<yMin) points[i].point[1] = yMin;
				if( points[i].point[1]>yMax) points[i].point[1] = yMax;
			}
	resetKnots( );
}

template< class Real >
bool KnottedPlot< Real >::PassiveMotion( int x , int y )
{
	if( lockPlot ) return false;
	Point2D< Real > p = getSpacePoint( Point2D< Real >( x , y ) );
	if( !( p[0]>=xMin && p[0]<=xMax && p[1]>=yMin && p[1]<=yMax ) ) return false;
	xPosition = p[0];
	yPosition = p[1];
	overIndex = -1;
	double d = 2*POINT_SIZE;
	for( int i=0 ; i<points.size() ; i++ )
	{
		Real temp = fabs( double( x - getScreenPoint( points[i].point )[0] ) );
		if( !i || temp<d )
		{
			d = temp;
			overIndex=i;
		}
	}
	if( d>POINT_SIZE ) overIndex = -1;
	return true;
}

template< class Real >
bool KnottedPlot< Real >::DepressedMotion( int x , int y )
{
	if( lockPlot ) return false;
	Point2D< Real > p = getSpacePoint( Point2D< Real >( mouse.startX , mouse.startY ) );
	if( !( p[0]>=xMin && p[0]<=xMax && p[1]>=yMin && p[1]<=yMax ) ) return false;

	mouse.move( x , y );
	p = getSpacePoint( Point2D< Real >( x , y ) );
	xPosition = p[0];
	yPosition = p[1];
	if( mouse.leftDown && selectedIndex>=0 )
	{
		if( p[0]<xMin ) p[0] = xMin;
		if( p[0]>xMax ) p[0] = xMax;
		if( p[1]<yMin ) p[1] = yMin;
		if( p[1]>yMax ) p[1] = yMax;
		if( selectedIndex==0 || selectedIndex==points.size()-1 ) points[selectedIndex].point[1]=p[1];
		else
		{
			if( p[0]<points[selectedIndex-1].point[0] ) p[0] = points[selectedIndex-1].point[0];
			if( p[0]>points[selectedIndex+1].point[0] ) p[0] = points[selectedIndex+1].point[0];
			points[selectedIndex].point=p;
		}
		resetKnots();
		return true;
	}
	else return false;
}

template< class Real >
bool KnottedPlot< Real >::MouseDown( int button , int x , int y )
{
	if( lockPlot ) return false;
	Point2D< Real > p = getSpacePoint( Point2D< Real >( x , y ) );
	if( !( p[0]>=xMin && p[0]<=xMax && p[1]>=yMin && p[1]<=yMax ) ) return false;

	selectedIndex=-1;
	mouse.update( button , GLUT_DOWN , x , y );
	newPoint=0;

	if( mouse.scrollDown ) return false;

	if( mouse.leftDown )
	{
		Real d = 2*POINT_SIZE*POINT_SIZE;
		// Check for the closest point
		for( int i=0 ; i<points.size() ; i++ )
		{
			Real temp = Point2D< Real >::SquareNorm( Point2D< Real >( x , y ) - getScreenPoint( points[i].point ) );
			if( !i || temp<d )
			{
				d=temp;
				selectedIndex=i;
			}
		}
		// If its close enough, we're done
		if( d<=POINT_SIZE*POINT_SIZE )
		{
			overIndex = -1;
			return true;
		}
		// Otherwise, check for the closest line
		d = 2*POINT_SIZE*POINT_SIZE;
		for( int i=0 ; i<points.size() ; i++ )
		{
			Real temp = ( x - getScreenPoint( points[i].point )[0] ) * ( x - getScreenPoint( points[i].point )[0] );
			if( !i || temp<d )
			{
				d=temp;
				selectedIndex=i;
			}
		}
		// If its close enough, we're done
		if( d<=POINT_SIZE*POINT_SIZE )
		{
			overIndex = -1;
			newPoint = 1;
			points[selectedIndex].point[1] = getSpacePoint( Point2D< Real >( x , y ) )[1];
			return true;
		}

		// Otherwise, add a new point
		if( mouse.shiftDown ) selectedIndex = addPoint( PlotPoint( getSpacePoint( Point2D< Real >( x , y ) ) , 1 ) );
		else                  selectedIndex = addPoint( PlotPoint( getSpacePoint( Point2D< Real >( x , y ) ) , 2 ) );
		newPoint = 1;
	}
	return ( selectedIndex>=0 );
}

template< class Real >
bool KnottedPlot< Real >::MouseUp( int button , int x , int y )
{
	if( lockPlot ) return false;
	Point2D< Real > p = getSpacePoint( Point2D< Real >( mouse.startX , mouse.startY ) );
	if( !( p[0]>=xMin && p[0]<=xMax && p[1]>=yMin && p[1]<=yMax ) ) return false;
	if( mouse.leftDown )
	{
		if     ( selectedIndex==-1 ) addPoint( getSpacePoint( Point2D< Real >( x , y ) ) );
		else if( selectedIndex!=-1 && fabs( double(mouse.startX-x) )<1 && fabs( double(mouse.startY-y) )<1 && !newPoint )
		{
			if     ( selectedIndex!=0 && selectedIndex!=points.size()-1) removePoint( selectedIndex );
			else if( selectedIndex==0               ) points[selectedIndex].point[1] = points[selectedIndex+1].point[1];
			else if( selectedIndex==points.size()-1 ) points[selectedIndex].point[1] = points[selectedIndex-1].point[1];
		}
	}
	mouse.update( button , GLUT_UP , x , y );
	selectedIndex = -1;
	return true;
}
template< class Real >
Real KnottedPlot< Real >::getValue( Real t ) const
{
	float y=0;
	int d = 0;
	for( int j=0 ; j<points.size() ; j++ ) d += points[j].multiplicity;
	for( int j=0 ; j<d+dim-2 ; j++ )
	{
		int idx=j-(dim/2-1);
		if( idx<0 )       y += NonUniformNonRationalBSpline< Real >::GetWeight( t , j , dim , &knots[0] ) * points[0].point[1];
		else if( idx>=d ) y += NonUniformNonRationalBSpline< Real >::GetWeight( t , j , dim , &knots[0] ) * points[points.size()-1].point[1];
		else              y += NonUniformNonRationalBSpline< Real >::GetWeight( t , j , dim , &knots[0] ) * points[ indices[ idx ] ].point[1];
	}
	return y;
}
template< class Real >
void KnottedPlot< Real >::getValues( Real* values , int count ) const
{
	for( int i=0 ; i<count ; i++ )
	{
		Real x = xMin + ( (xMax-xMin) * (i+0.5) ) / count;
		values[i] = getValue( x );
	}
}
template< class Real > void KnottedPlot< Real >::setLineWidth( int e ){ lineWidth = e; }
template< class Real > void KnottedPlot< Real >::setStepNum  ( int e ){ stepNum   = e; }
template< class Real > void KnottedPlot< Real >::setDimension( int d ){ dim = d , resetKnots(); }

template< class Real >
void KnottedPlot< Real >::drawOpenGL( void )
{
	int i;
	float x , y;
	Point2D< Real > p;
	glDisable( GL_DEPTH_TEST );
	glDisable( GL_LIGHTING );

	glMatrixMode( GL_PROJECTION );
	glPushMatrix( );
	glLoadIdentity();
	glOrtho( 0 , windowWidth , 0 , windowHeight , 0 , 1 );
	glMatrixMode( GL_MODELVIEW );
	glPushMatrix( );
	glLoadIdentity();

	glLineWidth( lineWidth );
	glBegin( GL_LINE_STRIP );
	glColor3f( color[0] , color[1] , color[2] );
	for( i=0 ; i<stepNum ; i++ )
	{
		x = xMin+((xMax-xMin)*.99999*i)/(stepNum-1);
//		x = xMin+((xMax-xMin)*i)/(stepNum);
		y = getValue( x );
		Point2D< Real > p = getScreenPoint( Point2D< Real >( x , y ) );
		glVertex2f( p[0] , p[1] );
	}
	glEnd();

	if( !lockPlot && drawKnots )
	{
		// Draw the lines
		glColor3f( color[0]*.65 , color[1]*.65 , color[2]*.65 );
		glEnable( GL_LINE_STIPPLE );
		glLineStipple( 1,0x0F0F );
		if( selectedIndex!=-1 )
		{
			glBegin( GL_LINE_STRIP );
			p = getScreenPoint( Point2D< Real >( points[selectedIndex].point[0] , yMin ) );
			glVertex2f( p[0] , p[1] );
			p = getScreenPoint( Point2D< Real >( points[selectedIndex].point[0] , yMax ) );
			glVertex2f( p[0] , p[1] );
			glEnd();
		}
		if( overIndex!=-1 && overIndex!=selectedIndex )
		{
			glBegin( GL_LINE_STRIP );
			p = getScreenPoint( Point2D< Real >( points[overIndex].point[0] , yMin ) );
			glVertex2f( p[0] , p[1] );
			p = getScreenPoint( Point2D< Real >( points[overIndex].point[0] , yMax ) );
			glVertex2f( p[0] , p[1] );
			glEnd();
		}
		glDisable( GL_LINE_STIPPLE );

		// Draw the points
		glColor3f( color[0]*.5 , color[1]*.5 , color[2]*.5 );
		glPointSize( POINT_SIZE );
		glBegin( GL_POINTS );
		for( i=0 ; i<points.size() ; i++ )
		{
			p = getScreenPoint( points[i].point );
			glVertex2f( p[0] , p[1] );
		}
		glEnd();
	}

	glPopMatrix( );
	glMatrixMode( GL_PROJECTION );
	glPopMatrix( );
	glEnable( GL_LIGHTING );
	glEnable( GL_DEPTH_TEST );
}
