/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#include "glCamera.h"
#include "mouse.h"

class glStage
{
public:
	enum
	{
		NO_WIREFRAME , 
		FULL_WIREFRAME , 
		SILHOUETTE_WIREFRAME ,
		WIREFRAME_COUNT
	};
	bool drawEdges , drawTriangles;
	int wireframeState;
	Point3D< float > color;
	Point3D< float > center;
	float radius;

	unsigned int vbo , tbo;
	
	// Visualization
	std::vector< Point3D<float> > vertices;
	std::vector< TriangleIndex > triangles;
	std::vector< std::pair< int , int > > edges;

	glCamera* camera;
	void drawOpenGL( void );
	void subDrawOpenGL( void );


	void setTriangleBufferObject( void );
	void setVertexBufferObject( const float* wValues=NULL );

	bool KeyboardFunction( unsigned char c );
	bool SpecialFunction( int key );
	bool MotionFunction( Mouse mouse );
	void Reshape( int width , int height );

	~glStage( void ); 
	glStage( void );
};
//=============================================== End of Header ===================================================
glStage::~glStage( void ){ if(camera){delete camera;} }
glStage::glStage( void )
{
	camera = new glCamera();
	drawTriangles  = true;
	drawEdges      = true;
	wireframeState = NO_WIREFRAME;
	vbo = tbo = 0;
} 

void glStage::subDrawOpenGL( void )
{
	glColor3f( color[0] , color[1] , color[2] );
	glEnableClientState( GL_VERTEX_ARRAY );

	glBindBuffer( GL_ARRAY_BUFFER , vbo );
	glVertexPointer( 4 , GL_FLOAT , sizeof(GLfloat)*4 , 0 );

	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , tbo );
	glDrawElements( GL_TRIANGLES , triangles.size()*3 , GL_UNSIGNED_INT, 0 );

	glDisableClientState( GL_VERTEX_ARRAY );
}
void glStage::drawOpenGL( void )
{
	camera->drawOpenGL();
	glMatrixMode( GL_MODELVIEW );
	glPushMatrix();

	if( drawTriangles || wireframeState!=NO_WIREFRAME )
	{
		glPolygonMode( GL_FRONT_AND_BACK , GL_FILL );
		if( !drawTriangles ) glColorMask( GL_FALSE , GL_FALSE , GL_FALSE , GL_FALSE );
		subDrawOpenGL( );
		if( !drawTriangles ) glColorMask( GL_TRUE , GL_TRUE , GL_TRUE , GL_TRUE );
	}

	if( wireframeState==SILHOUETTE_WIREFRAME )
	{
		float lineWidth = 2;
		Point3D< float > edgeColor( 0 , 0 , 0 );
		glEnable( GL_POLYGON_OFFSET_LINE );
		glPolygonOffset( lineWidth+1 , (lineWidth+1)*2 );
		glDisable( GL_LIGHTING );
		glDepthMask( GL_FALSE );
		glColor3f( edgeColor[0] , edgeColor[1] , edgeColor[2] );
		glLineWidth( lineWidth );
		glPolygonMode( GL_FRONT_AND_BACK , GL_LINE );
		subDrawOpenGL( );
		glPolygonMode( GL_FRONT_AND_BACK , GL_FILL );
		glDepthMask( GL_TRUE );
		glEnable( GL_LIGHTING );
		glDisable( GL_POLYGON_OFFSET_LINE );
	}

	if( wireframeState==FULL_WIREFRAME )
	{
		glDisable( GL_LIGHTING );
		glEnable( GL_POLYGON_OFFSET_LINE );
		glPolygonOffset( -1 , -2 );
		glLineWidth( 1 );
		glPolygonMode( GL_FRONT_AND_BACK , GL_LINE );
		subDrawOpenGL( );
		glPolygonMode( GL_FRONT_AND_BACK , GL_FILL );
		glDisable( GL_POLYGON_OFFSET_LINE );
	}
	if( drawEdges )
	{
		Point3D< float > edgeColor( 0 , 0 , 0 );
		float edgeWidth = 3.f;
		glEnable( GL_POLYGON_OFFSET_FILL );
		glDisable( GL_LIGHTING );
		glPolygonOffset( -1 , -2 );
		glLineWidth( edgeWidth );
		glColor3f( edgeColor[0] , edgeColor[1] , edgeColor[2] );
		glBegin( GL_LINES );
		{
			for( int i=0 ; i<edges.size() ; i++ )
			{
				Point3D< float > v[2] = { vertices[edges[i].first], vertices[edges[i].second] };
				glVertex3f( v[0][0], v[0][1], v[0][2] );
				glVertex3f( v[1][0], v[1][1], v[1][2] );
			}
		}
		glEnd();
		glEnable( GL_LIGHTING );
		glDisable( GL_POLYGON_OFFSET_FILL );
	}
	glPopMatrix();
	return;
}
bool glStage::KeyboardFunction( unsigned char c )
{
	switch( c )
	{
	case 'c':
		if( glIsEnabled( GL_CULL_FACE ) )
		{
			GLint face;
			glGetIntegerv( GL_CULL_FACE_MODE , &face );
			if( face==GL_BACK ) glCullFace( GL_FRONT );
			else                glDisable( GL_CULL_FACE );
		}
		else glEnable( GL_CULL_FACE ) , glCullFace( GL_BACK );
		break;
	case 'e':
		wireframeState = (wireframeState+1) % WIREFRAME_COUNT;
		break;
	case 't':
		drawTriangles = !drawTriangles;
		break;
	case 'b':
		drawEdges = !drawEdges;
		break;
	case 'W': camera->rotateUp   ( center, -PI/8  ); break;
	case 23 : camera->rotateUp   ( center, -PI/120 ); break;
	case 'w': camera->rotateUp   ( center, -0.025 ); break;

	case 'Q': camera->rotateUp   ( center,  PI/8  ); break;
	case 17 : camera->rotateUp   ( center,  PI/120 ); break;
	case 'q': camera->rotateUp   ( center,  0.025 ); break;

	case 'Z': camera->rotateRight( center, -PI/8  ); break;
	case 26 : camera->rotateRight( center, -PI/120 ); break;
	case 'z': camera->rotateRight( center, -0.025 ); break;

	case 'A': camera->rotateRight( center,  PI/8  ); break;
	case 1  : camera->rotateRight( center,  PI/120 ); break;
	case 'a': camera->rotateRight( center,  0.025 ); break;

	case 'X': camera->rotateDir  ( center, -PI/8  ); break;
	case 24 : camera->rotateDir  ( center, -PI/120 ); break;
	case 'x': camera->rotateDir  ( center, -0.025 ); break;

	case 'S': camera->rotateDir  ( center,  PI/8  ); break;
	case 19 : camera->rotateDir  ( center,  PI/120 ); break;
	case 's': camera->rotateDir  ( center,  0.025 ); break;
	default: return false;
	}
	return true;
}
bool glStage::SpecialFunction( int key )
{
	switch( key )
	{
	case GLUT_KEY_PAGE_UP:   camera->moveForward(  radius/30 ) ; break;
	case GLUT_KEY_PAGE_DOWN: camera->moveForward( -radius/30 ) ; break;
	case GLUT_KEY_RIGHT:     camera->moveRight  ( -radius/30 ) ; break;
	case GLUT_KEY_LEFT:      camera->moveRight  (  radius/30 ) ; break;
	case GLUT_KEY_UP:        camera->moveUp     ( -radius/30 ) ; break;
	case GLUT_KEY_DOWN:      camera->moveUp     (  radius/30 ) ; break;
	default: return false;
	}
	return true;
}
bool glStage::MotionFunction( Mouse mouse )
{
	// Camera transformation
	if( mouse.middleDown || (mouse.leftDown && mouse.shiftDown) )
	{
		camera->rotateUp   ( center , -0.01 *  mouse.shiftX );
		camera->rotateRight( center , -0.01 * -mouse.shiftY );
	}
	// More camera transformations
	else if( mouse.leftDown )
	{  
		if( mouse.altDown ) camera->moveForward(  radius/30  * -mouse.shiftY );
		else                camera->moveRight  ( -radius/250 *  mouse.shiftX ) , camera->moveUp( radius/250 * -mouse.shiftY );   
	}
	else return false;
	return true;
}
void glStage::Reshape( int width , int height )
{
	float d = radius + sqrt( Point3D<float>::SquareNorm( camera->position - center ) );
	glMatrixMode( GL_PROJECTION );
	glLoadIdentity( );
	gluPerspective( camera->heightAngle , ((float)width) / height , .1*d , 1*d );
	glViewport( 0 , 0 , width , height );
	camera->drawOpenGL();
}
void glStage::setTriangleBufferObject( void )
{
	if( !tbo ) glGenBuffers( 1, &tbo);
	unsigned int tSize = triangles.size()*3;
	GLuint  *cuTriangles = new GLuint [ tSize ];
	for( int i=0 ; i<triangles.size() ; i++ ) for( int j=0 ; j<3 ; j++ ) cuTriangles[i*3+j] = triangles[i][j];
	glBindBuffer( GL_ELEMENT_ARRAY_BUFFER , tbo );
	glBufferData( GL_ELEMENT_ARRAY_BUFFER , tSize*sizeof(GLuint) , cuTriangles , GL_STATIC_DRAW );
	delete[] cuTriangles;
}
void glStage::setVertexBufferObject( const float* wValues )
{
	if( !vbo ) glGenBuffers( 1, &vbo );
	unsigned int vSize = vertices.size()*4;
	GLfloat *cuVertices = new GLfloat[ vSize ];
	for( int i=0 ; i<vertices.size() ; i++ )
	{
		for( int j=0 ; j<3 ; j++ ) cuVertices[i*4+j] = (GLfloat)vertices[i][j];
		if( wValues ) cuVertices[i*4+3] = wValues[i];
		else          cuVertices[i*4+3] = 1.0;
	}

	glBindBuffer( GL_ARRAY_BUFFER , vbo );
	glBufferData( GL_ARRAY_BUFFER , vSize*sizeof(GLfloat) , cuVertices, GL_DYNAMIC_DRAW );
	delete[] cuVertices;
}
