/*
Copyright (c) 2011, Michael Kazhdan and Ming Chuang
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of
conditions and the following disclaimer. Redistributions in binary form must reproduce
the above copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the distribution. 

Neither the name of the Johns Hopkins University nor the names of its contributors
may be used to endorse or promote products derived from this software without specific
prior written permission. 

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
TO, PROCUREMENT OF SUBSTITUTE  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
DAMAGE.
*/
#include "Geometry.h"

class Matrix4D
{
public:
	float m[4][4];
	Point3D<float> multiplication( const Point3D<float>& point, bool LinearOnly );
	Matrix4D();
};

class glCamera 
{
public:
	double heightAngle;
	double aspectRatio;
	Point3D< float > position , ref , up , right;

	void rotateUp( Point3D< float > center , float angle );
	void rotateRight( Point3D< float > center , float angle );
	void rotateDir( Point3D< float > center , float angle );
	void moveForward( float dist );
	void moveRight( float dist );
	void moveUp( float dist );
	Matrix4D get( void );
	void set( Matrix4D m );

	void drawOpenGL( void );
	glCamera::~glCamera( void );
	glCamera::glCamera( void );
};
//=============================================== End of Header ===================================================
Matrix4D::Matrix4D(void)
{
	m[0][0]=m[1][1]=m[2][2]=m[3][3]=1;
	m[0][1]=m[1][0]=m[2][0]=m[0][2]=m[3][0]=m[0][3]=0;
	m[1][2]=m[2][1]=m[1][3]=m[3][1]=m[2][3]=m[3][2]=0;
}
Point3D<float> Matrix4D::multiplication(const Point3D<float> &point, bool LinearOnly)
{
	Point3D<float> pt;
	for(int i=0;i<3;i++)
	{
		if(LinearOnly)  pt[i]=0.0;
		else			pt[i]=m[3][i];
		for(int j=0;j<3;j++) pt[i]+=m[j][i]*point[j];
	}
	if(LinearOnly) return pt;
	else		   return pt/m[3][3];
}

glCamera::~glCamera(void){;};
glCamera::glCamera(void)
{	
    position[0] = 0.5;
	position[1] = 0.5;
    position[2] = 2.0;
	
	ref[0] = 0.5;
	ref[1] = 0.5;
	ref[2] = 0.5;
	
	up[0] = 0.0;
	up[1] = 1.0;
	up[2] = 0.0;

	CrossProduct<float>( ref-position, up, right );
	right = right / sqrt( right.InnerProduct(right) );

	heightAngle = 45.0;
}

void glCamera::drawOpenGL(void)
{
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	gluLookAt(	position[0], position[1], position[2], 
				ref[0],      ref[1],      ref[2],
				up[0],       up[1],       up[2]       );  
	
}

void glCamera::rotateUp(Point3D<float> center,float angle)
{
    float x=up[0], y=up[1], z=up[2];
    static Matrix4D matrix;
    matrix.m[0][0] = 1 + (1-cos(angle))*(x*x-1);
    matrix.m[1][0] = -z*sin(angle)+(1-cos(angle))*x*y;
    matrix.m[2][0] = y*sin(angle)+(1-cos(angle))*x*z;
    matrix.m[0][1] = z*sin(angle)+(1-cos(angle))*x*y;
    matrix.m[1][1] = 1 + (1-cos(angle))*(y*y-1);
    matrix.m[2][1] = -x*sin(angle)+(1-cos(angle))*y*z;
    matrix.m[0][2] = -y*sin(angle)+(1-cos(angle))*x*z;
    matrix.m[1][2] = x*sin(angle)+(1-cos(angle))*y*z;
    matrix.m[2][2] = 1 + (1-cos(angle))*(z*z-1); 
    position -= center;
    position = matrix.multiplication( position, false );
    position += center;
    ref -= center;
    ref = matrix.multiplication( ref, false );
    ref += center;
	CrossProduct<float>(ref-position, up, right);
    right = right /  sqrt(right.InnerProduct(right));
}

void glCamera::rotateRight(Point3D<float> center,float angle)
{
    float x=right[0], y=right[1], z=right[2];
    static Matrix4D matrix;
    matrix.m[0][0] = 1 + (1-cos(angle))*(x*x-1);
    matrix.m[1][0] = -z*sin(angle)+(1-cos(angle))*x*y;
    matrix.m[2][0] = y*sin(angle)+(1-cos(angle))*x*z;
    matrix.m[0][1] = z*sin(angle)+(1-cos(angle))*x*y;
    matrix.m[1][1] = 1 + (1-cos(angle))*(y*y-1);
    matrix.m[2][1] = -x*sin(angle)+(1-cos(angle))*y*z;
    matrix.m[0][2] = -y*sin(angle)+(1-cos(angle))*x*z;
    matrix.m[1][2] = x*sin(angle)+(1-cos(angle))*y*z;
    matrix.m[2][2] = 1 + (1-cos(angle))*(z*z-1); 
    position -= center;
    position = matrix.multiplication( position, false );
    position += center;
    ref -= center;
    ref = matrix.multiplication( ref, false );
    ref += center;
	CrossProduct<float>( right, ref-position, up );
    up = up /  sqrt(up.InnerProduct(up));
}

void glCamera::rotateDir(Point3D<float> center,float angle)
{
	Point3D<float> dir = ref-position;
    float x=dir[0], y=dir[1], z=dir[2];
    static Matrix4D matrix;
    matrix.m[0][0] = 1 + (1-cos(angle))*(x*x-1);
    matrix.m[1][0] = -z*sin(angle)+(1-cos(angle))*x*y;
    matrix.m[2][0] = y*sin(angle)+(1-cos(angle))*x*z;
    matrix.m[0][1] = z*sin(angle)+(1-cos(angle))*x*y;
    matrix.m[1][1] = 1 + (1-cos(angle))*(y*y-1);
    matrix.m[2][1] = -x*sin(angle)+(1-cos(angle))*y*z;
    matrix.m[0][2] = -y*sin(angle)+(1-cos(angle))*x*z;
    matrix.m[1][2] = x*sin(angle)+(1-cos(angle))*y*z;
    matrix.m[2][2] = 1 + (1-cos(angle))*(z*z-1); 
    
	up = matrix.multiplication( up, true );
	CrossProduct<float>( dir, up, right);
    right = right /  sqrt(right.InnerProduct(right));
	up = up /  sqrt(up.InnerProduct(up)); 
}

void glCamera::moveForward(float dist)
{
	Point3D<float> dir = ref - position;
	position += dir/sqrt(dir.InnerProduct(dir)) * dist;
};
void glCamera::moveRight(float dist)
{
	Point3D<float> shift = right/sqrt(right.InnerProduct(right)) * dist;
	position += shift;
	ref += shift;
};
void glCamera::moveUp(float dist)
{

	Point3D<float> shift = up/sqrt(up.InnerProduct(up)) * dist;
	position += shift;
	ref += shift;
};
Matrix4D glCamera::get( void )
{
	Point3D< float > forward = ref-position;
	Matrix4D m;
	m.m[3][3] = 1;
	for( int i=0 ; i<3 ; i++ )
	{
		m.m[0][i] = right[i];
		m.m[1][i] = up[i];
		m.m[2][i] = -forward[i];
		m.m[3][i] = position[i];
	}
	return m;

}
void glCamera::set( Matrix4D m )
{
	Point3D< float > forward;
	for( int i=0 ; i<3 ; i++ )
	{
		up      [i] =  m.m[1][i];
		forward [i] = -m.m[2][i];
		position[i] =  m.m[3][i];
	}
	ref = forward + position;
	CrossProduct<float>( ref-position, up, right );
	right = right / sqrt( right.InnerProduct(right) );
}